


/**
 ******************************************************************************
 *
 * @file        BSP_2ColorDotMatrixLED.c
 * @brief       This is two color dot matrix LED C file.
                (Use 74HC595 to control).
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2022/07/01
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
/*==============================================================================
                                 User Notes
How To use this function:
-----------------------
   1. The code use 74HC595 (Serial to parameter shift register) to control 
      2 color dot matrix LED.
   2. Use BSP_2ColorDotMatrixLED_Init() to inital.
      (1) Default is OFF.
   3. Periodic call BSP_2ColorDotMatrixLED_main() function.
      (1) The freq * 8 is 2 color dot matrix LED(8x8) refresh rate.  
   4. 2 color dot matrix display different picture by calling BSP_2ColorDotMatrixLED_RefreshFrame() function. 
      (1) Input parameter is new display picture data pointer.
      (2)   COL 1      2      3      4      5      6      7      8
          ROW 1 Byte0  Byte1  Byte2  Byte3  Byte4  Byte5  Byte6  Byte7
              2 Byte8  Byte9  Byte10 Byte11 Byte12 Byte13 Byte14 Byte15        
              3 Byte16 Byte17 Byte18 Byte19 Byte20 Byte21 Byte22 Byte23
              4 Byte24 Byte25 Byte26 Byte27 Byte28 Byte29 Byte30 Byte31
              5 Byte32 Byte33 Byte34 Byte35 Byte36 Byte37 Byte38 Byte39
              6 Byte40 Byte41 Byte42 Byte43 Byte44 Byte45 Byte46 Byte47
              7 Byte48 Byte49 Byte50 Byte51 Byte52 Byte53 Byte54 Byte55
              8 Byte56 Byte57 Byte58 Byte59 Byte60 Byte61 Byte62 Byte63
              
Driver architecture:
--------------------
   + MG32_GPIO_DRV
   
Known Limitations:
------------------

Require parameter
------------------
    Require module : CSC / GPIO
    
    GPIO pin configuration : 
        Pin / IO mode / AFS
        ---  --------  -----
        PB12/ PPO     / GPIO
        PB13/ PPO     / GPIO
        PB14/ PPO     / GPIO
        PB15/ PPO     / GPIO
    
Example codes:
------------------

==============================================================================*/ 
 
/* Includes ------------------------------------------------------------------*/ 
#include "BSP_17_2ColorDotMatrixLED.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO
#define S2P595_SDI_PIN          PX_Pin_12
#define S2P595_LCH_PIN          PX_Pin_13
#define S2P595_SCLK_PIN         PX_Pin_14
#define S2P595_OEn_PIN          PX_Pin_15
#define S2P595_PIN_Mode         PINX_Mode_PushPull_O
#define S2P595_PIN_AFS          0
#define S2P595_IOM              IOMB
#define S2P595_PORT             GPIOB

//Signal control
#define S2P595_SDI_HIGH()       PB12 = 1
#define S2P595_SDI_LOW()        PB12 = 0

#define S2P595_LCH_HIGH()       PB13 = 1
#define S2P595_LCH_LOW()        PB13 = 0

#define S2P595_SCLK_HIGH()      PB14 = 1
#define S2P595_SCLK_LOW()       PB14 = 0

#define S2P595_nOE_HIGH()       PB15 = 1
#define S2P595_nOE_LOW()        PB15 = 0

#define S2P595_nOE_ENABLE()     PB15 = 0
#define S2P595_nOE_DISABLE()    PB15 = 1

#define DOTLED_RED_DATA         0
#define DOTLED_GREEN_DATA       1

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/ 
static uint8_t DotLED_DramaAct;
static uint8_t DotLED_RefreshFrameFlag;

static uint8_t DotLED_COMTable[8] = {0xFE,0xFD,0xFB,0xF7,0xEF,0xDF,0xBF,0x7F};

static uint8_t DotLED_ShowData[16];
static uint8_t DotLED_ShowDataTmp[16];

/* Private function prototypes -----------------------------------------------*/
static void BSP_2ColorDotMatrixLED_Series2Parallel(uint8_t R_LED, uint8_t G_LED, uint8_t LED_Com);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief	  2 color dot matrix LED inital
 * @details    
 * @return     
 * @exception No 
 * @note      No
 *******************************************************************************
 */
void BSP_2ColorDotMatrixLED_Init(void) 
{ 
    PIN_InitTypeDef  DotLED_Pin;
    uint8_t          DotLED_Tmp;

    /*GPIO inital*/
    GPIO_SetPortBit(S2P595_PORT, (uint16_t)((S2P595_SDI_PIN | S2P595_LCH_PIN | S2P595_SCLK_PIN)));
    
    
    DotLED_Pin.PINX_Pin                = (S2P595_SDI_PIN | S2P595_LCH_PIN | S2P595_SCLK_PIN | S2P595_OEn_PIN);
    DotLED_Pin.PINX_Mode               = S2P595_PIN_Mode;
    DotLED_Pin.PINX_PUResistant        = PINX_PUResistant_Disable;
    DotLED_Pin.PINX_Speed              = PINX_Speed_Low;
    DotLED_Pin.PINX_Inverse            = PINX_Inverse_Disable;
    DotLED_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    DotLED_Pin.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    DotLED_Pin.PINX_Alternate_Function = S2P595_PIN_AFS;                          
    GPIO_PortMode_Config( S2P595_IOM ,&DotLED_Pin);
    
    /*Parameter inital*/
    DotLED_DramaAct             = 0;
    DotLED_RefreshFrameFlag     = 0;


    for( DotLED_Tmp = 0; DotLED_Tmp < 16; DotLED_Tmp++)
    {
        DotLED_ShowData[ DotLED_Tmp]    = 0xFF;
        DotLED_ShowDataTmp[ DotLED_Tmp] = 0xFF;
    }
}
/**
 *******************************************************************************
 * @brief	  Update 2 color dot matrix LED show.
 * @details   
 * @return     
 * @exception No 
 * @note      1. Update ShowBuffer of DotMatrixLED before call the function.
              2. ShowBuffer[8][8] mapping real dot LED location.
              3. ShowBuffer data = 
                 (0).DOTLED_DISABLE: disable in the LED dot.
                 (1).DOTLED_RED    : show Red in the LED dot.
                 (2).DOTLED_GREEN  : show Green in the LED dot.
                 (3).DOTLED_ORANGE : show ORANGE ( Red + Green) in the dot.
 *******************************************************************************
 */
void BSP_2ColorDotMatrixLED_RefreshFrame(uint8_t *pDotLED_Buf)
{
    uint8_t DotLED_RefreshTmp;
    uint8_t DotLED_RefreshRed;
    uint8_t DotLED_RefreshGreen;
    uint8_t DotLED_RefreshEN;
    uint8_t DotLED_RefreshRow;
    
    DotLED_RefreshFrameFlag = 1;
    DotLED_RefreshRow       = 0;
    DotLED_RefreshRed       = 0xFF;
    DotLED_RefreshGreen     = 0xFF;
    DotLED_RefreshEN        = 0x80;
    
    for( DotLED_RefreshTmp = 0; DotLED_RefreshTmp < (8*8); DotLED_RefreshTmp++)
    {
        if( pDotLED_Buf[DotLED_RefreshTmp] & DOTLED_RED)
        {
            DotLED_RefreshRed &= (~DotLED_RefreshEN);
        }
        if( pDotLED_Buf[DotLED_RefreshTmp] & DOTLED_GREEN)
        {
            DotLED_RefreshGreen &= (~DotLED_RefreshEN);
        }
        
        /*Judge to whether change next row or not.*/
        if( DotLED_RefreshEN == 0x01)
        {
            DotLED_RefreshEN = 0x80;
            
            DotLED_ShowDataTmp[ (DotLED_RefreshRow) * 2 + DOTLED_RED_DATA ]   = DotLED_RefreshRed;
            DotLED_ShowDataTmp[ (DotLED_RefreshRow) * 2 + DOTLED_GREEN_DATA ] = DotLED_RefreshGreen;
            
            DotLED_RefreshRow   = DotLED_RefreshRow + 1;
            DotLED_RefreshRed   = 0xFF;
            DotLED_RefreshGreen = 0xFF;
            DotLED_RefreshEN    = 0x80;
        }
        else
        {
            DotLED_RefreshEN = DotLED_RefreshEN >> 1;
        }
    }
    DotLED_RefreshFrameFlag = 0;
}
/**
 *******************************************************************************
 * @brief	  2 color dot matrix main flow. 
 * @details   According visiting function frequency and RefreshFlag parameter 
              to control 2 color dot matrix refresh frequency.
 * @return     
 * @exception No 
 * @note      No
 *******************************************************************************
 */
void BSP_2ColorDotMatrixLED_main(void)
{
    uint8_t BSP_2ColorDotMatrixLED_mainTmp;
    
    
    if( DotLED_DramaAct == 0 && DotLED_RefreshFrameFlag == 0)
    {
        /*Update new picture.*/
        for( BSP_2ColorDotMatrixLED_mainTmp = 0; BSP_2ColorDotMatrixLED_mainTmp < 16; BSP_2ColorDotMatrixLED_mainTmp++)
        {
            DotLED_ShowData[BSP_2ColorDotMatrixLED_mainTmp] = DotLED_ShowDataTmp[BSP_2ColorDotMatrixLED_mainTmp];
        }
    }
    
    BSP_2ColorDotMatrixLED_Series2Parallel( DotLED_ShowData[(DotLED_DramaAct * 2) + DOTLED_RED_DATA],
                                            DotLED_ShowData[(DotLED_DramaAct * 2) + DOTLED_GREEN_DATA], 
                                            DotLED_COMTable[DotLED_DramaAct]);
    
    /*Control display row*/
    DotLED_DramaAct = DotLED_DramaAct + 1;
    
    if( DotLED_DramaAct == 8)
    {
        DotLED_DramaAct = 0;
    }
}
/**
 *******************************************************************************
 * @brief	  Update 2 color dot matrix LED status.
 * @details   Use serial to parallel shift register(74HC595) to control
              2 color dot matrix LED.
 * @param[in] R_LED  : Control Red LED in N column. 
 * @param[in] G_LED  : Control Green LED in N column.
 * @param[in] LED_Com: Control LED display column.
 * @return     
 * @exception No 
 * @note      No
 *******************************************************************************
 */
static void BSP_2ColorDotMatrixLED_Series2Parallel(uint8_t R_LED, uint8_t G_LED, uint8_t LED_Com)
{
    uint8_t R_LED_Temp = R_LED;
    uint8_t G_LED_Temp = G_LED;
    uint8_t LED_COM_Temp = LED_Com;
    uint8_t lCount8;

    S2P595_nOE_LOW(); // nOE Enable
    //S2P595_LCH_HIGH();
    
    //==============================================
    //Control LED column.
    lCount8 = 8;
    do{
        if(LED_COM_Temp & 0x01)
            S2P595_SDI_HIGH();
        else
            S2P595_SDI_LOW();
        __ISB();

        S2P595_SCLK_HIGH();
        //S2P595_LCH_HIGH();
        __ISB();
        S2P595_SCLK_LOW();
        //S2P595_LCH_LOW();
        LED_COM_Temp = LED_COM_Temp >> 1;
    }while(-- lCount8 != 0);
    
    //==============================================
    //Control Green LED in n column.
    lCount8 = 8;
    do{
        if(G_LED_Temp & 0x01)
            S2P595_SDI_HIGH();
        else
            S2P595_SDI_LOW();
        __ISB();

        S2P595_SCLK_HIGH();
        //S2P595_LCH_HIGH();
        __ISB();
        S2P595_SCLK_LOW();
        //S2P595_LCH_LOW();
        G_LED_Temp = G_LED_Temp >> 1;
    }while(-- lCount8 != 0);
    
    //==============================================
    //Control Red LED in n column.
    lCount8 = 8;
    do{
        if(R_LED_Temp & 0x01)
            S2P595_SDI_HIGH();
        else
            S2P595_SDI_LOW();
        __ISB();

        S2P595_SCLK_HIGH();
        //S2P595_LCH_HIGH();
        __ISB();
        S2P595_SCLK_LOW();
        //S2P595_LCH_LOW();
        R_LED_Temp = R_LED_Temp >> 1;
    }while(-- lCount8 != 0);

    S2P595_LCH_LOW();
    __ISB();
    S2P595_LCH_HIGH();
    __ISB();
}






