/*********************************************************************
*	 项目名称:MG82F6P32-DEMO
*			测试板使用 TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV Board
*			CpuCLK=12MHz, SysCLK=12MHz
*			功能描述：
*			UART1_ARGB_Mode	(TXD1/P1.3)
*			Master
*			注意事项：
*
*	 建立时间: 2024.2.26
*	 修改日志:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*定义系统时钟SysClk (MAX.50MHz)
*可选:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*定义CPU时钟 CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	DUTY_1_3		3		// Duty	1/3
#define	DUTY_1_4		4		// Duty	1/4
#define	ARGB_DUTY			DUTY_1_4

//10 pcs ARGB LED
bit	argb_update_flag;

#define	Breathing_Frame_Time_Default		30 //30*1ms	= 30ms
u8 Breathing_Counter;
u8 Breathing_Frame_Value;
u8 Breathing_Frame_Time_Value;

/***********************************************************************************
*函数名称:	 void INT_T0(void)
*功能描述:T0 中断服务程序
*输入参数:
*返回参数:
*************************************************************************************/
void INT_T0(void) interrupt	INT_VECTOR_T0
{
	  Breathing_Counter	++;
	  TH0=TIMER_1T_1ms_TH;
	  TL0=TIMER_1T_1ms_TL;
}

/***********************************************************************************
*函数名称:	 void INT_UART1(void)
*功能描述:UART1	中断服务程序
*输入参数:
*返回参数:
*************************************************************************************/
void INT_UART1(void) interrupt INT_VECTOR_UART1
{
	_push_(SFRPI);		   //保护SFRPI寄存器值

	SFR_SetPage(1);		   //S0CON仅0页操作
	if(TI1)
	{
	}
	if(RI1)
	{
		IO_LED_R=!IO_LED_R;
		argb_update_flag=1;
		RI1= 0;
	}
	_pop_(SFRPI);		   //恢复SFRPI寄存器值
}

/*************************************************
*函数名称:	   void	DelayXus(u16 xUs)
*功能描述:		延时程序，单位为us
*输入参数:	   u8 Us ->	*1us  (1~255)
*返回参数:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*函数名称:	   void	DelayXms(u16 xMs)
*功能描述:	   延时程序，单位为ms
*输入参数:	   u16 xMs -> *1ms	(1~65535)
*返回参数:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*函数名称:	 void InitPort(void)
*功能描述:	 IO配置
*输入参数:
*返回参数:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// 设置P22,P24,P26为推挽输出(LED)
	PORT_SetP1PushPull(BIT3);						// 设置P13(TXD1)为推挽输出
}

/***********************************************************************************
*函数名称:	 void InitClock(void)
*功能描述:	 时钟初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 输出 MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*函数名称:	 void UART1ARGBTransceiver(u8 S1_DATA)
*功能描述:	 串口ARGBI主机发送函数
*输入参数:	 u8	S1_DATA
*返回参数:
*************************************************************************************/
void UART1ARGBTransceiver(u8 S1_DATA)
{
	SFR_SetPage(1);
	S1BUF=S1_DATA;
	while(TI1==0);
	TI1=0;
	SFR_SetPage(0);
}

/***********************************************************************************
*函数名称:	 void InitTimer0(void)
*功能描述:Timer0初始化设置
*输入参数:
*返回参数:
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// 设置T0模式为16位模式
	TM_SetT0Clock_SYSCLK();					// 设置T0时钟源为	T1OF
	TM_SetT0Gate_Disable();
	TM_SetT0LowByte(TIMER_1T_1ms_TL);		// 设置T0低8位
	TM_SetT0HighByte(TIMER_1T_1ms_TH);		// 设置T0高8位
	TM_EnableT0();							// 使能T0
}

/***********************************************************************************
*函数名称:	 void InitUart1_ARGB(void)
*功能描述:	 Uart1初始化 ARGB
*输入参数:
*返回参数:
*************************************************************************************/
void InitUart1_ARGB(void)
{

	UART1_SetModeARGB();				// 设置串口ARGB主机模式
	UART1_SetRXD1TXD1_P12P13();			// 设置串口TXD1引脚为P13

#if(ARGB_DUTY==DUTY_1_3)
	UART1_SetARGBDuty_1_3();
#else
	UART1_SetARGBDuty_1_4();
#endif
	UART1_SetARGBResetTime_BaudRate_x8x20();

	UART1_SetS1BRGSelSYSCLK();			// 设置S0BRG时钟输入为Sysclk
	UART1_SetS1BRGValue(256-((MCU_SYSCLK/750000)/ARGB_DUTY));
	UART1_EnS1BRG();					// 开启S0BRG时钟
	UART1_SetDataMSB();

	UART1_SetARGBTI1Interrupt_Disable();

	argb_update_flag=0;


}

/***********************************************************************************
*函数名称:	 void InitInterrupt(void)
*功能描述:	 中断初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER0();				// 使能T0中断
	INT_EnUART1();				// 使能UART1中断
}

/***********************************************************************************
*函数名称:	 void InitSystem(void)
*功能描述:	 系统初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();

	InitUart1_ARGB();
	InitTimer0();
	InitInterrupt();
}

//**********************Breathing Lighting Effect=****************************
WORD m_breathing_formula(BYTE Data)
{
	  BYTE opposite	= 0;
	WORD wTmp;

	  if (Data < 94)
	  {
	  }
	  else if (Data	< 188)
	  {
			Data = Data-94;
			opposite = 1;
	  }
	  else
	  {
			Data = 0;
	  }

	  if (Data < 64)
	  {
			wTmp = 256-(4*Data);
	  }
	  else
	  {
			wTmp = 0;
	  }

	  if (opposite)
	  {
			wTmp = 256 - wTmp;
	  }

	  return wTmp;
}

void Breathing_Lighting_Effect(void)
{
	  WORD B_Tmp;
	  char x;

	  if (Breathing_Counter	>= Breathing_Frame_Time_Default) //Breathing
	  {
			Breathing_Counter =	CLR;
			B_Tmp =	m_breathing_formula(Breathing_Frame_Value);//Breathing Frame 0~187,	total 188 frame	= 188*30ms = 5.64sec

			if (Breathing_Frame_Value >= 187)
		{
				  Breathing_Frame_Value	= CLR;
		}
			else
				  Breathing_Frame_Value	++;


		for	(x=0; x<10;	x++)
		{
			UART1ARGBTransceiver(HIBYTE( B_Tmp * 0x0F ));
			UART1ARGBTransceiver(HIBYTE( B_Tmp * 0x1F ));
			UART1ARGBTransceiver( 0	);
		}

	}
}
//****************************************************************************

void main()
{
	u8 x;

	InitSystem();

	INT_EnAll();

	for	(x=0; x<87;	x++)
	{
		UART1ARGBTransceiver(0x00);
	}

	while(1)
	{
		DelayXms(1);
		x++;
		if(x>=100)
		{
			x=0;
			IO_LED_G=!IO_LED_G;
		}
		if (argb_update_flag)
		{
			Breathing_Lighting_Effect();
			argb_update_flag = 0;
		}
	}
}




