/*********************************************************************
*	 项目名称:MG82F6P32-DEMO
*			测试板使用 TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV Board
*			CpuCLK=24MHz, SysCLK=24MHz
*		功能描述：
*			定义T2为Mode6:DutyCapture模式 P11
*			定义T3为Mode6:DutyCapture模式 P32
*
*		注意事项：
*
*	 建立时间: 2024.2.26
*	 修改日志:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*定义系统时钟SysClk (MAX.50MHz)
*可选:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		24000000
/*************************************************/
/*************************************************
*定义CPU时钟 CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_TEST_0		P12
#define	IO_TEST_1		P13

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

bit	bT2Captureing;
bit	bT2CaptureOk;
WordTypeDef	T2CaptureFirst;
WordTypeDef	T2CaptureSecond;

bit	bT3Captureing;
bit	bT3CaptureOk;
WordTypeDef	T3CaptureFirst;
WordTypeDef	T3CaptureSecond;

/***********************************************************************************
*函数名称:	 void INT_T2(void)
*功能描述:T2 中断服务程序
*输入参数:
*返回参数:
*************************************************************************************/
void INT_T2(void) interrupt	INT_VECTOR_T2
{
	_push_(SFRPI);
	SFR_SetPage(0);			// TF2,EXF2仅在0页操作, 所以需设置页面为0, 而且在中断里需保护SFRPI
	if(TF2)
	{
		TF2=0;
		IO_LED_R=!IO_LED_R;

	}
	if(EXF2)
	{
		EXF2=0;
		if(bT2Captureing==TRUE)
		{
				bT2Captureing=FALSE;
				bT2CaptureOk=TRUE;
			T2CaptureFirst.B.BHigh=RCAP2H;
			T2CaptureFirst.B.BLow=RCAP2L;
				T2CaptureSecond.B.BHigh=TH2;
				T2CaptureSecond.B.BLow=TL2;
			}
		IO_TEST_0=!IO_TEST_0;
	}
	_pop_(SFRPI);
}

/***********************************************************************************
*函数名称:	 void INT_T3(void)
*功能描述:T3 中断服务程序
*输入参数:
*返回参数:
*************************************************************************************/
void INT_T3(void) interrupt	INT_VECTOR_T3
{
	_push_(SFRPI);
	SFR_SetPage(1);				// TF3仅在1页操作
	if(TF2)
	{
		TF2=0;
		IO_LED_Y=!IO_LED_Y;
	}
	SFR_SetPage(1);				// EXF3仅在1页操作
	if(EXF3)
	{
		EXF3=0;
		if(bT3Captureing==TRUE)
		{
				bT3Captureing=FALSE;
				bT3CaptureOk=TRUE;
			T3CaptureFirst.B.BHigh=RCAP3H;
			T3CaptureFirst.B.BLow=RCAP3L;
				T3CaptureSecond.B.BHigh=TH3;
				T3CaptureSecond.B.BLow=TL3;
			}
		IO_TEST_1=!IO_TEST_1;
	}
	_pop_(SFRPI);
}

/***********************************************************************************

/*************************************************
*函数名称: char	putchar	(char c)
*功能描述: 用于printf	的串口输出程序
*输入参数:	   char	c
*返回参数:
*************************************************/
char putchar (char c)
{
	bit	bES;
	bES=ES0;
	ES0=0;
	S0BUF =	c;
	while(TI0==0);
	TI0=0;
	ES0=bES;
	return 0;
}

/***********************************************************************************
*函数名称:void DelayXus(u8 xUs)
*功能描述:延时程序，单位为us
*输入参数:u8 Us	-> *1us	 (1~255)
*返回参数:
*************************************************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/***********************************************************************************
*函数名称:void DelayXms(u16	xMs)
*功能描述:延时程序，单位为ms
*输入参数:u16 xMs -> *1ms  (1~65535)
*返回参数:
*************************************************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*函数名称:	 void InitTimer2(void)
*功能描述:Timer2初始化设置
*输入参数:
*返回参数:
*************************************************************************************/
void InitTimer2(void)
{
	TM_SetT2Mode_DutyCapture();						// 设置T2模式为DutyCapture模式
	TM_SetT2Clock_SYSCLKDiv12();					// 设置T2时钟源为	SYSCLK/12

	TM_SetT2LowByte(0);								// 设置TL2
	TM_SetT2RLLowByte(0);							// 设置RCAP2L

	TM_SetT2HighByte(0);							// 设置TH2
	TM_SetT2RLHighByte(0);							// 设置RCAP2H

	TM_SetT2Capture_T2EXPin();						// 使能T2外部中断为T2EX引脚
	TM_SetT2CKOT2EX_P10P11();						// 使能T2EX引脚为P11
	TM_DisableT2EX_DetectFalling();					// 禁止T2EX引脚检测下降沿
	TM_EnableT2EX_DetectRising();					// 使能T2EX引脚检测上升沿

}

/***********************************************************************************
*函数名称:	 void InitTimer3(void)
*功能描述:Timer3初始化设置
*输入参数:
*返回参数:
*************************************************************************************/
void InitTimer3(void)
{
	TM_SetT3Mode_DutyCapture();						// 设置T3模式为DutyCapture模式
	TM_SetT3Clock_SYSCLKDiv12();					// 设置T3时钟源为	SYSCLK/12

	TM_SetT3LowByte(0);								// 设置TL3
	TM_SetT3RLLowByte(0);							// 设置RCAP3L

	TM_SetT3HighByte(0);							// 设置TH3
	TM_SetT3RLHighByte(0);							// 设置RCAP3H

	TM_SetT3Capture_T3EXPin();						// 使能T3外部中断为T3EX引脚
	TM_SetT3CKOT3EX_P33P32();						// 使能T3EX引脚为P32
	TM_DisableT3EX_DetectFalling();					// 禁止T3EX引脚检测下降沿
	TM_EnableT3EX_DetectRising();					// 使能T3EX引脚检测上升沿
}

/***********************************************************************************
*函数名称:	 void InitInterrupt(void)
*功能描述:InitInterrupt初始化设置
*输入参数:
*返回参数:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER2();
	INT_EnTIMER3();
}

/***********************************************************************************
*函数名称:	 void InitPort(void)
*功能描述:	 IO配置
*输入参数:
*返回参数:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// 设置P22,P24,P26为推挽输出(LED)
	PORT_SetP1PushPull(BIT2|BIT3);					// 设置P12,P13为推挽输出
	PORT_SetP1OpenDrainPu(BIT1);										// 设置P11为开漏带上拉，用于T2EX引脚
	PORT_SetP3QuasiBi(BIT0|BIT1|BIT2);								// 设置P30,P31为准双向口，用于URT0引脚,P32为准双向口，用于T3EX引脚
}

/***********************************************************************************
*函数名称:	 void InitUart0_S0BRG(void)
*功能描述:	 Uart0初始化 使用S0BRG作为波特率来源
*输入参数:
*返回参数:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// 8位 可变波特率
	UART0_EnReception();				// 使能接收
	UART0_SetBRGFromS0BRG();			// 设置波特率来源为	S0BRG
	UART0_SetS0BRGBaudRateX2();			// 设置2x波特率
	UART0_SetS0BRGSelSYSCLK();			// 设置S0BRG时钟输入为Sysclk
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_24000000_1T);

	UART0_EnS0BRG();					// 开启S0BRG时钟
}

/***********************************************************************************
*函数名称:	 void InitClock(void)
*功能描述:	 时钟初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 输出 MCK/4
	//CLK_P60OC_MCKDiv4();
}


/***********************************************************************************
*函数名称:	 void InitSystem(void)
*功能描述:	 系统初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitSystem(void)
{

	InitPort();
	InitClock();
	InitUart0_S0BRG();

	InitTimer2();
	InitTimer3();

	InitInterrupt();

	INT_EnAll();			// 使能全局中断


}

void StartT2CaptureDuty(void)
{
	bT2CaptureOk=FALSE;
	bT2Captureing=TRUE;
	TM_SetT2HighByte(0);
	TM_SetT2LowByte(0);
	TM_EnableT2();
}

void StartT3CaptureDuty(void)
{
	bT3CaptureOk=FALSE;
	bT3Captureing=TRUE;
	TM_SetT3HighByte(0);
	TM_SetT3LowByte(0);
	TM_EnableT3();
}

void main()
{
	float duty;
	float freq;

	InitSystem();

	printf("\nHello!!");

	IO_LED_G=0;IO_LED_R=0;IO_LED_Y=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_R=1;IO_LED_Y=1;

	StartT2CaptureDuty();
	StartT3CaptureDuty();

	while(1)
	{
		IO_LED_G=0;
		DelayXms(200);
		IO_LED_G=1;
		DelayXms(100);

		if(bT2CaptureOk)
		{
			duty=(float)(((double)(T2CaptureFirst.W)*100L)/(float)T2CaptureSecond.W);
			freq=(float)(1000)/((float)(T2CaptureSecond.W)/2);
			printf("\nT2 Caputre: Duty=%f,Freq=%fKHz",duty,freq);
			StartT2CaptureDuty();
		}
		if(bT3CaptureOk)
		{
			duty=(float)(((double)(T3CaptureFirst.W)*100L)/(float)T3CaptureSecond.W);
			freq=(float)(1000)/((float)(T3CaptureSecond.W)/2);
			printf("\nT3 Caputre: Duty=%f,Freq=%fKHz",duty,freq);
			StartT3CaptureDuty();
		}
	}
}


