/*********************************************************************
*	 项目名称:MG82F6P32-DEMO
*			测试板使用 TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV Board
*			CpuCLK=12MHz, SysCLK=12MHz
*	功能描述：
*			内部IVR2.4v作为VREF+,计算VDD和P10的输入电压
*	注意事项：
*
*	 建立时间: 2024.2.26
*	 修改日志:
*
*********************************************************************/

#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*定义系统时钟SysClk (MAX.50MHz)
*可选:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*定义CPU时钟 CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/

#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	VCC_REF					(3300L)

/*************************************************
*函数名称: char	putchar	(char c)
*功能描述: 用于printf	的串口输出程序
*输入参数:	   char	c
*返回参数:
*************************************************/
char putchar (char c)
{
	bit	bES;
	bES=ES0;
	ES0=0;
	S0BUF =	c;
	while(TI0==0);
	TI0=0;
	ES0=bES;
	return 0;
}

/*************************************************
*函数名称:	   void	DelayXus(u16 xUs)
*功能描述:		延时程序，单位为us
*输入参数:	   u8 Us ->	*1us  (1~255)
*返回参数:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*函数名称:	   void	DelayXms(u16 xMs)
*功能描述:	   延时程序，单位为ms
*输入参数:	   u16 xMs -> *1ms	(1~65535)
*返回参数:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*函数名称:	 void InitUart0(void)
*功能描述:	 Uart0初始化 使用S0BRG作为波特率来源
*输入参数:
*返回参数:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// 8位 可变波特率
	UART0_EnReception();				// 使能接收
	UART0_SetBRGFromS0BRG();			// 设置波特率来源为	S0BRG
	UART0_SetS0BRGBaudRateX2();			// 设置2x波特率
	UART0_SetS0BRGSelSYSCLK();			// 设置S0BRG时钟输入为Sysclk
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_12000000_1T);

	UART0_EnS0BRG();					// 开启S0BRG时钟
}

/***********************************************************************************
*函数名称:	 void InitPort(void)
*功能描述:	 IO配置
*输入参数:
*返回参数:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);		// 设置P22,P24,P26为推挽输出(LED)
	PORT_SetP3QuasiBi(BIT0|BIT1);					// 设置P30,P31 作为准双向口
	PORT_SetP1AInputOnly(BIT0);						// 设置P10(AIN0)为仅模拟输入
}

/***********************************************************************************
*函数名称:	 void InitADC(void)
*功能描述:	 ADC初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitADC(void)
{
	ADC_Enable();									// 使能ADC
	ADC_SetADCConversion_30ADCClock();
	ADC_SetClock_SYSCLKDiv8();						// ADC转换时钟为	SYSCLK	   转换率=	12M/8/30=50k
	ADC_SetMode_SetADCS();							// ADC启动模式,	置ADCS
	ADC_SetADCData_12Bit();							// ADC数据位数为12位
	ADC_SetRightJustified();						// ADC转换数据右对齐(数据位8位时,要设置成左对齐, ADCDH保存8位ADC值)
	ADC_SetVREF_IVR24();							// 设置IVR 2.4V为VREF+
	ADC_SetInput_AIN();								// ADC输入为ADC 通道
	IVREF_Enable();									// 使能内部IVR 2.4V
}

/***********************************************************************************
*函数名称:	 u16 GetAdcData()
*功能描述:	 读取d当前通道的ADC值
*输入参数:
*返回参数:	 u16 :ADC值
*************************************************************************************/
u16	GetAdcValue()
{
	WordTypeDef	wAdcValue;
	ADCON0 = ADCON0|ADCS;				// 置位ADCS,启动ADC
	while((ADCON0&ADCI)==0);			// 等待转换完成
	wAdcValue.B.BHigh=ADCDH;
	wAdcValue.B.BLow=ADCDL;
	ADCON0 = ADCON0&(~ADCI);					//清ADCI
	return wAdcValue.W&0x0FFF;
}

/***********************************************************************************
*函数名称:	 u16 GetIVRef()
*功能描述:	 读取内部IVRef的校准值
*输入参数:
*返回参数:	 u16 :IVRef	在3.3V下的ADC值
*************************************************************************************/
u16	GetIVRef()
{
	WordTypeDef	IVR_Code;
	ISPCR =	0x80;
	IFMT = 0x06;
	IFADRH = 0x00;
	IFADRL = 0xC0;

	SCMD = 0x46;
	SCMD = 0xB9;
	IVR_Code.B.BHigh = IFD;

	IFADRL ++;
	SCMD = 0x46;
	SCMD = 0xB9;
	IVR_Code.B.BLow= IFD;

	ISPCR =	0x00;

	return IVR_Code.W;
}

/***********************************************************************************
*函数名称:	 void InitClock(void)
*功能描述:	 时钟初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 输出 MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*函数名称:	 void InitSystem(void)
*功能描述:	 系统初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitUart0_S0BRG();
	InitADC();
}

void main()
{
	u8 i;
	DWordTypeDef dwTemp;
	WordTypeDef	wVCCValue,wVinValue,wIVRefValue;
	WordTypeDef	wIVRAdjAdc,wVDDAdc,wVinAdc;

	InitSystem();

	IO_LED_G=0;IO_LED_R=0,IO_LED_Y=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_R=1,IO_LED_Y=1;

	wIVRAdjAdc.W=GetIVRef();

	// 检测IVRef校准值是否在范围内(2.3~2.5)
	if((wIVRAdjAdc.W <2854)||(wIVRAdjAdc.W>=3103))
	{ // 存储的IVRef值有问题,设为默认值	2.4V@3.3V
		// 串口输出
		printf("\nErr:%d",wIVRAdjAdc.W);
		IO_LED_R=0;
		wIVRAdjAdc.W=2979;
	}
	else
	{
		// 串口输出
		printf("\nIVR ADC@3.3V:%d",wIVRAdjAdc.W);
	}
	// 计算IVR的电压值
	dwTemp.DW=wIVRAdjAdc.W;
	wIVRefValue.W=(dwTemp.DW*VCC_REF)/4096;
	printf("\nIVRef:%d",wIVRefValue.W);


	while(1)
	{
		IO_LED_G=0;
		DelayXms(200);
		IO_LED_G=1;
		DelayXms(200);

		ADC_SetChannel_VDD_Div4();	// 设置ADC通道为VDD/4
		// 读取16次ADC值,计算平均值
		wVDDAdc.W=0x0000;
		for(i=0;i<16;i++)
		{
			wVDDAdc.W=wVDDAdc.W+GetAdcValue();
		}
		wVDDAdc.W=wVDDAdc.W>>4;

		// 计算当前的VCC值
		dwTemp.DW=wIVRefValue.W;
		wVCCValue.W=(dwTemp.DW*wVDDAdc.W)/(4096L/4);

		// 串口输出
		printf("\nVCC:%d",wVCCValue.W);

		ADC_SetChannel_AIN0();		// 设置ADC通道为AIN0/P10
		// 读取16次ADC值,计算平均值
		wVinAdc.W=0x0000;
		for(i=0;i<16;i++)
		{
			wVinAdc.W=wVinAdc.W+GetAdcValue();
		}
		wVinAdc.W=wVinAdc.W>>4;

		// 计算AIN0的输入电压值
		dwTemp.DW=wIVRefValue.W;
		wVinValue.W=dwTemp.DW*wVinAdc.W/(4096L);

		// 串口输出
		printf(" Vin:%d",wVinValue.W);
	}
}


