#include <intrins.h>
#include "REG_MGEQ1C064.h"
#include "TYPEDEF.h"
#include "stdio.h"
#include "MGEQ1C064_COMMON_DRV.h"
#include "MGEQ1C064_GPIO_DRV.h"
#include "MGEQ1C064_Timer_DRV.h"
#include "MGEQ1C064_UART0_DRV.h"
#include "MGEQ1C064_INT_DRV.h"
#include "MGEQ1C064_INT_VECTOR.h"

/**
 ******************************************************************************
 * do-while template
 ******************************************************************************
 */
#define MWT( __stuff__ )  do { __stuff__ } while (0)

/****************************************************************
 *  Function : UART0_9600_Init
 *  Use the Timer1(T1OF) as UART0 clock source.
 ****************************************************************/
void UART0_9600_Init(void)
{
    // UART0 Register initial Tx:P31 Rx:P30
    __DRV_URT0_SetSM10(UART0_MODE1_8BIT);
    __DRV_URT0_SerialReception_Cmd(MW_ENABLE);
    TI0 = 1;

    // Use Timer1 Auto reload mode to generate the baud rate 9600
    __DRV_TIMER1_Mode_Select(TIMER1_MODE2_8BIT_AUTORELOAD);
    __DRV_TIMER1_Clock_Source_Select(TIMER1_CLOCK_SOURCE_SYSCLK);
    __DRV_TIMER1_Set8BitIntervalAutoReload(217);
    __DRV_TIMER1_Run_Cmd(MW_ENABLE);
}

/****************************************************************
Variables
Timer4 overflow Interrupt(TH4) => TF4_Flag
Timer4 external Interrupt      => EXF4_Flag
Timer4 overflow Interrupt(TL4) => TF4L_Flag
****************************************************************/
uint8_t TF4_Flag, EXF4_Flag, TF4L_Flag;

void main()
{
    /****************************************************************
    Variables
    Temp_TH4    => Store the TH4 value
    Temp_TL4    => Store the TL4 value
    Temp_RCAP4H => Store the RCAP4H value
    Temp_RCAP4L => Store the RCAP4L value
    Period      => [TH4:TL4]
    Duty        => [RCAP4H:RCAP4L]
    ****************************************************************/
    uint8_t Temp_RCAP4H = 0, Temp_RCAP4L = 0;
    uint8_t Temp_TH4 = 0, Temp_TL4 = 0;
    uint16_t Period = 0, Duty = 0;

    /****************************************************************
     *  UART0 Initialized
     ****************************************************************/
    UART0_9600_Init();

    /****************************************************************
     *  Interrupt Initialized
     *  1. Enable Timer4 interrupt (ET4) = Enable;
     *  2. Global enables all interrupts (EA) = Enable;
     ****************************************************************/
    __DRV_TIMER4_IT_Cmd(MW_ENABLE);
    __DRV_INT_ITEA_Enable();

    /****************************************************************
     *  GPIO Initialized
     *  1. P55 used for LED_D4
     *  2. P71 used for T4EX Pin
     *  3. P34 used for T0CKO
     ****************************************************************/
    __DRV_GPIO_P5ModeSelect(P55_PushPull);
    __DRV_GPIO_P7ModeSelect(P71_OpenDrainPullUp);
    __DRV_GPIO_P3ModeSelect(P34_QuasiMode);

    /****************************************************************
     *  Timer0 Initialized
     *  PWM Duty high = 90.2% (230/255)
     *  PWM Duty low = 9.8% (25/255)
     ****************************************************************/
    __DRV_TIMER0_Mode_Select(TIMER0_MODE0_8BIT_PWM);
    __DRV_TIMER0_Clock_Source_Select(TIMER0_CLOCK_SOURCE_ILRCO);
    __DRV_TIMER0_SetHighByte(25);
    __DRV_TIMER0_T0CKO_Cmd(MW_ENABLE);
    __DRV_TIMER0_Run_Cmd(MW_ENABLE);

    /****************************************************************
     *  Timer4 Initialized
     *  Interrupt time => (1 / (SYSCLK/12)) * 65536 * 15 = 0.983 S
     ****************************************************************/
    __DRV_TIMER4_Mode_Select(TIMER4_MODE6_DUTY_CAPTURE);
    __DRV_TIMER4_Clock_Source_Select(TIMER4_CLOCK_SOURCE_SYSCLK_DIV_12);
    __DRV_TIMER4_Capture_Source_Dectect(Timer4_RISING_EDGE);
    __DRV_TIMER4_Run_Cmd(MW_ENABLE);

    while (1)
    {
        /****************************************************************
         *  Timer4 Duty Capture
         *  T0CKO PWM Signal(P34) output to LED_D4(P55) and T4EX Pin(P71)
         ****************************************************************/
        __DRV_GPIO_WriteP5(P55, P34);
        __DRV_GPIO_WriteP7(P71, P34);
        if (EXF4_Flag)
        {
            Temp_TH4 = DRV_TIMER4_GetTH4();
            Temp_TL4 = DRV_TIMER4_GetTL4();
            Period = Temp_TH4 << 8 | Temp_TL4;
            Temp_RCAP4H = DRV_TIMER4_GetRCAP4H();
            Temp_RCAP4L = DRV_TIMER4_GetRCAP4L();
            Duty = Temp_RCAP4H << 8 | Temp_RCAP4L;
            printf("Duty Time = %.3f mS, Period Time = %.3fmS\n", Duty * 0.001, Period * 0.001);
            printf("Duty = %.3f%% \n\n", ((float)Duty / (float)Period) * 100);
            EXF4_Flag = 0;
        }
    }
}

/****************************************************************
Timer4 interrupt
The Timer4 has TF4, EXF4, TF4L flags to trigger the Timer4 interrupt.
****************************************************************/
void TIMER4_TF4_EXF4_TF4L_ISR(void) interrupt TIMER4_ISR_VECTOR
{
    if (DRV_TIMER4_GetTF4() == 1)
    {
        TF4_Flag = 1;
        DRV_TIMER4_ClearTF4();
    }

    if (DRV_TIMER4_GetTF4L() == 1)
    {
        TF4L_Flag = 1;
        DRV_TIMER4_ClearTF4L();
    }

    if (DRV_TIMER4_GetEXF4() == 1)
    {
        EXF4_Flag = 1;
        DRV_TIMER4_ClearEXF4();
    }
}
