#include <intrins.h>
#include "REG_MGEQ1C064.h"
#include "TYPEDEF.h"
#include "stdio.h"
#include "MGEQ1C064_COMMON_DRV.h"
#include "MGEQ1C064_GPIO_DRV.h"
#include "MGEQ1C064_Timer_DRV.h"
#include "MGEQ1C064_UART0_DRV.h"
#include "MGEQ1C064_INT_DRV.h"
#include "MGEQ1C064_INT_VECTOR.h"

/**
 ******************************************************************************
 * do-while template
 ******************************************************************************
 */
#define MWT( __stuff__ )  do { __stuff__ } while (0)

/****************************************************************
 *  Function : UART0_9600_Init
 *  Use the Timer1(T1OF) as UART0 clock source.
 ****************************************************************/
void UART0_9600_Init(void)
{
    // UART0 Register initial Tx:P31 Rx:P30
    __DRV_URT0_SetSM10(UART0_MODE1_8BIT);
    __DRV_URT0_SerialReception_Cmd(MW_ENABLE);
    TI0 = 1;

    // Use Timer1 Auto reload mode to generate the baud rate 9600
    __DRV_TIMER1_Mode_Select(TIMER1_MODE2_8BIT_AUTORELOAD);
    __DRV_TIMER1_Clock_Source_Select(TIMER1_CLOCK_SOURCE_SYSCLK);
    __DRV_TIMER1_Set8BitIntervalAutoReload(217);
    __DRV_TIMER1_Run_Cmd(MW_ENABLE);
}

/****************************************************************
Variables
Timer3 overflow Interrupt(TH3) => TF3_Flag
Timer3 external Interrupt      => EXF3_Flag
Timer3 overflow Interrupt(TL3) => TF3L_Flag
****************************************************************/
uint8_t TF3_Flag, EXF3_Flag, TF3L_Flag;

void main()
{
    /****************************************************************
    Variables
    Temp_TH3    => Store the TH3 value
    Temp_TL3    => Store the TL3 value
    Temp_RCAP3H => Store the RCAP3H value
    Temp_RCAP3L => Store the RCAP3L value
    Period      => [TH3:TL3]
    Duty        => [RCAP3H:RCAP3L]
    ****************************************************************/
    uint8_t Temp_RCAP3H = 0, Temp_RCAP3L = 0;
    uint8_t Temp_TH3 = 0, Temp_TL3 = 0;
    uint16_t Period = 0, Duty = 0;

    /****************************************************************
     *  UART0 Initialized
     ****************************************************************/
    UART0_9600_Init();

    /****************************************************************
     *  Interrupt Initialized
     *  1. Enable Timer3 interrupt (ET3) = Enable;
     *  2. Global enables all interrupts (EA) = Enable;
     ****************************************************************/
    __DRV_TIMER3_IT_Cmd(MW_ENABLE);
    __DRV_INT_ITEA_Enable();

    /****************************************************************
     *  GPIO Initialized
     *  1. P55 used for LED_D4
     *  2. P00 used for T3EX Pin
     *  3. P34 used for T0CKO
     ****************************************************************/
    __DRV_GPIO_P5ModeSelect(P55_PushPull);
    __DRV_GPIO_P0ModeSelect(P00_OpenDrainPullUp);
    __DRV_GPIO_P3ModeSelect(P34_QuasiMode);

    /****************************************************************
     *  Timer0 Initialized
     *  PWM Duty high = 90.2% (230/255)
     *  PWM Duty low = 9.8% (25/255)
     ****************************************************************/
    __DRV_TIMER0_Mode_Select(TIMER0_MODE0_8BIT_PWM);
    __DRV_TIMER0_Clock_Source_Select(TIMER0_CLOCK_SOURCE_ILRCO);
    __DRV_TIMER0_SetHighByte(25);
    __DRV_TIMER0_T0CKO_Cmd(MW_ENABLE);
    __DRV_TIMER0_Run_Cmd(MW_ENABLE);

    /****************************************************************
     *  Timer3 Initialized
     *  Interrupt time => (1 / (SYSCLK/12)) * 65536 * 15 = 0.983 S
     ****************************************************************/
    __DRV_TIMER3_Mode_Select(TIMER3_MODE6_DUTY_CAPTURE);
    __DRV_TIMER3_Clock_Source_Select(TIMER3_CLOCK_SOURCE_SYSCLK_DIV_12);
    __DRV_TIMER3_Capture_Source_Dectect(Timer3_RISING_EDGE);
    __DRV_TIMER3_Run_Cmd(MW_ENABLE);

    while (1)
    {
        /****************************************************************
         *  Timer2 Duty Capture
         *  T0CKO PWM Signal(P34) output to LED_D4(P55) and T3EX Pin(P00)
         ****************************************************************/
        __DRV_GPIO_WriteP5(P55, P34);
        __DRV_GPIO_WriteP0(P00, P34);
        if (EXF3_Flag)
        {
            Temp_TH3 = DRV_TIMER3_GetTH3();
            Temp_TL3 = DRV_TIMER3_GetTL3();
            Period = Temp_TH3 << 8 | Temp_TL3;
            Temp_RCAP3H = DRV_TIMER3_GetRCAP3H();
            Temp_RCAP3L = DRV_TIMER3_GetRCAP3L();
            Duty = Temp_RCAP3H << 8 | Temp_RCAP3L;
            printf("Duty Time = %.3f mS, Period Time = %.3fmS\n", Duty * 0.001, Period * 0.001);
            printf("Duty = %.3f%% \n\n", ((float)Duty / (float)Period) * 100);
            EXF3_Flag = 0;
        }
    }
}

/****************************************************************
Timer3 interrupt
The Timer3 has TF3, EXF3, TF3L flags to trigger the Timer3 interrupt.
****************************************************************/
void TIMER3_TF3_EXF3_TF3L_ISR(void) interrupt TIMER3_ISR_VECTOR
{
    if (DRV_TIMER3_GetTF3() == 1)
    {
        TF3_Flag = 1;
        DRV_TIMER3_ClearTF3();
    }

    if (DRV_TIMER3_GetTF3L() == 1)
    {
        TF3L_Flag = 1;
        DRV_TIMER3_ClearTF3L();
    }

    if (DRV_TIMER3_GetEXF3() == 1)
    {
        EXF3_Flag = 1;
        DRV_TIMER3_ClearEXF3();
    }
}
