/**
 ******************************************************************************
 *
 * @file        MG32_TM_MID.H
 *
 * @brief       This file provides firmware functions to manage the following 
 *              functionalities of the TM peripheral:
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2025/04/17
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2016 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par 		Disclaimer 
 *		The Demo software is provided "AS IS"  without any warranty, either 
 *		expressed or implied, including, but not limited to, the implied warranties 
 *		of merchantability and fitness for a particular purpose.  The author will 
 *		not be liable for any special, incidental, consequential or indirect 
 *		damages due to loss of data or any other reason. 
 *		These statements agree with the world wide and local dictated laws about 
 *		authorship and violence against these laws. 
 ******************************************************************************
 ******************************************************************************
 */ 




#ifndef _MG32_TM_MID_H

/*!< _MG32_TM_DRV_H */ 
#define _MG32_TM_MID_H

#include "MG32.H"
#include "MG32_Common_MID.H"
#include "MG32__Common_DRV.H"
#include "MG32_TM_DRV.H"



/** 
 * @enum		MID_TM_ChannelSelectDef
 * @brief		declare Timer channel selection 
 */        
typedef enum 
{ 
    MID_TM_Channel0,    /*!< Specifies the TM Channel0 */
    MID_TM_Channel1,    /*!< Specifies the TM Channel1 */
    MID_TM_Channel2,    /*!< Specifies the TM Channel2 */
    MID_TM_Channel3,    /*!< Specifies the TM Channel3 */
    MID_TM_ChannelAll,  /*!< Specifies the all TM Channel */      
} MID_TM_ChannelSelectDef;

/** 
 * @enum		MID_TM_StateTypeDef
 * @brief		MID State structures definition 
 */        
typedef enum 
{ 
    MID_TM_STATE_BUSY,      /*!< An internal process is running */ 
    MID_TM_STATE_READY,     /*!< Peripheral Initialized and ready for use */
    MID_TM_STATE_RESET,
    
} MID_TM_StateTypeDef;


/*! @struct TM_HandleTypeDef
    @brief  TM's timebase/procedure control
*/ 
typedef struct 
{ 
    TM_Struct               *Instance;          /*!< Register base address. e.g TM00, TM01 ... TM36 */
    
    TM_TimeBaseInitTypeDef  TimeBaseInit;       /*!< TM Time Base required parameters */
    
    MID_TM_ChannelSelectDef TM_ActiveChnnel;    /*!< Active channel  */ 
    
    MID_LockTypeDef         Lock;               /*!< Locking object */
    
    MID_TM_StateTypeDef     State;              /*!< TM operation state  */
    
} TM_HandleTypeDef;

/*! @struct TM_OC_InitTypeDef
    @brief  TM Output Compare Configuration Structure definition
*/ 
typedef struct
{
    CHFMDef         OCMode;       /*!< Specifies the channel mode.
                                      This parameter can be a value of @ref CHFMDef */

    uint16_t        ReloadValue;  /*!< Specifies the reload value into the Capture/Compare Register.
                                       This parameter can be a number between Min_Data = 0x0000 and Max_Data = 0xFFFF.
                                       (8bitx2) Low byte specifies for compare-L. High byte specifies for compare-H */

    FunctionalState OCxEnable;    /*!< Specifies the TM Output Compare OCx pin.
                                       This parameter can be: ENABLE or Disable. */

    FunctionalState OCxInverse;   /*!< Specifies Timer OCx output inverse function.
                                       This parameter can be: ENABLE or Disable. */

    BitAction       OCxIdleState; /*!< Specifies the TM OCx Output state during Idle state.
                                       This parameter can be: CLR or SET. */

    FunctionalState OCNEnable;    /*!< Specifies the TM Output complement pin .
                                       This parameter can be: ENABLE or Disable. */

    FunctionalState OCNInverse;   /*!< Specifies Timer channel complement output inverse function.
                                       This parameter can be: ENABLE or Disable. */

    BitAction       OCNIdleState; /*!< Specifies the TM OCxN Output state during Idle state.
                                       This parameter can be: CLR or SET. */

    FunctionalState OCHInverse;   /*!< Specifies Timer channel compare-H output inverse function.
                                       This parameter can be: ENABLE or Disable. */

    BitAction       OCHIdleState; /*!< Specifies the TM OCxH Output state during Idle state.
                                       This parameter can be: CLR or SET. */

                                     
} TM_OC_InitTypeDef;






/**
 * @name    Function announce
 * @brief   Time base initial/Deinitial	
 */ 
///@{
MID_StatusTypeDef MID_TM_PWM_Init(TM_HandleTypeDef *htim);
MID_StatusTypeDef MID_TM_PWM_DeInit(TM_HandleTypeDef *htim);
///@}

/**
 * @name    Function announce
 * @brief   PWM channel/Duty/Start/Stop config	
 */ 
///@{
MID_StatusTypeDef MID_TM_PWM_ConfigChannel(TM_HandleTypeDef *htim, 
                            TM_OC_InitTypeDef* sConfig, MID_TM_ChannelSelectDef Channel);
MID_StatusTypeDef MID_TM_PWM_Start(TM_HandleTypeDef *htim);
MID_StatusTypeDef MID_TM_PWM_Stop(TM_HandleTypeDef *htim, FunctionalState NewState);
///@}

/**
 * @name    Function announce
 * @brief   Timer PWM interrupt config	
 */ 
///@{
MID_StatusTypeDef MID_TM_PWM_Start_IT(TM_HandleTypeDef *htim, MID_TM_ChannelSelectDef Channel);
MID_StatusTypeDef MID_TM_PWM_Stop_IT(TM_HandleTypeDef *htim, MID_TM_ChannelSelectDef Channel);
///@}

/**
 * @name    Function announce
 * @brief   Accept/Reject reload function	
 */ 
///@{
MID_StatusTypeDef MID_TM_AcceptUpdateOC(TM_HandleTypeDef *htim);
MID_StatusTypeDef MID_TM_RejectUpdateOC(TM_HandleTypeDef *htim);
///@}

/**
 * @name    Function announce
 * @brief   Reload 16bit/8 bit, 8bit for Channel0L or Channel0H
 */ 
///@{
void MID_TM_UpdateChannel0(TM_HandleTypeDef *htim, uint16_t ReloadV);
void MID_TM_UpdateChannel0H(TM_HandleTypeDef *htim, uint8_t ReloadV);
void MID_TM_UpdateChannel0L(TM_HandleTypeDef *htim, uint8_t ReloadV);
///@}

/**
 * @name    Function announce
 * @brief   Reload 16bit/8 bit, 8bit for Channel1L or Channel1H
 */ 
///@{
void MID_TM_UpdateChannel1(TM_HandleTypeDef *htim, uint16_t ReloadV);
void MID_TM_UpdateChannel1H(TM_HandleTypeDef *htim, uint8_t ReloadV);
void MID_TM_UpdateChannel1L(TM_HandleTypeDef *htim, uint8_t ReloadV);
///@}

/**
 * @name    Function announce
 * @brief   Reload 16bit/8 bit, 8bit for Channel2L or Channel2H
 */ 
///@{
void MID_TM_UpdateChannel2(TM_HandleTypeDef *htim, uint16_t ReloadV);
void MID_TM_UpdateChannel2H(TM_HandleTypeDef *htim, uint8_t ReloadV);
void MID_TM_UpdateChannel2L(TM_HandleTypeDef *htim, uint8_t ReloadV);
///@}

/**
 * @name    Function announce
 * @brief   Reload 16bit/8 bit, 8bit for Channel3L or Channel3H
 */ 
///@{
void MID_TM_UpdateChannel3(TM_HandleTypeDef *htim, uint16_t ReloadV);
void MID_TM_UpdateChannel3H(TM_HandleTypeDef *htim, uint8_t ReloadV);
void MID_TM_UpdateChannel3L(TM_HandleTypeDef *htim, uint8_t ReloadV);
///@}


#endif
