
/**
 ******************************************************************************
 *
 * @file        Sample_URT0_SPIFlash.c
 * @brief       Use URT of MG32 to control MX25R512F (SPI Flash) standard read / write. 
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2022/08/18
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 ******************************************************************************
 */



/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_URT_DRV.h"
#include "MG32_GPIO_DRV.h"
#include "Sample_URT0_SPIFlash.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
FlashCTR_TypeDef Flash0;

/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
uint8_t Sample_URT_ProgramFlashBuffer[10] = 
{
    0x05, 0x30, 0x06, 0x15, 0x04, 0x03, 0x11, 0x20, 0x09 , 0x12
};
uint8_t Sample_URT_ReadFlashBuffer[10];

/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	   SPI write data. 
 * @details     
 * @param[in]  DataLen: Data size
 * @param[in]  FlashData: Send data
 * @return                        
 *******************************************************************************
 */
void Sample_URT_SetData(uint8_t DataLen , uint32_t FlashData)
{
    URT_SetTXData(URTX,DataLen , FlashData);    
    while((URT_GetITAllFlagStatus(URTX) & URT_IT_TC) !=URT_IT_TC);
    URT_ClearITFlag(URTX, (URT_IT_TC | URT_IT_TX | URT_IT_RX));
}

/**
 *******************************************************************************
 * @brief	   SPI read data 
 * @details     
 * @return     SPI get data.                  
 *******************************************************************************
 */
uint8_t Sample_URT_ReadData(void)
{
    URT_SetTXData(URTX,1,Flash_Dummy_Data);
    while((URT_GetITAllFlagStatus(URTX) & URT_IT_RX) !=URT_IT_RX);
    return((uint8_t)URT_GetRXData(URTX));
}

/**
 *******************************************************************************
 * @brief	   Control Flash WREN(write enable)
 * @details     
 * @param[in]   WREN: Flash write control
 *  @arg\b      ENABLE:  ENABLE Flash write.
 *  @arg\b      DISABLE: DISABLE Flash write.
 * @return                       
 *******************************************************************************
 */
void Sample_URT_WriteFlash_Cmd(FunctionalState WREN)
{
	uint32_t RDAT;
	
    do{
        
        /* Write enable */
        Flash_nCS = FLASH_nCS_ACTIVE;
        
        if(WREN==ENABLE)
        {
            Sample_Flash_SetData(1,FLASH_WRITE_ENABLE);
            Flash_nCS = FLASH_nCS_NoACTIVE;
            
            /* Read status register */
	        Flash_nCS = FLASH_nCS_ACTIVE;
	        Sample_Flash_SetData(1,FLASH_READ_STATUS);
            
            RDAT = Sample_Flash_ReadData();
            Flash_nCS = FLASH_nCS_NoACTIVE;
        }
        else
        {
            Sample_Flash_SetData(1,FLASH_WRITE_DISABLE);
            Flash_nCS = FLASH_nCS_NoACTIVE;
            return;
        }
        
        
    /* Check WEL == 1 */    
    }while((RDAT & 0x02)==0x00);
    
}
/**
 *******************************************************************************
 * @brief	    Wait program or erase to end. 
 * @details     
 * @return                 
 *******************************************************************************
 */ 
void Sample_URT_CheckFlashBusy(void)
{
    uint8_t RDAT;
    
    //============================================
    /* Read status register */
    Flash_nCS = FLASH_nCS_ACTIVE;
    
    
    Sample_Flash_SetData(1,FLASH_READ_STATUS);

    //===========================================
    /* Check erase or write complete */
    /* Get State */
    do{
        RDAT = Sample_Flash_ReadData();
        
    }while((RDAT & 0x01)==0x01);
    
	Flash_nCS = FLASH_nCS_NoACTIVE;   
    
}

/**
 *******************************************************************************
 * @brief	    Read the flash manufacturer ID and memory type + memory density.
 * @details     
 * @return      Flash RDID                
 *******************************************************************************
 */ 
uint32_t Sample_URT_ReadFlashRDID(void)
{
    ctype RDAT;
    
    Flash_nCS = FLASH_nCS_ACTIVE;     
    
    Sample_Flash_SetData(1,FLASH_READ_IDENTIFICATION);
    
    RDAT.B[0] = Sample_Flash_ReadData();
    RDAT.B[1] = Sample_Flash_ReadData();
    RDAT.B[2] = Sample_Flash_ReadData();
    RDAT.B[3] = 0;
    
    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    return(RDAT.W);
}
/**
 *******************************************************************************
 * @brief	  Read Configuration Register  
 * @details     
 * @return  
 *   @arg\b   1. Byte0 = Configuration register-1
 *   @arg\b   2. Byte1 = Configuration register-2              
 *******************************************************************************
 */ 
uint16_t Sample_URT_ReadFlashRDCR(void)
{
    ctype RDAT;
    
    Flash_nCS = FLASH_nCS_ACTIVE;
    
    Sample_Flash_SetData(1,FLASH_READ_CONFIGURATION);
    RDAT.B[0] = Sample_Flash_ReadData();
    RDAT.B[1] = Sample_Flash_ReadData();
    
    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    return(RDAT.H[0]);
}


/**
 *******************************************************************************
 * @brief  	    Flash erase
 * @details  
 * @param[in]   EraseType: Erase type
 * 	@arg\b	        FLASH_ERASE_SECTOR  : Flash sector erase ( sector unit is 4K byte)
 *  @arg\b          FLASH_ERASE_BLOCK32K: Flash erase unit is 32K bytes.
 *  @arg\b          FLASH_ERASE_BLOCK64K: Flash erase uint is 64K bytes.
 *  @arg\b          FLASH_ERASE_CHIP    : Flash All erase.
 * @param[in]   Page_Num : Set Erase size.
 * 	@arg\b          if EraseType = FLASH_ERASE_SECTOR   & Page_Num = 0 , erase address range is 0x000000 ~ 0x000FFF.
 * 	@arg\b          if EraseType = FLASH_ERASE_BLOCK32K & Page_Num = 0 , erase address range is 0x000000 ~ 0x007FFF.
 * 	@arg\b          if EraseType = FLASH_ERASE_BLOCK64K & Page_Num = 0 , erase address range is 0x000000 ~ 0x00FFFF.
 * 	@arg\b          if EraseType = FLASH_ERASE_CHIP     , the Page_Num value is don't care. 
 * @return	    
 * @note        No
 * @par         Example
 * @code
    API_Flash_Erase (FLASH_ERASE_SECTOR, 1);    // Erase address is 0x1000 ~ 0x1FFF
 * @endcode
 *******************************************************************************
 */
void Sample_URT_EraseFlash(Flash_ERASE_TypeDef EraseType , uint32_t Page_Num)
{
    ctype AddressIndex;

    
    switch(EraseType)
    {
        case FLASH_ERASE_CHIP:
                                  break;
        case FLASH_ERASE_BLOCK64K:
                                  AddressIndex.W = 0x10000 * Page_Num;
                                  break; 
        case FLASH_ERASE_BLOCK32K:
                                  AddressIndex.W = 0x8000  * Page_Num;
                                  break;
        case FLASH_ERASE_SECTOR:
                                  AddressIndex.W = 0x1000  * Page_Num;
                                  break;    
    }
    
    if(AddressIndex.W > FLASH_MAXSIZE)
    {
        return;
    }

    Sample_URT_WriteFlash_Cmd(ENABLE);
    
    
    Flash_nCS = FLASH_nCS_ACTIVE;
    
    Sample_Flash_SetData(1,EraseType);
    
    if(EraseType!=FLASH_ERASE_CHIP)
    {
        Sample_Flash_SetData(1,AddressIndex.B[2]);
        Sample_Flash_SetData(1,AddressIndex.B[1]);
        Sample_Flash_SetData(1,AddressIndex.B[0]);
    }

    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    
    Sample_URT_CheckFlashBusy();
    
    
}
/**
 *******************************************************************************
 * @brief	   Read flash data
 * @details     
 * @param[in]  FlashX: flash control parameter.
 * @return                      
 *******************************************************************************
 */ 
void Sample_URT_ReadFlash(FlashCTR_TypeDef* FlashX)
{
    uint32_t  Tmp;
    
    
    Flash_nCS = FLASH_nCS_ACTIVE;
    
    //============================================
    //Read Command
    Sample_Flash_SetData(1, FLASH_FAST_READ);
    //===========================================
    //Read Address
    Sample_Flash_SetData(1, FlashX->Address.B[2]);
    Sample_Flash_SetData(1, FlashX->Address.B[1]);
    Sample_Flash_SetData(1, FlashX->Address.B[0]);
    Sample_Flash_SetData(1, Flash_Dummy_Data);
    
    //===========================================
    //Read Data
    for(Tmp=0;Tmp<FlashX->Total_Length;Tmp++)
    {
        FlashX->rBuffer[Tmp] = Sample_Flash_ReadData();
    }
    
    Flash_nCS = FLASH_nCS_NoACTIVE;
}

/**
 *******************************************************************************
 * @brief	   Program flash 
 * @details     
 * @param[in]  FlashX: flash control parameter.
 * @return                         
 *******************************************************************************
 */
void Sample_URT_ProgramFlash(FlashCTR_TypeDef *FlashX)
{
    uint16_t DataCnt;
    ctype    AddressIndex;

    
    AddressIndex.W = FlashX->Address.W;
    FlashX->DataCnt = 0;
    
    do{
        if(FlashX->Total_Length > FLASH_PROGRAM_MAX)
        {
            FlashX->BLen = FLASH_PROGRAM_MAX;
        }
        else
        {
            FlashX->BLen = FlashX->Total_Length;
        }
    
        Sample_URT_WriteFlash_Cmd(ENABLE);
        
        Flash_nCS = FLASH_nCS_ACTIVE;
        
        //===============================================
        //Program command
        Sample_Flash_SetData(1, FLASH_PAGE_PROGRAM);
        
        //==============================================
        //Program Addresss
        Sample_Flash_SetData(1,AddressIndex.B[2]);
        Sample_Flash_SetData(1,AddressIndex.B[1]);
        Sample_Flash_SetData(1,AddressIndex.B[0]);
        
        //===============================================
        //Data0 ~ DataN
        for(DataCnt = 0; DataCnt < FlashX->BLen; DataCnt++)
        {
            Sample_Flash_SetData(1,FlashX->pBuffer[FlashX->DataCnt]);
            FlashX->DataCnt = FlashX->DataCnt + 1;
        }
      
        
        Flash_nCS = FLASH_nCS_NoACTIVE;
        
        AddressIndex.W = AddressIndex.W + FlashX->BLen;
        FlashX->Total_Length = FlashX->Total_Length - FlashX->BLen;
        
        Sample_URT_CheckFlashBusy(); 
    }while(FlashX->Total_Length!=0);
    

}

/**
 *******************************************************************************
 * @brief	   Set URT mode to SPI mode0
 * @details     
 * @return                      
 *******************************************************************************
 */ 
void Sample_URT_SPIMode_Init(void)
{
    URT_BRG_TypeDef  URT_BRG;
    URT_Data_TypeDef DataDef;
    
    //==Set CSC init
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 11.0592M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->Port B/URT0 = Enable
    //Configure Peripheral On Mode Clock->URT0->Select URT0_PR Source = CK_APB(11.0592)
    
    //==Set GPIO init
    //1. MOSI Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin8
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin8 mode is PPO (Push pull output)
    //    (4).Pin8 pull-up resister Enable
    //    (5).Pin8 function URT0_TX
    //2. MISO Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin9
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin8 mode is ODO (Open drain)
    //    (4).Pin8 pull-up resister Enable
    //    (5).Pin9 function URT0_RX
    //3. SPICLK Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOC->Pin0
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin8 mode is PPO (Push pull output)
    //    (4).Pin8 pull-up resister Enable
    //    (5).Pin3 function URT0_CLK
    //4. NSS Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin10
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin8 mode is PPO (Push pull output)
    //    (4).Pin8 pull-up resister Enable
    //    (5).Pin3 function GPIO (the NSS is software control)
    
    
    //=====Set Clock=====//
    //---Set BaudRate---//
    URT_BRG.URT_InternalClockSource = URT_BDClock_PROC;
    URT_BRG.URT_BaudRateMode = URT_BDMode_Separated;
    URT_BRG.URT_PrescalerCounterReload = 1;	                //Set PSR
    URT_BRG.URT_BaudRateCounterReload  = 0;	                //Set RLR
    URT_BaudRateGenerator_Config(URTX, &URT_BRG);		    //BR115200 = f(CK_URTx)/(PSR+1)/(RLR+1)/(OS_NUM+1)
    URT_BaudRateGenerator_Cmd(URTX, ENABLE);	            //Enable BaudRateGenerator
    //---TX/RX Clock---//
    URT_TXClockSource_Select(URTX, URT_TXClock_Internal);	//URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(URTX, URT_RXClock_Internal);	//URT_RX use BaudRateGenerator
    URT_TXOverSamplingSampleNumber_Select(URTX, 3);	        //Set TX OS_NUM
    URT_RXOverSamplingSampleNumber_Select(URTX, 3);	        //Set RX OS_NUM
    URT_RXOverSamplingMode_Select(URTX, URT_RXSMP_3TIME);
    URT_TX_Cmd(URTX, ENABLE);	                            //Enable TX
    URT_RX_Cmd(URTX, ENABLE);	                            //Enable RX
    

    //=====Set Mode=====//
    //---Set Data character config---//
    DataDef.URT_TX_DataLength  = URT_DataLength_8;
    DataDef.URT_RX_DataLength  = URT_DataLength_8;
    DataDef.URT_TX_DataOrder   = URT_DataTyped_MSB;
    DataDef.URT_RX_DataOrder   = URT_DataTyped_MSB;
    DataDef.URT_TX_Parity      = URT_Parity_No;
    DataDef.URT_RX_Parity      = URT_Parity_No;
    DataDef.URT_TX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_RX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_TX_DataInverse = DISABLE;
    DataDef.URT_RX_DataInverse = DISABLE;
    URT_DataCharacter_Config(URTX, &DataDef);
    //---Set Mode Select---//
    URT_Mode_Select(URTX, URT_SYNC_mode);
    //---Set DataLine Select---//
    URT_DataLine_Select(URTX, URT_DataLine_2);
    
    //=====Set Error Control=====//
    // to do...
    
    //=====Set Bus Status Detect Control=====//
    // to do...
    
    //==Set SPI Mode = (CPHA = 0 , CPOL = 0)====//
    URT_CPHAMode_Select( URTX , URT_CPHA0_LeadEdge);
    URT_CPOLMode_Select( URTX , URT_CPOL0_Low );

    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    //=====Set Data Control=====//
    URT_TXGuardTime_Select(URTX, 0);


    //=====Enable CLK(SYSCLK)====//
    URT_CLKSignal_Cmd(URTX,ENABLE);
    
    //=====Enable URT=====//
    URT_Cmd(URTX, ENABLE);
		
    
}
/**
 *******************************************************************************
 * @brief	   Simple flash sample code. 
 * @details     
 * 	@arg\b     1. Check Flash ID
 *  @arg\b     2. Erase Flash (sector erase)
 *  @arg\b     3. Program Flash.
 *  @arg\b     4. Read Flash.
 * @return                     
 *******************************************************************************
 */ 
void Sample_URT_MX25R512FFlash(void)
{
    uint8_t Tmp;
    
    
    //==================================
    //Inital URT mode = SPI mode.
    Sample_URT_SPIMode_Init();
    
    //==================================
    //Check Flash ID
    if(Sample_URT_ReadFlashRDID()!= FLASH_RDID_CMP)
    {
        //RDID fail to do......
    }
    else
    {
        //============================
        //Flash sector erase 
        Sample_URT_EraseFlash(FLASH_ERASE_SECTOR,1);          //Erase sector 1 ( Address = 0x1000 ~ 0x1FFF)
        
        //===========================
        //Flash program
        Flash0.Address.W = 0x1048;                            //Define address of flash program. 
        Flash0.pBuffer = &Sample_URT_ProgramFlashBuffer[0];   //Define datas of flash program.
        Flash0.Total_Length = 10;                             //Define size of flash program.
        Sample_URT_ProgramFlash(&Flash0);                     
        
        //===========================
        //Flash read
        Flash0.rBuffer = &Sample_URT_ReadFlashBuffer[0];      //Define storage address of receive data from flash.
        Flash0.Total_Length = 10;                             //Define size of read data from flash.
        Sample_URT_ReadFlash(&Flash0);                        //Flash address is Flash0.Address.W . 
        
        //==========================
        //Compare data
        for(Tmp=0;Tmp<Flash0.Total_Length;Tmp++)
        {
            //----------------------------------------
            //Read data and write data is no same.
            if(Sample_URT_ReadFlashBuffer[Tmp] != Sample_URT_ProgramFlashBuffer[Tmp])
            {
                //to do......                  
            }
        }
    }
}


















