#include "GLOBAL.H"
#include "MCU.H"
#include "KEYBOARD.H"
#include "USB.H"
#include "USB_DSC.H"


//-----MCU.C-----//
extern void Reboot_Mcu( BYTE );
extern void Delay_Xms( BYTE );
extern void UART_Send_Byte(BYTE dta);

//-----KEYBOARD.C-----//
extern bdata BYTE LED_STATUS;
extern KEYBOARD KB;
extern void KB_LED_Status( void );
extern void KB_LED_Off( void );
extern void Initial_KB( void );


EP0INFO Ep0;


void USB_Write_SFR( BYTE Addr , BYTE Data )
  { while(USBADR & BUSY);                        //wait for BUSY free ( free = 0 );
    USBADR = Addr;
    USBDAT = Data;
  }

BYTE USB_Read_SFR( BYTE Addr )
  { BYTE var;
    
    
    while(USBADR & BUSY);                        //wait for BUSY free ( free = 0 );
    USBADR = Addr;
    var = USBDAT;
    return var;
  }


void Initial_USB( void )
  { BYTE tmp;
    
    
    USB_Write_SFR(IEN, CLR);                     // Disable USB all related interrupts
    
    USB_Write_SFR(EPINDEX, EP0);                 // Change to EP0 
    
    tmp = ( TXEPEN | RXEPEN );
    USB_Write_SFR(EPCON, tmp);                   // Enable Tx & Rx
    
    tmp = USB_Read_SFR(EPCON);
    tmp |= ( TXSTL | RXSTL );
    USB_Write_SFR(EPCON, tmp);                   // Set RXSTL & TXSTL (prevent not SETUP Token)
    
    USB_Write_SFR(TXCON, TXCLR);
    USB_Write_SFR(RXCON, RXCLR);
    
    tmp = ( UTXIE0 | URXIE0 | UTXIE1 | ASOFIE );
    USB_Write_SFR(UIE, tmp);
 
    tmp = ( EF | EFSR );
    USB_Write_SFR(IEN, tmp);                     // Enable USB all related interrupts
    
    Ep0.Unit = EP0_MAX;                          // 
    Ep0.EmuOk = CLR;                             // Check Emulation status
    Ep0.RWEN = CLR;                              // Remote/Wakeup flag ( Default " Disable " )
    
    tmp = USB_Read_SFR(UPCON);
    tmp |= CONEN;
    USB_Write_SFR(UPCON, tmp);                   // Connecter to Host ( Enable the pull-up resistor )
  }


void USB_Rd_FIFO( BYTE *Buffer , BYTE Cnt )
  { BYTE i;
    
    
    for ( i=0 ; i<Cnt ; i++ )  
       Buffer[i] = USB_Read_SFR(RXDAT);
  }


void USB_Wr_FIFO( BYTE *Buffer , BYTE Cnt )
  { BYTE i;
    
    
    for ( i=0 ; i<Cnt ; i++ )  
       USB_Write_SFR(TXDAT, Buffer[i]);
  }


void USB_Ep0_FIFO( void )
  { BYTE BLen, tmp;
    
    
    USB_Write_SFR(TXCON, TXCLR);                 // Clear Tx FIFO

    tmp = USB_Read_SFR(RXSTAT);
    if ( tmp & ( STOVW | EDOVW ))                // Setup Token
      { Ep0.Stage = SETUPSTAGE;
        while(1)
          { tmp = USB_Read_SFR(RXSTAT);
            while(tmp & STOVW)                  // waiting STOVE = 0
              { tmp = USB_Read_SFR(RXSTAT);
              }
            
            tmp = USB_Read_SFR(RXSTAT);
            while(!( tmp & EDOVW ))              // waiting EDOVW = 1
              { tmp = USB_Read_SFR(RXSTAT);
              }
            
            tmp = USB_Read_SFR(RXSTAT);
            tmp &= ~EDOVW;
            USB_Write_SFR(RXSTAT, tmp);          // Clear the EDOVW bit when reading the contents of the FIFO
            
            USB_Write_SFR(UIFLG, URXD0);         // Write "1" to Clear this Flag
            
            BLen = USB_Read_SFR(RXCNT);          // Chk total Rx Data count in Byte
            USB_Rd_FIFO( Ep0.RxTx , BLen );
            
            tmp = USB_Read_SFR(RXSTAT);
            if (!(tmp & ( STOVW | EDOVW )))
              break;
          }
        
        tmp = USB_Read_SFR(EPCON);
        tmp &= ~( RXSTL | TXSTL );
        USB_Write_SFR(EPCON, tmp);               // Release Rx/Tx STAL
		
        tmp = USB_Read_SFR(RXSTAT);
        tmp &= ~RXSETUP;
        USB_Write_SFR(RXSTAT, tmp);              // Release Setup Token flag

        Ep0.All = BLen;                          // Only for analytic "Urd"
      }
    else if ( Ep0.Stage == DFU_STAGE )
      { BLen = USB_Read_SFR(RXCNT);             // Chk total Rx Data count in Byte  
      	Ep0.All -= BLen;
      	
      	USB_Rd_FIFO( Ep0.RxTx , BLen );
      	if ( BLen == EP0_MAX )
      	  { USB_Write_SFR(TXCNT, 0);
      	    Ep0.Stage = Ep0.RxTx[0];
      	  }
      	else
      	  { LED_STATUS = Ep0.RxTx[0];
      	    KB_LED_Status();
            USB_Write_SFR(TXCNT, 0);
            Ep0.Stage = STATUSSTAGE;
      	  }
      }
    else                                         // Zero Length Data in RXCNT
      { Ep0.Stage = STATUSSTAGE;
        
        tmp = USB_Read_SFR(EPCON);
        tmp |= ( RXSTL | TXSTL );
        USB_Write_SFR(EPCON, tmp);               // Receive the host "ACK" transaction , we should set Rx/Tx STAL	
      }
    
    tmp = USB_Read_SFR(RXCON);
    tmp |= RXFFRC;
    USB_Write_SFR(RXCON, tmp);                   // Set this bit , meaning that Rx Data has Read Complete
  }


void USB_Get_Status( void )
  { BYTE tmp;
    
    
    Ep0.All = 2;                                 // Only 2 byte transfer to the host
    
    Ep0.RxTx[1] = 0;
    switch( Ep0.RxTx[0] & 0x03 )                 // Request Type ( Reserve low 2 bit )
      { case DEVICEREQUEST:    if ( Ep0.RWEN )
                                 Ep0.RxTx[0] = 0x02;// Return Function Remove Wake-up Enable
                               else
                                 Ep0.RxTx[0] = 0x00;// Return Function Remove Wake-up Disable
                               break;            // Chk Remote wakeup enabled or not
        case ENDPOINTREQUEST:  tmp = (Ep0.RxTx[4] & 0x0F);
                               USB_Write_SFR(EPINDEX, tmp);
                               
                               tmp = USB_Read_SFR(EPCON);
                               if (( tmp & RXSTL )||( tmp & TXSTL ))
                                 Ep0.RxTx[0] = 0x01;// if EndPoint Rx/Tx STAL then set EndPoint Halt
                               else
                                 Ep0.RxTx[0] = 0x00;// else seting this EndPoint Avaliable for Rx/Tx  
                               USB_Write_SFR(EPINDEX, EP0);
                               break;
        case INTERFACEREQUEST: break;
        default:               tmp = USB_Read_SFR(EPCON);
                               tmp |= ( RXSTL | TXSTL );
                               USB_Write_SFR(EPCON, tmp);
                               break; 
      }
  }  	


void USB_Clear_Feature( void )
  { BYTE tmp;
    
    
    switch( Ep0.RxTx[0] & 0x03 )                 // Request Type ( Reserve low 2 bit )
      { case DEVICEREQUEST:    if ( Ep0.RxTx[2] == DEVICE_REMOTE_WAKEUP )
                                 Ep0.RWEN = CLR;
                               else
                                 { tmp = USB_Read_SFR(EPCON);
                                   tmp |= ( RXSTL | TXSTL );
                                   USB_Write_SFR(EPCON, tmp);
                                 }
                               break;            // Disable the Device Remote Wakeup function
        case ENDPOINTREQUEST:  if ( Ep0.RxTx[2] == ENDPOINT_HALT )
                                 { tmp = (Ep0.RxTx[4] & 0x0F);
                                   USB_Write_SFR(EPINDEX, tmp);
                                   
                                   tmp = USB_Read_SFR(EPCON);
                                   tmp &= ~( RXSTL | TXSTL );
                                   USB_Write_SFR(EPCON, tmp);
                                   USB_Write_SFR(RXSTAT, RXSOVW);// Enable RXSEQ/TXSEQ bit can be Updata , and
                                   USB_Write_SFR(TXSTAT, TXSOVW);// set Rx/Tx toggle buffer into DATA0
                                   USB_Write_SFR(EPINDEX, EP0);
                                 }
                               else
                                 { tmp = USB_Read_SFR(EPCON);
                                   tmp |= ( RXSTL | TXSTL );
                                   USB_Write_SFR(EPCON, tmp);
                                 }
                               break;
        case INTERFACEREQUEST: break;
        default:               tmp = USB_Read_SFR(EPCON);
                               tmp |= ( RXSTL | TXSTL );
                               USB_Write_SFR(EPCON, tmp);
                               break;
      }
  }
 
  
void USB_Set_Feature( void )
  { BYTE tmp;
	switch( Ep0.RxTx[0] & 0x03 )                 // Request Type ( Reserve low 2 bit )
      { case DEVICEREQUEST:    if ( Ep0.RxTx[2] == DEVICE_REMOTE_WAKEUP )
                                 Ep0.RWEN = SET;
                               else
                                 { tmp = USB_Read_SFR(EPCON);
                                   tmp |= ( RXSTL | TXSTL );
                                   USB_Write_SFR(EPCON, tmp);
                                 }
                               break;            // Disable the Device Remote Wakeup function
        case ENDPOINTREQUEST:  if ( Ep0.RxTx[2] == ENDPOINT_HALT )
                                 { tmp = (Ep0.RxTx[4] & 0x0F);
                                   USB_Write_SFR(EPINDEX, tmp);
                                   
                                   tmp = USB_Read_SFR(EPCON);
                                   tmp |= ( RXSTL | TXSTL );
                                   USB_Write_SFR(EPCON, tmp);
                                   USB_Write_SFR(EPINDEX, EP0);
                                 }
                               else
                                 { tmp = USB_Read_SFR(EPCON);
                                   tmp |= ( RXSTL | TXSTL );
                                   USB_Write_SFR(EPCON, tmp);
                                 }
                               break;
        case INTERFACEREQUEST: break;
        default:               tmp = USB_Read_SFR(EPCON);
                               tmp |= ( RXSTL | TXSTL );
                               USB_Write_SFR(EPCON, tmp);
                               break; 
      }
  }  


void USB_Get_Descriptor( void )
  { BYTE tmp;
	union WTYPE WLen;
    
    
    WLen.B[0] = 0x00;                            // MSB
    Ep0.All = Ep0.RxTx[7];                       // MSB
    Ep0.All <<= 8;
    Ep0.All += Ep0.RxTx[6];                      // LSB
    switch( Ep0.RxTx[3] )	
      { case DEVICEDESCRIPTOR:        Ep0.Buf = DEVICE_DESCRIPTOR;
                                      WLen.B[1] = Ep0.Buf[0];
                                      break;
        case CONFIGURATIONDESCRIPTOR: Ep0.Buf = CONGFIGURATION_DESCRIPTOR;
                                      WLen.B[1] = Ep0.Buf[2];// LSB
                                      break;
        case STRINGDESCRIPTOR:        switch( Ep0.RxTx[2] )
                                        { case 0:  Ep0.Buf = LANGUAGEID_DESCRIPTOR;
                                                   break;
                                          case 1:  if ( DEVICE_DESCRIPTOR[14] )
                                                     Ep0.Buf = MANUFACTURER_DESCRIPTOR;
                                                   else
                                                     { tmp = USB_Read_SFR(EPCON);
                                                       tmp |= ( RXSTL | TXSTL );
                                                       USB_Write_SFR(EPCON, tmp);
                                                     }
                                                   break;
                                          case 2:  if ( DEVICE_DESCRIPTOR[15] )
                                                     Ep0.Buf = PRODUCT_DESCRIPTOR;
                                                   else
                                                     { tmp = USB_Read_SFR(EPCON);
                                                       tmp |= ( RXSTL | TXSTL );
                                                       USB_Write_SFR(EPCON, tmp);
                                                     }
                                                   break;
                                          case 3:  if ( DEVICE_DESCRIPTOR[16] )
                                                     Ep0.Buf = SERIALNUMBER_DESCRIPTOR;
                                                   else
                                                     { tmp = USB_Read_SFR(EPCON);
                                                       tmp |= ( RXSTL | TXSTL );
                                                       USB_Write_SFR(EPCON, tmp);
                                                     }
                                                   break;
                                          default: tmp = USB_Read_SFR(EPCON);
                                                   tmp |= ( RXSTL | TXSTL );
                                                   USB_Write_SFR(EPCON, tmp);
                                                   break;
                                        }
                                      WLen.B[1] = Ep0.Buf[0];
                                      break;
        case HIDDESCRIPTOR:           Ep0.Buf = KB_HID_DESCRIPTOR;
                                      WLen.B[1] = Ep0.Buf[0];
                                      break;
      	case HIDREPORT:               Ep0.Buf = KB_HID_REPORT;
      	                              WLen.B[1] = KB_HID_DESCRIPTOR[7];
      	                              WLen.B[0] = KB_HID_DESCRIPTOR[8];
      	                              break;
      	default:                      tmp = USB_Read_SFR(EPCON);
                                      tmp |= ( RXSTL | TXSTL );
                                      USB_Write_SFR(EPCON, tmp);
                                      break;
      }
    
    if ( Ep0.All > WLen.W )
      Ep0.All = WLen.W;
    
  }


void USB_Set_Configuration( void )
  { BYTE i, tmp;
    
    
    Ep0.Tmp = Ep0.RxTx[2];
    if ( Ep0.Tmp )
      { for ( i=1 ; i<6 ; i++ )
           { USB_Write_SFR(EPINDEX, i);
           
             tmp = ( TXEPEN | RXEPEN );
             USB_Write_SFR(EPCON, tmp);          // Enable Receive Input/Transmit Output
             USB_Write_SFR(RXCON, RXCLR);        // Clear Rx FIFO
             USB_Write_SFR(TXCON, TXCLR);        // Clear Tx FIFO
             USB_Write_SFR(RXSTAT, RXSOVW);      // Enable RXSEQ/TXSEQ bit can be Updata , and
             USB_Write_SFR(TXSTAT, TXSOVW);      // set Rx/Tx toggle buffer into DATA0
           }
        Ep0.EmuOk = SET;                         // Emulation Flow pass
      }
    else                                         // Set EP1,2,3,4 into Address state
      { for ( i=1 ; i<6 ; i++ )
           { USB_Write_SFR(EPINDEX, i);
             USB_Write_SFR(EPCON, 0x00);
           }
      }
    USB_Write_SFR(EPINDEX, EP0);
  }


void USB_Set_Interface( void )
  { BYTE tmp;
    
    
    switch( Ep0.RxTx[4] )
      { case 0:  if ( Ep0.RxTx[2] > 0 )          // For Interface1 ( HID )
                   { tmp = USB_Read_SFR(EPCON);
                     tmp |= ( RXSTL | TXSTL );
                     USB_Write_SFR(EPCON, tmp);
                   }
                 else
                   { USB_Write_SFR(TXCNT, 0x00);
                   }
                 break;
        default: tmp = USB_Read_SFR(EPCON);
                 tmp |= ( RXSTL | TXSTL );
                 USB_Write_SFR(EPCON, tmp);
                 break;
      }
  }


void USB_Get_Interface( void )
  { BYTE tmp;
    
    
    Ep0.All = 1;
    switch( Ep0.RxTx[4] )
      { case 0:  Ep0.RxTx[0] = 0;                // For Interface1 ( HID )
                 break;
        default: tmp = USB_Read_SFR(EPCON);
                 tmp |= ( RXSTL | TXSTL );
                 USB_Write_SFR(EPCON, tmp);
                 break;
      }
  }


void USB_CtrlRd( void )                          // Host In , USB Out ( Only for EP0 )
  { BYTE BLen, tmp;
    
    
    if ( Ep0.Stage == DATASTAGE )                // In DATASTAGE we should move Data to TXFIFO
      { if ( Ep0.All > Ep0.Unit )
          BLen = Ep0.Unit;
        else
          BLen = Ep0.All;
      	USB_Wr_FIFO( Ep0.Buf , BLen );
      	USB_Write_SFR(TXCNT, BLen);              // Set this byte will Trigger USB to Transmit Data to the Host
        Ep0.All -= BLen;                         // Calculated the Remain Data size
        Ep0.Buf += BLen;                         // Move Buffer Address in Right position
      }
    else
      { tmp = USB_Read_SFR(EPCON);
        tmp |= ( RXSTL | TXSTL );
        USB_Write_SFR(EPCON, tmp);               // In STATUSSTAGE we should STAL Rx/Tx
        if ( Ep0.Stage == SETADDRESS )           // Different from other STATUSSTAGE(importent)
          { USB_Write_SFR(UADDR, Ep0.Tmp);       // Set Address
          }
        if ( Ep0.Stage == DFU_RESET )
          { Reboot_Mcu( REBOOT_TO_ISP );
          }
      }
  }


void USB_Stardard_Request( void )
  { BYTE tmp;
    
    
    switch( Ep0.RxTx[1] )                        // Request Code
      { case GET_STATUS:        Ep0.Stage = DATASTAGE;
                                USB_Get_Status();
                                USB_CtrlRd();
                                break;
        case CLEAR_FRATURE:	Ep0.Stage = STATUSSTAGE;  
                                USB_Clear_Feature();
                                USB_Write_SFR(TXCNT, 0x00);// USB will return ACK immediately when receive IN transaction
                                break;
        case SET_FEATURE:       Ep0.Stage = STATUSSTAGE;
                                USB_Set_Feature();
                                USB_Write_SFR(TXCNT, 0x00);// USB will return ACK immediately when receive IN transaction
                                break;
        case SET_ADDRESS:       Ep0.Stage = SETADDRESS;// Different from other STATUSSTAGE
                                Ep0.Tmp = Ep0.RxTx[2];
                                USB_Write_SFR(TXCNT, 0x00);// USB will return ACK immediately when receive IN transaction
                                break;
        case GET_DESCRIPTOR:    Ep0.Stage = DATASTAGE;
                                USB_Get_Descriptor();
                                USB_CtrlRd();
                                break;
//        case SET_DESCRIPTOR:    break;
        case GET_CONFIGURATION: Ep0.Stage = DATASTAGE;
                                Ep0.RxTx[0] = Ep0.Tmp;// This value get from SET_CONFIGURATION transaction
                                Ep0.All = 1;     // Only 1 byte transfer to the host
                                USB_CtrlRd();
                                break;
        case SET_CONFIGURATION: Ep0.Stage = STATUSSTAGE;
                                USB_Set_Configuration(); // Will store configuration value to Ep0.Tmp
                                USB_Write_SFR(TXCNT, 0x00);// USB will return ACK immediately when receive IN transaction
                                break;
        case SET_INTERFACE:     Ep0.Stage = STATUSSTAGE;
                                USB_Set_Interface();
                                break;
        case GET_INTERFACE:     Ep0.Stage = DATASTAGE;
                                USB_Get_Interface();
                                USB_CtrlRd();
                                break;
        case SYNCH_FRAME:       Ep0.Stage = STATUSSTAGE;
                                USB_Write_SFR(TXCNT, 0x00);// USB will return ACK immediately when receive IN transaction
                                Ep0.All = 0;
                                break;
        default:                tmp = USB_Read_SFR(EPCON);
                                tmp |= ( RXSTL | TXSTL );
                                USB_Write_SFR(EPCON, tmp);
                                break;           // Set Rx/Tx STAL 
      }
  }


void USB_Class_Request( void )
  { BYTE tmp;
    
    
    switch( Ep0.RxTx[1] )                        // Request Code
      { case GET_IDLE:     Ep0.Stage = DATASTAGE;
                           Ep0.RxTx[0] = KB.Key.IdleRate;
                           Ep0.All = 1;          // Only 1 byte transfer to the host
                           USB_CtrlRd();
                           break;
        case GET_PROTOCOL: Ep0.Stage = DATASTAGE;
                           Ep0.RxTx[0] = KB.Key.Protocol;
                           Ep0.All = 1;          // Only 1 byte transfer to the host
                           USB_CtrlRd();
                           break;
        case GET_REPORT:   Ep0.Stage = DATASTAGE;
                           Ep0.RxTx[0] = LED_STATUS;
                           Ep0.All = 1;          // Only 1 byte transfer to the host
                           USB_CtrlRd();
                           break;
        case SET_IDLE:     Ep0.Stage = STATUSSTAGE;
                           KB.Key.IdleRate = Ep0.RxTx[3];
                           USB_Write_SFR(TXCNT, 0x00);// USB will return ACK immediately when receive IN transaction
                           break;
        case SET_PROTOCOL: Ep0.Stage = STATUSSTAGE;
                           KB.Key.Protocol = Ep0.RxTx[2];
                           USB_Write_SFR(TXCNT, 0x00);// USB will return ACK immediately when receive IN transaction
                           break;
        case SET_REPORT:   Ep0.Stage = DFU_STAGE;
                           Ep0.All = Ep0.RxTx[7];
                           Ep0.All <<= 8;
                           Ep0.All += Ep0.RxTx[6];
                           break;
        default:           tmp = USB_Read_SFR(EPCON);
                           tmp |= ( RXSTL | TXSTL );
                           USB_Write_SFR(EPCON, tmp);
                           break;                // Set Rx/Tx STAL 
      }
  }


void USB_CtrlWr( void )                          // Host Out , USB In ( Only for EPO )
  { BYTE tmp;
    
    
    Ep0.Buf = Ep0.RxTx;                          // Move Buffer address to RxTx[8] array , Use for USB_CtrlRd();
    USB_Ep0_FIFO();                              // Move Rx Data to RxTxBuf buffer
    if ( Ep0.Stage == SETUPSTAGE )               // if Setup Transection will set Ep0.All = 8
      {	Ep0.All = 0;
        switch( Ep0.RxTx[0] & 0x60 )             // Request Type
          { case STANDARD_REQUEST: USB_Stardard_Request();
                                   break;
            case CLASS_REQUEST:    USB_Class_Request();
                                   break;
            default:               tmp = USB_Read_SFR(EPCON);
                                   tmp |= ( RXSTL | TXSTL );
                                   USB_Write_SFR(EPCON, tmp);
                                   break;       // Set Rx/Tx STAL 
          }
      }    
  }


void USB_Set_TxStatus( BYTE Flag )
  { BYTE Status;
    
    
    Status = USB_Read_SFR(TXCON);
    Status &= 0x0F;                              // Reserve Low nibble
    Status |= Flag;
    USB_Write_SFR(TXCON, Status);                // Clear Tx FIFO
  }



void USB_Int( void )
  { BYTE Status, tmp;
  
    
    Status = USB_Read_SFR(UPCON);                // Status in Power Control Register
    
    if ( Status & 0x0F )                         // Into Power Control mode
      { if ( Status & USUS )                     // Suspend mode
          { Status &= 0xF0;
            Status = ( Status | USUS );
            USB_Write_SFR(UPCON, Status);        // Write "1" to Clear this Flag
            
            Status = USB_Read_SFR(UPCON);        // Status in Power Control Register
            if (( Status & 0x0F ) == 0x00 )      // Into Power Control mode
              { KB_LED_Off();                    // Set all LED Off
              
              	if ( Ep0.RWEN == SET )
              	  { //P1 = 0x00;                   // Set scan out line to low
//                    P2 = 0x00;                   // mean for wakeup function
//                    P3 &= 0xE7;                  // P3.3 / P3.4 ( scan out line )
                    
                    P3 = 0xFF;
                    KBMASK = 0xFF;               // Will Enable KP Interrupt
                  }
                P0 = 0;
                P1 = 0;
                P2 = 0x80;
                P3 = 0x06;
                P4 = 0;
                P5 = 0;
                SFRPI = 1;
                PUCON1 = 0x1C;
                P6 = 0x1F;
                SFRPI = 0;
             	
             	PCON0 |= 0x02;                    // Set CPU(8051) into PowerDown mode
              	
              	Delay_Xms( 2 );
      	        
     	        if ( Ep0.RWEN == SET )
      	          { if ( KBMASK == 0x00 )        // Clear by KP ISR in " MCU.C "
      	              { tmp = USB_Read_SFR(UPCON);
                        tmp |= URWU;
                        USB_Write_SFR(UPCON, tmp);// Set USB into Wake-up mode
                      }
      	            else
      	              KBMASK = 0x00;             // Will Disable KP Interrupt
//                    P1 = 0xFF;
//                    P2 = 0xFF;
//                    P3 |= 0x18;
      	          }

      	        if ( Ep0.EmuOk == SET )          // Set USB LED ON after " SET_CONFIGURATION "
      	          KB_LED_Status();               // Restore LED status
              }
          }
        else
          { // Status = USB_Read_SFR(UPCON);
            if ( Status & URST )                 // Reset mode
              { Status &= 0xF0;
                Status = ( Status | URST );
                USB_Write_SFR(UPCON, Status);    // Write "1" to Clear this Flag
                Initial_KB();
                Initial_USB();
              }
            // Status = USB_Read_SFR(UPCON);
            else if ( Status & URSM )            // Resume mode  
              { Status &= 0xF0;
                //USB[UPCON] = ( Status | URSM );  // Write "1" to Clear this Flag
                Status = ( Status | URSM );
                Delay_Xms( 1 );
                USB_Write_SFR(UPCON, Status);
              }
          }
      }
    else  
      { Status = USB_Read_SFR(UIFLG);
        if ( Status & UTXD1 )                    // EP1 Transmit
          { USB_Write_SFR(UIFLG, UTXD1);         // Write "1" to Clear this Flag
            USB_Write_SFR(EPINDEX, EP1);
            USB_Set_TxStatus( TXFFRC );
            if ( KB.Key.Busy == SET )
              KB.Key.Busy = CLR;
          }
        else if ( Status & URXD0 )               // EP0 Receive
          { USB_Write_SFR(UIFLG, URXD0);         // Write "1" to Clear this Flag
            USB_Write_SFR(EPINDEX, EP0);
            USB_CtrlWr();                        // USB Standard Device Request(maybe)
          }
        else if ( Status & UTXD0 )               // EP0 Transmit
          { USB_Write_SFR(UIFLG, UTXD0);         // Write "1" to Clear this Flag
            USB_Write_SFR(EPINDEX, EP0);
            USB_CtrlRd();
          } 
      }	  
  }