#include "drv_iic_soft.h"


#define IIC_SCL_H()   GPIO_SetBits(GPIOB, GPIO_Pin_10)      /* SCL = 1 */
#define IIC_SCL_L()   GPIO_ResetBits(GPIOB, GPIO_Pin_10)    /* SCL = 0 */

#define IIC_SDA_H()     GPIO_SetBits(GPIOB, GPIO_Pin_11)    /* SDA = 1 */
#define IIC_SDA_L()     GPIO_ResetBits(GPIOB, GPIO_Pin_11)  /* SDA = 0 */
#define IIC_SDA_READ()  GPIO_ReadInputDataBit(GPIOB, GPIO_Pin_11)     /* Read SDA Pin */


/**
  * @name    IIC_Init
  * @brief   Initial soft IIC GPIO
  * @param   No
  * @retval  No
  */
void IIC_Init(void)
{
  /* Open GPIO clock */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_BMX1 |RCC_APB1Periph_GPIOB, ENABLE);

  /* Configure to Opendrain, need a pull-up resistor. */
  GPIO_Init(GPIOB, GPIO_Pin_10 |GPIO_Pin_11, GPIO_MODE_OUT |GPIO_OTYPE_OD);

  /* Giving a STOP signal, reset I2C devices on the I2C bus to standby mode. */
  IIC_Stop();
}

static void iic_delay(void)
{ // #CONFIG#
  volatile uint32_t n = 100;
  while(n--);
}

/**
  * @name    IIC_Start
  * @brief   Generate IIC START signal
  * @param   No
  * @retval  No
  */
void IIC_Start(void)
{
  IIC_SDA_H();    /* SDA pull high */
  IIC_SCL_H();    /* SCL pull high */
  iic_delay();
  IIC_SDA_L();    /* SDA pull low */
  iic_delay();
}

/**
  * @name    IIC_Stop
  * @brief   Generate IIC STOP signal
  * @param   No
  * @retval  No
  */
void IIC_Stop(void)
{
  IIC_SDA_L();    /* SDA pull low */
  IIC_SCL_H();    /* SCL pull high */
  iic_delay();
  IIC_SDA_H();    /* SDA pull high */
  iic_delay();
}

/**
  * @name    IIC_Ack
  * @brief   Generate IIC ACK signal
  * @param   No
  * @retval  No
  */
void IIC_Ack(void)
{
  IIC_SDA_L();    /* SDA pull low */
  IIC_SCL_L();    /* SCL pull low */
  iic_delay();
  IIC_SCL_H();    /* SCL pull high */
  iic_delay();
  IIC_SCL_L();    /* SCL pull low */
  iic_delay();
  IIC_SDA_H();    /* Release SDA BUS */
}

/**
  * @name    IIC_NAck
  * @brief   Generate IIC NACK signal
  * @param   No
  * @retval  No
  */
void IIC_NAck(void)
{
  IIC_SDA_H();    /* SDA pull high */
  IIC_SCL_L();    /* SCL pull low */
  iic_delay();
  IIC_SCL_H();    /* SCL pull high */
  iic_delay();
  IIC_SCL_L();    /* SCL pull low */
  iic_delay();
}

/**
  * @name    IIC_WaitAck
  * @brief   Wait for ACK signal coming
  * @param   No
  * @retval  1: receive ACK failed
  *          0: receive ACK succeed
  */
uint8_t IIC_WaitAck(void)
{
  IIC_SDA_H();    /* Release SDA BUS */
  iic_delay();
  IIC_SCL_H();    /* SCL pull high */
  iic_delay();
  
  if(IIC_SDA_READ())
  {
    IIC_SCL_L();    /* SCL pull low */
    iic_delay();
    IIC_Stop();
    return 1;   /* receive ACK failed */
  }
  else
  {
    IIC_SCL_L();    /* SCL pull low */
    iic_delay();
    return 0;   /*receive ACK succeed*/
  }
}

/**
  * @name    IIC_SendByte
  * @brief   Send one byte data
  * @param   data: data need to be sent
  * @retval  No
  */
void IIC_SendByte(uint8_t data)
{
  uint8_t bit_of_byte = 0x80;
  
  while (bit_of_byte)
  {
    IIC_SCL_L();    /* SCL pull low */
    if(data & bit_of_byte) {
      IIC_SDA_H();
    }
    else {
      IIC_SDA_L();
    }
    bit_of_byte >>= 1;
    iic_delay();
    IIC_SCL_H();    /* SCL pull high */
    iic_delay();
  }
  IIC_SCL_L();    /* SCL pull low */
}


/**
  * @name    IIC_ReadByte
  * @brief   Read one byte data
  * @param   No
  * @retval  Read data received
  */
uint8_t IIC_ReadByte(void)
{
  uint8_t bit_of_byte = 0x80;
  uint8_t receive_byte = 0;
  
  while (bit_of_byte)
  {
    IIC_SCL_L();    /* SCL pull low */
    iic_delay();
    IIC_SCL_H();    /* SCL pull high */
    iic_delay();
    if(IIC_SDA_READ()) {
      receive_byte |= bit_of_byte;
    }
    bit_of_byte >>= 1;
  }
  
  IIC_SCL_L();    /* SCL pull low */
  
  return receive_byte;
}

/**
  * @name    IIC_CheckDevice
  * @brief   Check I2C BUS devices, CPU send device address, judge whether the device exists through whether there is ACK
  * @param   No
  * @retval  0: Detect device succeed
  *          1: Detect device failed
  */
uint8_t IIC_CheckDevice(uint8_t address)
{
  uint8_t ack;
  
  IIC_Start();
  IIC_SendByte(address | IIC_WR);
  ack = IIC_WaitAck();
  IIC_Stop();
  
  return ack;
}



