/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=12MHz, SysCLK=12MHz
*	 Description:
*			UART0_ARGB_Mode	(TXD0/P3.1)
*			Master
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)


#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

typedef	struct
{
	BYTE h;
	BYTE s;
	BYTE v;
} HsvColor;

#define	DUTY_1_3		3		// Duty	1/3
#define	DUTY_1_4		4		// Duty	1/4
#define	ARGB_DUTY			DUTY_1_4

bit	argb_update_flag;
#define	ColorCycle_Frame_Time_Default		32 //32*1ms	= 32ms
u8 ColorCycle_R;
u8 ColorCycle_G;
u8 ColorCycle_B;
u8 ColorCycle_Counter;
u8 ColorCycle_Frame_Value;
u8 ColorCycle_Frame_Time_Value;
HsvColor hsv;


/***********************************************************************************
*Function:		void INT_T0(void)
*Description:	T0 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_T0(void) interrupt	INT_VECTOR_T0
{
	  ColorCycle_Counter ++;
	  TH0=TIMER_1T_1ms_TH;
	  TL0=TIMER_1T_1ms_TL;
}

/***********************************************************************************
*Function:		void INT_UART0(void)
*Description:	UART0 Interrupt	handler
*Input:
*Output:
*************************************************************************************/
void INT_UART0(void) interrupt INT_VECTOR_UART0
{
	_push_(SFRPI);		   //Protect SFRPI

	SFR_SetPage(0);		   //S0CON:Page	0 only
	if(TI0)
	{
	}
	if(RI0)
	{
		IO_LED_R=!IO_LED_R;
		argb_update_flag=1;
		RI0	= 0;
	}
	_pop_(SFRPI);		   //restore SFRPI
}

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void Uart0SPITransceiver(u8	S0_DATA)
*Description:	UART0 ARGB Master transmit data
*Input:			u8 S0_DATA
*Output:
*************************************************************************************/
void UART0ARGBTransceiver(u8 S0_DATA)
{
	S0BUF=S0_DATA;										// Send	data
	while(TI0 == 0);									// Wait	complete
	TI0=0;//UART0_ARGB_ClearAGBF();						// Clear flag
}

/***********************************************************************************
*Function:		void InitUart0_ARGB(void)
*Description:	Initialize Uart0 for ARGB
*Input:
*Output:
*************************************************************************************/
void InitUart0_ARGB(void)
{
	UART0_SetModeARGB();				// Set UART1 as	ARGB master	mode
	UART0_SetRxTxP30P31();				// Set TXD0(P31)

#if(ARGB_DUTY==DUTY_1_3)
	UART0_SetARGBDuty_1_3();
#else
	UART0_SetARGBDuty_1_4();
#endif
	UART0_SetARGBResetTime_BaudRate_x8x20();

	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source:YSCLK
	UART0_SetS0BRGValue(256-((MCU_SYSCLK/750000)/ARGB_DUTY));
	UART0_EnS0BRG();					//	Enable S0BRG
	UART0_SetDataMSB();

	UART0_SetARGBTI0Interrupt_Disable();

	argb_update_flag=0;
}

/***********************************************************************************
*Function:		void InitTimer0()
*Description:	Initialize Timer0
*Input:
*Output:
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// TIMER0 Mode:	16-bit
	TM_SetT0Clock_SYSCLK();					// TIMER0 Clock	source:	SYSCLK
	TM_SetT0Gate_Disable();
	TM_SetT0LowByte(TIMER_1T_1ms_TL);		// Set TL0 value
	TM_SetT0HighByte(TIMER_1T_1ms_TH);		// Set TH0 value
	TM_EnableT0();							// Enable T0
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// SetP22,P24,P26 as push-pull(LED)
	PORT_SetP3PushPull(BIT1);						// Set P31(TXD0) as	push-pull
}

/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		void InitInterrupt()
*Description:	Initialize Interrupt
*Input:
*Output:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER0();			//	Enable Timer0 interrupt
	INT_EnUART0();			//	Enable UART0 interrupt
}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitUart0_ARGB();
	InitTimer0();
	InitInterrupt();

}

//************************ColorCycle Lighting Effect*****************************
void m_HsvToRgb(BYTE Data)
{
	unsigned char region, remainder, p,	q, t;

	if (hsv.s == 0)
	{
		ColorCycle_R = hsv.v;
		ColorCycle_G = hsv.v;
		ColorCycle_B = hsv.v;
		return ;
	}

	/* make	region 0-5 */
	region = Data /	43;
	  /* find remainder	part, make it from 0-255 */
	remainder =	(Data -	(region	* 43)) * 6;

	p =	(WORD)(hsv.v * (255	- hsv.s)) >> 8;
	q =	(WORD)(hsv.v * (255	- ((WORD)(hsv.s	* remainder) >>	8))) >>	8;
	t =	(WORD)(hsv.v * (255	- ((WORD)(hsv.s	* (255 - remainder)) >>	8))) >>	8;

	switch (region)
	{
		case 0:
					  ColorCycle_R = hsv.v;
					ColorCycle_G = t;
					ColorCycle_B = p;
					break;
		case 1:
					  ColorCycle_R = q;
					ColorCycle_G = hsv.v;
					ColorCycle_B = p;
					break;
		case 2:
					  ColorCycle_R = p;
					ColorCycle_G = hsv.v;
					ColorCycle_B = t;
					break;
		case 3:
					  ColorCycle_R = p;
					ColorCycle_G = q;
					ColorCycle_B = hsv.v;
					break;
		case 4:
					  ColorCycle_R = t;
					ColorCycle_G = p;
					ColorCycle_B = hsv.v;
					break;
		default:
					  ColorCycle_R = hsv.v;
					ColorCycle_G = p;
					ColorCycle_B = q;
					break;
	}
}

void ColorCycle_Lighting_Effect(void)
{
	char x;

	if (ColorCycle_Counter >= ColorCycle_Frame_Time_Default)
	  {
			ColorCycle_Counter = CLR;

			//P32 =	~P32;	//Check	32ms Frame Time
			m_HsvToRgb(ColorCycle_Frame_Value);//Color Cycle Frame 0~255, total	256	frame =	256*31ms = 7.936sec
			ColorCycle_Frame_Value ++;

		for	(x=0; x<10;	x++)//About	frame rate 1.72ms
		{

			UART0ARGBTransceiver(ColorCycle_G>>2);
			UART0ARGBTransceiver(ColorCycle_R>>2);
			UART0ARGBTransceiver(ColorCycle_B>>2);
		}
	}
}
//*******************************************************************************

void main()
{
	u8 x;

	InitSystem();

	hsv.s=255;
	hsv.v=175;

	INT_EnAll();

	while (!argb_update_flag);

	for	(x=0; x<64;	x++)//About	frame rate 1.74ms
	{
		UART0ARGBTransceiver(0);//G
		UART0ARGBTransceiver(0);//R
		UART0ARGBTransceiver(0);//B
	}

	while(1)
	{
		DelayXms(1);
		x++;
		if(x>=100)
		{
			x=0;
			IO_LED_G=!IO_LED_G;
		}
		if (argb_update_flag)
		{
			ColorCycle_Lighting_Effect();
			argb_update_flag = 0;
		}
	}
}



