/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=24MHz, SysCLK=24MHz
*	 Description:
*			Split T2 mode: DutyCapture
*			Split T3 mode: DutyCapture
*
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
**********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		24000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_TEST_0		P12
#define	IO_TEST_1		P13

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

bit	bT2Captureing;
bit	bT2CaptureFirst;
bit	bT2CaptureOk;
u8 TH2Cnt;
WordTypeDef	T2CaptureFirst;
WordTypeDef	T2CaptureSecond;

bit	bT3Captureing;
bit	bT3CaptureFirst;
bit	bT3CaptureOk;
u8 TH3Cnt;
WordTypeDef	T3CaptureFirst;
WordTypeDef	T3CaptureSecond;

/***********************************************************************************
*Function:		void INT_T2(void)
*Description:	T2 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_T2(void) interrupt	INT_VECTOR_T2
{
	_push_(SFRPI);
	SFR_SetPage(0);			// TF2,EXF2	only Page-0
	if(TF2)
	{
		TF2=0;
		if(bT2Captureing==TRUE)
		{
			TH2Cnt++;
		}

	}
	if(EXF2)
	{
		EXF2=0;
		if(bT2Captureing==TRUE)
		{
			if(bT2CaptureFirst==FALSE)
			{
				bT2CaptureFirst=TRUE;
				T2CaptureFirst.B.BHigh=TH2Cnt;
				T2CaptureFirst.B.BLow=RCAP2H;
			}
			else
			{
				bT2Captureing=FALSE;
				bT2CaptureOk=TRUE;
				T2CaptureSecond.B.BHigh=TH2Cnt;
				T2CaptureSecond.B.BLow=TH2;
			}
		}
		IO_TEST_0=!IO_TEST_0;
	}
	if(RCLK_TF2L)
	{
		IO_LED_R=!IO_LED_R;
		RCLK_TF2L=0;
	}
	_pop_(SFRPI);
}


/***********************************************************************************
*Function:		void INT_T3(void)
*Description:	T3 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_T3(void) interrupt	INT_VECTOR_T3
{
	_push_(SFRPI);
	SFR_SetPage(1);				// TF3,EXF3	only Page-1
	if(TF3)
	{
		TF3=0;
		if(bT3Captureing==TRUE)
		{
			TH3Cnt++;
		}

	}
	SFR_SetPage(1);				// TF3,EXF3	only Page-1
	if(EXF3)
	{
		EXF3=0;
		if(bT3Captureing==TRUE)
		{
			if(bT3CaptureFirst==FALSE)
			{
				bT3CaptureFirst=TRUE;
				T3CaptureFirst.B.BHigh=TH3Cnt;
				T3CaptureFirst.B.BLow=RCAP3H;
			}
			else
			{
				bT3Captureing=FALSE;
				bT3CaptureOk=TRUE;
				T3CaptureSecond.B.BHigh=TH3Cnt;
				T3CaptureSecond.B.BLow=TH3;
			}
		}
		IO_TEST_1=!IO_TEST_1;
	}
	SFR_SetPage(1);				// TF3L	only Page-1
	if(TF3L)
	{
		TF3L=0;
		IO_LED_Y=!IO_LED_Y;
	}
	_pop_(SFRPI);
}

/***********************************************************************************/
/*************************************************
*Function:		char putchar (char c)
*Description:	send for printf
*Input:			char c
*Output:
*************************************************/
char putchar (char c)
{
	bit	bES;
	bES=ES0;
	ES0=0;
	S0BUF =	c;
	while(TI0==0);
	TI0=0;
	ES0=bES;
	return 0;
}

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void InitTimer2(void)
*Description:	Initialize Timer2
*Input:
*Output:
*************************************************************************************/
void InitTimer2(void)
{
	TM_SetT2Mode_DutyCapture();						// T2 mode:	DutyCapture
	TM_SetT2Clock_SYSCLKDiv12();					// T2 clock	source:	SYSCLK/12=24MHz/12=2MHz	0.5us
	TM_SetTL2Clock_SYSCLKDiv12();					// T2L clock source: SYSCLK/12=24MHz/12=2MHz 0.5us
	TM_EnableAccessTF2L();							// Enable access TF2L and TF2L interrupt
	TM_DisableTR2LC();								// Disable TR2LC
	TM_EnT2Split();									// Enable split	T2

	TM_SetT2LowByte(256-200);						// T2 low byte 200*0.5us=100us
	TM_SetT2RLLowByte(256-200);						// T2 low byte reload

	TM_SetT2HighByte(0);							// T2 high byte
	TM_SetT2RLHighByte(0);							// T2 high byte	reload

	TM_SetT2Capture_T2EXPin();						// T2 EXI: T2EX	pin
	TM_SetT2CKOT2EX_P10P11();						// T2EX	pin: P11
	TM_EnableT2EX_DetectFalling();					// T2EX	detect:	Falling
	TM_EnableT2EX_DetectRising();					// T2EX	detect:	Rising

	TM_EnableTL2();									// Enable TL2

}

/***********************************************************************************
*Function:		void InitTimer3(void)
*Description:	Initialize Timer3
*Input:
*Output:
*************************************************************************************/
void InitTimer3(void)
{
	TM_SetT3Mode_DutyCapture();						// T3 mode:	DutyCapture
	TM_SetT3Clock_SYSCLKDiv12();					// T3 clock	source:	SYSCLK/12=24MHz/12=2MHz	0.5us
	TM_SetTL3Clock_SYSCLKDiv12();					// T3L clock source: SYSCLK/12=24MHz/12=2MHz 0.5us
	TM_DisableTR3LC();								// Enable access TF3L and TF3L interrupt
	TM_EnT3Split();									// Enable split	T3

	TM_SetT3LowByte(256-200);						// T3 low byte 200*0.5us=100us
	TM_SetT3RLLowByte(256-200);						// T3 low byte reload

	TM_SetT3HighByte(0);							// T3 high byte
	TM_SetT3RLHighByte(0);							// T3 high byte	reload

	TM_SetT3Capture_T3EXPin();						// T3 EXI: T3EX	pin
	TM_SetT3CKOT3EX_P33P32();						// T3/T3CKO:P33,T3EX:P32
	TM_EnableT3EX_DetectFalling();					// T3EX	detect:	Falling
	TM_EnableT3EX_DetectRising();					// T3EX	detect:	Rising


	TM_EnableTL3();									// Enable TL3
}

/***********************************************************************************
*Function:		void InitInterrupt()
*Description:	Initialize Interrupt
*Input:
*Output:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER2();
	INT_EnTIMER2L();
	INT_EnTIMER3();
	INT_EnTIMER3L();
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// Set P22,P24,P26 as push-pull(LED)
	PORT_SetP1PushPull(BIT2|BIT3);					// Set P12,P13as push-pull
	PORT_SetP1OpenDrainPu(BIT1);					// Set P11 as open-drain with pull-high	for	T2EX
	PORT_SetP3QuasiBi(BIT0|BIT1|BIT2);				// Set P30,P31 as Quasi-Bidirectional for UART,P32 as Quasi-Bidirectional for T3EX
}

/***********************************************************************************
*Function:		void InitUart0_S0BRG()
*Description:	Initialize Uart0, The source of	baud rate was S0BRG
*Input:
*Output:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R.	source:	S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source: Sysclk
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_24000000_1T);

	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{

	InitPort();
	InitClock();
	InitUart0_S0BRG();

	InitTimer2();
	InitTimer3();

	InitInterrupt();

	INT_EnAll();			// Enable global interrupt


}

void StartT2CaptureDuty(void)
{
	bT2CaptureFirst=FALSE;
	bT2CaptureOk=FALSE;
	bT2Captureing=TRUE;
	TM_SetT2HighByte(0);
	TH2Cnt=0;
	TM_EnableT2();
}

void StartT3CaptureDuty(void)
{
	bT3CaptureFirst=FALSE;
	bT3CaptureOk=FALSE;
	bT3Captureing=TRUE;
	TM_SetT3HighByte(0);
	TH3Cnt=0;
	TM_EnableT3();
}

void main()
{
	float duty;
	float freq;

	InitSystem();

	IO_LED_G=0;IO_LED_R=0;IO_LED_Y=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_R=1;IO_LED_Y=1;

	StartT2CaptureDuty();
	StartT3CaptureDuty();

	while(1)
	{
		IO_LED_G=0;
		DelayXms(200);
		IO_LED_G=1;
		DelayXms(100);

		if(bT2CaptureOk)
		{
			duty=(float)(((double)(T2CaptureFirst.W)*100L)/(float)T2CaptureSecond.W);
			freq=(float)(1000)/((float)(T2CaptureSecond.W)/2);
			printf("\nT2 Caputre: Duty=%f,Freq=%fKHz",duty,freq);
			StartT2CaptureDuty();
		}
		if(bT3CaptureOk)
		{
			duty=(float)(((double)(T3CaptureFirst.W)*100L)/(float)T3CaptureSecond.W);
			freq=(float)(1000)/((float)(T3CaptureSecond.W)/2);
			printf("\nT3 Caputre: Duty=%f,Freq=%fKHz",duty,freq);
			StartT3CaptureDuty();
		}
	}
}


