/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=32MHz, SysCLK=32MHz
*	 Description:
*			Set	P10,P11,P15,P16	for	INT1,INT0,INT2,INT3, falling edge trigger
*			20s	no interruption	trigger, enter Power-Down mode
*			INT1(P10),INT0(P15),INT2(P11),INT3(P16)	wake up	at low level
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		32000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	IO_nINT0		P10
#define	IO_nINT1		P15
#define	IO_nINT2		P11
#define	IO_nINT3		P16


u8 WakeUpSorce;
u8 INTCnt;

/***********************************************************************************
*Function:		void INT_INT0(void)
*Description:	INT0 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_INT0(void)		interrupt INT_VECTOR_INT0
{
	WakeUpSorce=0;
}

/***********************************************************************************
*Function:		void INT_INT1(void)
*Description:	INT1 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_INT1(void)		interrupt INT_VECTOR_INT1
{
	WakeUpSorce=1;
}

/***********************************************************************************
*Function:		void INT_INT2(void)
*Description:	INT2 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_INT2(void)		interrupt INT_VECTOR_INT2
{
	WakeUpSorce=2;
}

/***********************************************************************************
*Function:		void INT_INT3(void)
*Description:	INT3 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_INT3(void)		interrupt INT_VECTOR_INT3
{
	WakeUpSorce=3;
}

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		u8 IAP_ReadPPage(u8	PsfrAddr)
*Description:	Read P page	sfr
*Input:
*				u8 PsfrAddr: sfr Address
*Output:
*				u8 : sfr data
*************************************************************************************/
u8 IAP_ReadPPage(u8	PsfrAddr)
{
	bit	bEA=EA;
	EA = 0;						//Turn off Interrupt
	IFADRH = 0;					//IFADRH must be 0
	IFADRL=	PsfrAddr;			//sfr Address
	IFMT = ISP_READ_P;
	ISPCR =	0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR =	0;					//clear
	EA = bEA;					//restore interrupt
	return IFD;
}

/***********************************************************************************
*Function:		void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
*Description:	write P	page sfr
*Input:
*				u8 PsfrAddr: sfr Address
*				u8 PsfrData: sfr data
*Output:
*************************************************************************************/
void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
{
	bit	bEA=EA;
	EA = 0;						//Turn off Interrupt
	IFADRH = 0;					//IFADRH must be 0
	IFADRL=	PsfrAddr;			//sfr Address
	IFD= PsfrData;				//sfr data
	IFMT = ISP_WRITE_P;
	ISPCR =	0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR =	0;					//clear
	EA = bEA;					//restore interrupt
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// Set P22,P24,P26 as push-pull(LED)
	PORT_SetP1OpenDrainPu(BIT0|BIT1|BIT5|BIT6);		// Set P10,P11,P15,P16 as open-drain with pull-high	for	INT
}

/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		void InitInterrupt_WakeUp()
*Description:	Initialize Interrupt
*Input:
*Output:
*************************************************************************************/
void InitInterrupt_WakeUp(void)
{
	INT_DisINT0();							// Disable INT0	interrupt
	INT_DisINT1();							// Disable INT1	interrupt
	INT_DisINT2();							// Disable INT2	interrupt
	INT_DisINT3();							// Disable INT3	interrupt
	INT_DisAll();							// Disable global interrupt
}

/***********************************************************************************
*Function:		void InitInterrupt_PD()
*Description:	Initialize Interrupt
*Input:
*Output:
*************************************************************************************/
void InitInterrupt_PD(void)
{
	INT_EnINT0();							// Enable INT0 interrupt
	INT_EnINT1();							// Enable INT1 interrupt
	INT_EnINT2();							// Enable INT2 interrupt
	INT_EnINT3();							// Enable INT3 interrupt
	INT_EnAll();							// Enable global interrupt
}

/***********************************************************************************
*Function:	 void InitINT0(void)
*Description:	Initialize INT0
*Input:
*Output:
*************************************************************************************/
void InitINT0(void)
{
	INT_SetINT0P10();						// nINT0 : P10
	INT_SetINT0_DetectEdge();				// nINT0 Detect	type: edge,	On Power-Down mode,only	Level.
	INT_SetINT0_DetectLowFalling();			// nINT0 Detect: low level/falling edge
	INT_SetINT0Filter_SysclkDiv6_x3();		// nINT0 filter: (sysclk/6)*3

}

/***********************************************************************************
*Function:	 void InitINT1(void)
*Description:	Initialize INT1
*Input:
*Output:
*************************************************************************************/
void InitINT1(void)
{
	INT_SetINT1P15();						// nINT1 : P15
	INT_SetINT1_DetectEdge();				// nINT1 Detect	type: edge . On	Power-Down mode,only Level.
	INT_SetINT1_DetectLowFalling();			// nINT1 Detect: low level/falling edge
	INT_SetINT1Filter_SysclkDiv6_x3();		// nINT1 filter: (sysclk/6)*3

}

/***********************************************************************************
*Function:	 void InitINT2(void)
*Description:	Initialize INT2
*Input:
*Output:
*************************************************************************************/
void InitINT2(void)
{
	INT_SetINT2P11();						// nINT2 : P11
	INT_SetINT2_DetectEdge();				// nINT2 Detect	type: edge . On	Power-Down mode,only Level.
	INT_SetINT2_DetectLowFalling();			// nINT2 Detect: low level/falling edge
	INT_SetINT2Filter_SysclkDiv6_x3();		// nINT2 filter: (sysclk/6)*3

}

/***********************************************************************************
*Function:	 void InitINT3(void)
*Description:	Initialize INT3
*Input:
*Output:
*************************************************************************************/
void InitINT3(void)
{
	INT_SetINT3P16();						// nINT3 : P16
	INT_SetINT3_DetectEdge();				// nINT3 Detect	type: edge . On	Power-Down mode,only Level.
	INT_SetINT3_DetectLowFalling();			// nINT3 Detect: low level/falling edge
	INT_SetINT3Filter_SysclkDiv6_x3();		// nINT3 filter: (sysclk/6)*3

}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitINT0();
	InitINT1();
	InitINT2();
	InitINT3();
	InitInterrupt_WakeUp();
}

/***********************************************************************************
*Function:	 void PowerDownProc(void)
*Description:	Power down
*Input:
*Output:
*************************************************************************************/
void PowerDownProc(void)
{
	u8 i;
	u8 BakCKCON2;													// Backup CKCON2 (if used PLL CKM)

	// MCK=OSCin
	BakCKCON2=IAP_ReadPPage(CKCON2_P);								// Backup CKCON2 (if used PLL CKM)
	IAP_WritePPage(CKCON2_P,BakCKCON2&(~(MCKS0_P|MCKS1_P)));		// MCK=OSCin (if used PLL CKM)

	i=0;
	while(1)
	{
		DelayXus(10);
		if((IO_nINT0==0)||(IO_nINT1==0)||(IO_nINT2==0)||(IO_nINT3==0))
		{
			i=0;
		}
		else
		{
			i++;
			if(i>=200) break;
		}
	}
	InitInterrupt_PD();

	POW_SetMode_PD();												// MCU enter Power-Down	mode
	_nop_();

	InitInterrupt_WakeUp();

	while((CKCON1&PLLOCKF)==0);										// delay for stabilize CKM (if used	PLL	CKM)
	IAP_WritePPage(CKCON2_P,BakCKCON2);								// restore CKCON2,use CKM (if used PLL CKM)

}

void main()
{
	u8 i;

	InitSystem();
	IO_LED_G=0;IO_LED_Y=0;IO_LED_R=0;
	DelayXms(2000);
	IO_LED_G=1;IO_LED_Y=1;IO_LED_R=1;
	INTCnt=0;
	WakeUpSorce=0;
	while(1)
	{
		for(i=0;i<20;i++)
		{
		DelayXms(100);
			if(WakeUpSorce == 0)
			{
				IO_LED_G=!IO_LED_G;
			}
			else if(WakeUpSorce	== 1)
			{
				IO_LED_R=!IO_LED_R;
			}
			else if(WakeUpSorce	== 2)
			{
				IO_LED_Y=!IO_LED_Y;
			}
			else if(WakeUpSorce	== 3)
			{
				IO_LED_G=!IO_LED_G;
				IO_LED_Y=!IO_LED_Y;
			}
		}
		IO_LED_G=1;
		IO_LED_Y=1;
		IO_LED_R=1;

		PowerDownProc();

	}

}


