/**
  ******************************************************************************
 *
 * @file        Sample_MID_TM36_IC3_DMA.c
 *
 * @brief       TM36 Rising edge capture with DMA access.
 *
 * @par         Project
 *              MG32
 * @version     V1.13
 * @date        2021/05/27
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
* @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_TM_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static TM_HandleTypeDef    mTM36;
static DMA_HandleTypeDef   mDMA;
static uint16_t            FallingEdgeCapture[4];

/* Private function prototypes -----------------------------------------------*/
void Sample_MID_TM36_IC3_DMA(void);
void MID_TM_IC_CaptureCallback(TM_HandleTypeDef* mTM);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	    TM36 Rising edge capture with DMA access.
 * @details     1.Initial TM36 for 16-bit of IC.
 *    \n        2.Initial TM36 Clock mode.
 *    \n        3.Setup input capture parameters
 *    \n        4.Initial DMA.
 *    \n        5.Start IC capture.
 * @return      None
 * @code
    Sample_MID_TM36_IC3_DMA();
    
    // wait for DMA complete.
    // to do ...
    
    
    // When DMA complete, then MCU will call "MID_TM_IC_CaptureCallback" function call
    // example:
    void MID_TM_IC_CaptureCallback(TM_HandleTypeDef* hTM)    
    {
        uint16_t tmpREG;
    
        tmpREG = FallingEdgeCapture[0] - FallingEdgeCapture[0]; 
        // to do ...
    }
 * @endcode
 *******************************************************************************
 */
void Sample_MID_TM36_IC3_DMA(void)
{  
    TM_IC_InitTypeDef       iConfig;
    TM_ClockConfigTypeDef   CKConfig;

    // make sure :
    
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //  Select CK_PLLI Divider = CK_HS/2
    //  Select CK_PLL Multiplication factor = CK_PLLIx16
    //  Select CK_PLLO Divider = CK_PLL/4
    //Select CK_MAIN Source = CK_PLLO
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->TM36 = Enable
    //Configure Peripheral On Mode Clock->DMA = Enable
    //Configure Peripheral On Mode Clock->GPIOB = Enable
	
    //==Set GPIO init 
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->PB7
    //GPIO port initial is 0xFFFF
    //PB7 mode is ODO with Pull-up
    //PB7 AFS function is TM36_IC3
        
    // ------------------------------------------------------------------------
    // 1.Initial TM36 for 16-bit of IC.
    // ------------------------------------------------------------------------
    mTM36.Instance                  = TM36;
    mTM36.Init.TM_CounterMode       = TM_SEPARATE_UP;
    mTM36.Init.TM_Period            = 65535;
    mTM36.Init.TM_Prescaler         = 0;
    MID_TM_IC_Init(&mTM36);
    
    // ------------------------------------------------------------------------
    // 2.Initial TM36 Clock mode.
    // ------------------------------------------------------------------------
    CKConfig.TM_ClockSource         = TM_INTERNAL_CLOCK;
    CKConfig.TM_ExternalClockSource = 0;
    CKConfig.TM_INTClockDivision    = TM_INTERNALCLOCK_DIVDER_DIV1;
    CKConfig.TM_InternalClockSource = TM_INTERNALCLOCK_PROC;
    
    MID_TM_ConfigClockSource(&mTM36, &CKConfig);  
    
    // ------------------------------------------------------------------------
    // 3.Setup input capture parameters:
    //      1. Trigger edge
    //      2. Trigger source from PIN
    //      3. Capture data mode
    // ------------------------------------------------------------------------
    iConfig.ICPolarity              = TM_ICPOLARITY_FALLING;        // Falling edge event to capture data
    iConfig.ICSelection             = MID_TM_INPUTMUX_PIN; 
    iConfig.ICDataMode              = MID_TM_IC_OVERWRITTEN;
    
    MID_TM_IC_ConfigChannel(&mTM36, &iConfig, MID_TM_Channel3);
    
    // ------------------------------------------------------------------------
    // 4.Initial DMA. (P2M)
    // ------------------------------------------------------------------------
    mDMA.Instance		            = DMA;
	mDMA.DMAChannelIndex            = DMAChannel0;                  // only DMA channel0 can support M2M mode  
	mDMA.Init.SrcPeri	            = MID_DMA_TM36_CAPTURE3;        // Source symbol is TM36 IC3
	mDMA.Init.DesPeri	            = MID_DMA_MEMORY_WRITE;         // Destination symbol is Memory
	mDMA.Init.BSize 	            = DMA_BSIZE_2BYTE;              // Burst size is 2 byte
	mDMA.Init.MEMMode	            = MID_DMA_MEMORY_NORMAL;        // Normal memory access mode    
    mDMA.Init.LoopMode	            = DMA_LOOP_DISABLE;             // DISABLE Loop mode
    mDMA.Parent                     = &mTM36;
    
	MID_DMA_Init(&mDMA);

    
    // ------------------------------------------------------------------------
    // 5.Start IC capture. (TM36 will capture 4 times by DMA.)
    // ------------------------------------------------------------------------
    mTM36.hDMA[3] = &mDMA;                                                                      
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
    #pragma clang diagnostic push
    #pragma clang diagnostic ignored "-Wcast-align"
        MID_TM_IC_Start_DMA(&mTM36, MID_TM_Channel3,(uint32_t *) &FallingEdgeCapture, 8);
    #pragma clang diagnostic pop
#else
    MID_TM_IC_Start_DMA(&mTM36, MID_TM_Channel3,(uint32_t *) &FallingEdgeCapture, 8);
#endif
    

    return;

}

/**
 *******************************************************************************
 * @brief       Input Capture callback in non blocking mode.
 * @param[in]   mTM : pointer to a TM_HandleTypeDef
 * @return      None
 *******************************************************************************
 */
void MID_TM_IC_CaptureCallback(TM_HandleTypeDef* mTM)
{
	// to do ...
    UNUSED(mTM);
}

