/**
  ******************************************************************************
 *
 * @file        Sample_MID_SPI_Slave_TXRX.c
 *
 * @brief       SPIx/URTx slave transmit and receive sample code
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2021/05/27
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2020 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_SPI_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define SPI0_NSS            PB0                             /*!< SPI0 NSS pin */
#define URT0_NSS            PB10                            /*!< URT0 NSS pin*/

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static SPI_HandleTypeDef    mSPI1;

/* Private function prototypes -----------------------------------------------*/
void Sample_MID_SPI_Slave_Transfer (MID_SPI_Struct* MID_Module);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief       SPI0/URTx slave transmit and receive.
 * @details     1.Initial SPI/URTx.
 *    \n        2.Clear RX buffer
 *    \n        3.Transmit data.
 *    \n        4.Receive data.
 *    \n        5.Transmit data.
 *    \n        6.Receive data.
 * @return      None
 *******************************************************************************
 */
void Sample_MID_SPI_Slave_Transfer (MID_SPI_Struct* MID_Module)
{  
    uint8_t     RX_BUF8[63]={0};
    uint8_t     CNT;


    
    // make sure :
    //===Set CSC init===
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->SPI0 = Enable
    //Configure Peripheral On Mode Clock->GPIOB = Enable


    /*===Set GPIO init SPI0 for SPI0===*/
    //1. MOSI Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin3
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin3 mode is DIN (Data input)
    //    (4).Pin3 pull-up resister Enable
    //    (5).Pin3 function SPI0_MOSI
    //2. MISO Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin1
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin1 mode is PPO (Push pull output)
    //    (4).Pin1 pull-up resister Enable
    //    (5).Pin1 function SPI0_MISO
    //3. SPICLK Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin2
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin2 mode is DIN (Data input)
    //    (4).Pin2 pull-up resister Enable
    //    (5).Pin2 function SPI0_CLK
    //4. NSS Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin10
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin0 mode is DIN (Data input)
    //    (4).Pin0 pull-up resister Enable
    //    (5).Pin0 function SPI0_NSS

    // ------------------------------------------------------------------------
    // 1.Initial SPI/URTx.
    // ------------------------------------------------------------------------
    mSPI1.Instance = MID_Module;
    mSPI1.Init.Mode = SPI_MODE_SLAVE_NSS;
    mSPI1.Init.NSSPMode = SPI_NSS_PULSE_DISABLE;
    mSPI1.Init.NSS = SPI_NSS_IN_HARDWARE;
    mSPI1.Init.DataLine = SPI_STANDARD_SPI;
    mSPI1.Init.DataSize = SPI_DATASIZE_8BIT;
    mSPI1.Init.CLKPolarity = SPI_POLARITY_LOW;
    mSPI1.Init.CLKPhase = SPI_PHASE_1EDGE;
    mSPI1.Init.ClockDivider = SPI_CLOCKDIVIDER_8;
    mSPI1.Init.FirstBit = SPI_FIRSTBIT_MSB;
    mSPI1.Init.NSSPMode = SPI_NSS_PULSE_DISABLE;
    if (MID_SPI_Init(&mSPI1) != MID_OK)                 // When SPI initial failure, stop at there.
    {
        while(1);
    }
    
    // SPI slave mode transmitted data directly update enable
#if defined(MG32_2ND) || defined(MG32_3RD) || defined(MG32_4TH)
    if((mSPI1.Init.Mode & SPI_MDS_MASK) == 0) 
        mSPI1.Instance->CR2.B[3] = SPI_CR2_TXUPD_EN_mask_b3;
#endif
    
    // ------------------------------------------------------------------------
    // 2.Clear RX buffer
    // ------------------------------------------------------------------------
    for(CNT=0; CNT<63; CNT++)                       
    {
        RX_BUF8[CNT] = 0;       // Clear RX_BUF data
    }

    // ------------------------------------------------------------------------
    // 3.Receive data
    // ------------------------------------------------------------------------
    MID_SPI_Receive(&mSPI1, &RX_BUF8[0], 63, 0xf0000000);
    
    // ------------------------------------------------------------------------
    // 4.Transmit data
    // ------------------------------------------------------------------------
    MID_SPI_Transmit(&mSPI1, &RX_BUF8[0], 63, 0x80000000);
    
    // ------------------------------------------------------------------------
    // 5.Receive data
    // ------------------------------------------------------------------------
    while((mSPI1.Instance->CR2.W & SPI_CR2_NSS_SWI_mask_w) == 0);
    MID_SPI_Receive(&mSPI1, &RX_BUF8[0], 63, 0xf0000000);
    
    // ------------------------------------------------------------------------
    // 6.Transmit data
    // ------------------------------------------------------------------------
    while((mSPI1.Instance->CR2.W & SPI_CR2_NSS_SWI_mask_w) == 0);
    MID_SPI_Transmit(&mSPI1, &RX_BUF8[0], 63, 0x80000000);
}



