/**
  ******************************************************************************
 *
 * @file        Sample_MID_SPI_Master_TXRX_DMA.c
 *
 * @brief       SPIx/URTx master transmit and receive use DMA sample code
 * @note        URTx not support slave mode.
 * @par         Project
 *              MG32 
 * @version     V1.03
 * @date        2022/09/22
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2020 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"
#include "MG32_ChipInit.h"
#include "MG32_DMA_MID.h"
#include "MG32_URT_MID.h"
#include "MG32_SPI_MID.h"
#include "MG32_GPIO_MID.h"
#include "MG32_EXIC_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static SPI_HandleTypeDef     uSPI0;
static SPI_HandleTypeDef     sSPI0;
//static URT_HandleTypeDef     MURT_TX_DMA;
static DMA_HandleTypeDef     DMA_uSPI;
static DMA_HandleTypeDef     DMA_uSPI_1;
static DMA_HandleTypeDef     DMA_sSPI;

#if (defined(MG32F02A132) || defined(MG32_3RD))
static uint8_t TST_TXData_Table[48] __attribute__((section(".ARM.__at_0x20003A00"))) = 
{   0x55, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47};
static uint8_t TST_RXData_Table[48] __attribute__((section(".ARM.__at_0x20003A30")));
#else
static uint8_t TST_TXData_Table[48] = 
{   0x55, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47};
static uint8_t TST_RXData_Table[48];
#endif

/* Private function prototypes -----------------------------------------------*/
void SPI0_IRQHandler(void);
void URT0_IRQHandler(void);
void DMA_IRQHandler(void);
void Sample_MID_SPI_Master_TXRX_DMA (void);
                               
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief       SPI0(sSPI) module IRQ
 * @details  
 * @return  
 * @note
 *******************************************************************************
 */
void SPI0_IRQHandler(void)
{
    MID_SPI_IRQHandler(&sSPI0);
}


/**
 *******************************************************************************
 * @brief       URT0(uSPI) module IRQ
 * @details  
 * @return  
 * @note
 *******************************************************************************
 */
void URT0_IRQHandler(void)
{
    MID_SPI_IRQHandler(&uSPI0);
}


/**
 *******************************************************************************
 * @brief       DMA module IRQ
 * @details  
 * @return  
 * @note
 *******************************************************************************
 */
void DMA_IRQHandler(void)
{
    MID_DMA_IRQHandler(&DMA_uSPI);
    MID_DMA_IRQHandler(&DMA_uSPI_1);
    MID_DMA_IRQHandler(&DMA_sSPI);
}


/**
 *******************************************************************************
 * @brief       SPI0/URTx master/slave transmit and receive use interrupt.
 * @details     1.URT0 Inital
 *    \n        2.SPI0 Inital
 *    \n        3.SPI Master (URT) RX DMA to SPI Slave (SPI) TX DMA.
 *    \n        4.SPI Master (URT) TX DMA to SPI Slave (SPI) RX DMA.
 * @return      None
 *******************************************************************************
 */
void Sample_MID_SPI_Master_TXRX_DMA (void)
{    
    GPIO_InitTypeDef    GPIOX;
    uint8_t             CNT;
    uint8_t             CLR_CNT;
 
    ChipInit();

    // ------------------------------------------------------------------------
    // 1.URT0 Inital
    // ------------------------------------------------------------------------
    GPIOX.Mode              = GPIO_MODE_PUSHPULL_O;
    GPIOX.Pull              = GPIO_NOPULLUP;
    GPIOX.Speed             = GPIO_SPEED_LOW;
    GPIOX.Inverse           = GPIO_INVERSE_DISABLE;
    GPIOX.OUTDrive          = GPIO_OUTDRIVE_LEVEL0;
    GPIOX.FilterDivider     = GPIO_FILTERDIVIDER_BYPASS;
    GPIOX.Alternate         = 3;
    MID_GPIO_Pin_Init(PINB(8),&GPIOX);              // MOSI
    MID_GPIO_Pin_Init(PINC(0),&GPIOX);              // CLK

    GPIOX.Alternate         = 0;    
//    MID_GPIO_Pin_Init(PINB(10),&GPIOX);             // NSS
    MID_GPIO_Pin_Init(PINB(0),&GPIOX);              // NSS
    
    GPIOX.Alternate         = 3;
    GPIOX.Mode              = GPIO_MODE_DIGITAL_I;
    GPIOX.Pull              = GPIO_PULLUP;
    MID_GPIO_Pin_Init(PINB(9),&GPIOX);              // MISO
    
    uSPI0.Instance          = SPI_URT0;
    uSPI0.Init.Mode         = SPI_MODE_MASTER;
    uSPI0.Init.DataLine     = SPI_STANDARD_SPI;
    uSPI0.Init.DataSize     = SPI_DATASIZE_8BIT;
    uSPI0.Init.CLKPolarity  = SPI_POLARITY_LOW;
    uSPI0.Init.CLKPhase     = SPI_PHASE_1EDGE;
    uSPI0.Init.NSS          = SPI_NSS_OUT_HARDWARE;
    uSPI0.Init.ClockDivider = SPI_CLOCKDIVIDER_512;
    uSPI0.Init.FirstBit     = SPI_FIRSTBIT_MSB;
    uSPI0.Init.Swap         = SPI_SWAP_DISABLE;
    uSPI0.Init.IdleDataOut  = SPI_IDLE_DATA_OUT_DRIVING_1;

    NVIC_EnableIRQ(URT0_IRQn);
    
    MID_SPI_Init(&uSPI0);
    

    // ------------------------------------------------------------------------
    // 2.SPI0 Inital
    // ------------------------------------------------------------------------
    GPIOX.Mode              = GPIO_MODE_PUSHPULL_O;
    GPIOX.Pull              = GPIO_NOPULLUP;
    GPIOX.Speed             = GPIO_SPEED_LOW;
    GPIOX.Inverse           = GPIO_INVERSE_DISABLE;
    GPIOX.OUTDrive          = GPIO_OUTDRIVE_LEVEL0;
    GPIOX.FilterDivider     = GPIO_FILTERDIVIDER_BYPASS;
    GPIOX.Alternate         = 2;                            
    MID_GPIO_Pin_Init(PINB(1),&GPIOX);              //MISO
    
    GPIOX.Mode              = GPIO_MODE_DIGITAL_I;
    GPIOX.Pull              = GPIO_PULLUP;
    MID_GPIO_Pin_Init(PINB(2),&GPIOX);              //CLK
    MID_GPIO_Pin_Init(PINB(3),&GPIOX);              //MOSI
    
#if defined(MG32_1ST)
    GPIOX.Alternate         = 2;
    MID_GPIO_Pin_Init(PINB(6),&GPIOX);              //NSSI
#endif
#if defined(MG32_2ND) || defined(MG32_3RD) || defined(MG32_4TH)
    GPIOX.Alternate         = 10;
    MID_GPIO_Pin_Init(PINB(10),&GPIOX);             //NSSI
#endif

    sSPI0.Instance          = SPI_SPI0;
    sSPI0.Init.Mode         = SPI_MODE_SLAVE_NSSI;
    sSPI0.Init.DataLine     = SPI_STANDARD_SPI;
    sSPI0.Init.DataSize     = SPI_DATASIZE_8BIT;
    sSPI0.Init.CLKPolarity  = SPI_POLARITY_LOW;
    sSPI0.Init.CLKPhase     = SPI_PHASE_1EDGE;
    sSPI0.Init.NSS          = SPI_NSS_IN_HARDWARE;
    sSPI0.Init.ClockDivider = SPI_CLOCKDIVIDER_2;
    sSPI0.Init.FirstBit     = SPI_FIRSTBIT_MSB;
    sSPI0.Init.Swap         = SPI_SWAP_DISABLE;
    sSPI0.Init.IdleDataOut  = SPI_IDLE_DATA_OUT_DRIVING_1;
    
    NVIC_EnableIRQ(SPI0_IRQn);
    
    MID_SPI_Init(&sSPI0);
    

    // ------------------------------------------------------------------------
    // 3.SPI Master (URT) TX and RX DMA to SPI Slave (SPI) TX DMA.
    // ------------------------------------------------------------------------
    // Initial uSPI DMA init struct 
    DMA_uSPI_1.Instance         = DMA;
    DMA_uSPI_1.DMAChannelIndex  = DMAChannel2;              // only DMA channel2 can support M2P mode 
    DMA_uSPI_1.Init.SrcPeri     = MID_DMA_MEMORY_READ;      // Source symbol is memory
    DMA_uSPI_1.Init.DesPeri     = MID_DMA_URT0_WRITE;       // Destination symbol is URT0 TDAT
    DMA_uSPI_1.Init.BSize       = DMA_BSIZE_1BYTE;          // Burst size is 1 byte
    DMA_uSPI_1.Init.MEMMode     = MID_DMA_MEMORY_NORMAL;    // Normal memory access mode    
    DMA_uSPI_1.Init.LoopMode    = DMA_LOOP_DISABLE;         // DISABLE Loop mode     
    DMA_uSPI_1.Parent = &uSPI0;
    
    DMA_uSPI.Instance           = DMA;
    DMA_uSPI.DMAChannelIndex    = DMAChannel1;              // only DMA channel1 can support P2M mode 
    DMA_uSPI.Init.SrcPeri       = MID_DMA_URT0_READ;        // Source symbol is URT0 RDAT
    DMA_uSPI.Init.DesPeri       = MID_DMA_MEMORY_WRITE;     // Destination symbol is memory
    DMA_uSPI.Init.BSize         = DMA_BSIZE_1BYTE;          // Burst size is 1 byte
    DMA_uSPI.Init.MEMMode       = MID_DMA_MEMORY_NORMAL;    // Normal memory access mode    
    DMA_uSPI.Init.LoopMode      = DMA_LOOP_DISABLE;         // DISABLE Loop mode     
    DMA_uSPI.Parent = &uSPI0;

    // Initial sSPI DMA init struct 
    DMA_sSPI.Instance           = DMA;
    DMA_sSPI.DMAChannelIndex    = DMAChannel0;              // only DMA channel0 can support M2P mode  
    DMA_sSPI.Init.SrcPeri       = MID_DMA_MEMORY_READ;      // Source symbol is memory
    DMA_sSPI.Init.DesPeri       = MID_DMA_SPI0_WRITE;       // Destination symbol is SPI TDAT
    DMA_sSPI.Init.BSize         = DMA_BSIZE_1BYTE;          // Burst size is 1 byte
    DMA_sSPI.Init.MEMMode       = MID_DMA_MEMORY_NORMAL;    // Normal memory access mode    
    DMA_sSPI.Init.LoopMode      = DMA_LOOP_DISABLE;         // DISABLE Loop mode     
    DMA_sSPI.Parent = &sSPI0;

    // Run 3 times
    for(CNT=3; CNT>0; CNT--)
    {
        for(CLR_CNT=0; CLR_CNT<48; CLR_CNT++)                       // Clear TST_RXData_Table
            TST_RXData_Table[CLR_CNT] = 0;
        
        MID_DMA_Init(&DMA_uSPI_1);                                  // uSPI TX DMA init
        MID_DMA_Init(&DMA_uSPI);                                    // uSPI RX DMA init
        MID_DMA_Init(&DMA_sSPI);                                    // sSPI TX DMA init
        
        NVIC_EnableIRQ(DMA_IRQn);                                   // Enable DMA interrupt
        
        uSPI0.mDMATX = &DMA_uSPI_1;                                 // Set uSPI0 DMATX
        uSPI0.mDMARX = &DMA_uSPI;                                   // Set uSPI0 DMARX
        sSPI0.mDMATX = &DMA_sSPI;                                   // Set sSPI0 DMATX
        
        MID_SPI_Transmit_DMA(&sSPI0, &TST_TXData_Table[0] , 48);    // Start sSPI DMA TX
        
        PB0 = CLR;                                                  // NSS pin active
        MID_SPI_TransmitReceive_DMA(&uSPI0, &TST_TXData_Table[0], &TST_RXData_Table[0] , 48);
                                                                    // Start sSPI DMA TX and RX
        while(uSPI0.State != MID_SPI_STATE_READY);                  // Waiting uSPI transfer complete
        PB0 = SET;                                                  // NSS pin idle
    }


    // ------------------------------------------------------------------------
    // 4.SPI Master (URT) TX DMA to SPI Slave (SPI) RX DMA.
    // ------------------------------------------------------------------------
    // Initial sSPI DMA init struct 
    DMA_uSPI.Instance           = DMA;
    DMA_uSPI.DMAChannelIndex    = DMAChannel1;              // only DMA channel1 can support M2P mode  
    DMA_uSPI.Init.SrcPeri       = MID_DMA_MEMORY_READ;      // Source symbol is memory
    DMA_uSPI.Init.DesPeri       = MID_DMA_URT0_WRITE;       // Destination symbol is URT0 TDAT
    DMA_uSPI.Init.BSize         = DMA_BSIZE_1BYTE;          // Burst size is 1 byte
    DMA_uSPI.Init.MEMMode       = MID_DMA_MEMORY_NORMAL;    // Normal memory access mode    
    DMA_uSPI.Init.LoopMode      = DMA_LOOP_DISABLE;         // DISABLE Loop mode     
    DMA_uSPI.Parent = &uSPI0;

    // Initial uSPI DMA init struct 
    DMA_sSPI.Instance           = DMA;
    DMA_sSPI.DMAChannelIndex    = DMAChannel0;              // only DMA channel0 can support P2M mode  
    DMA_sSPI.Init.SrcPeri       = MID_DMA_SPI0_READ;        // Source symbol is SPI0 RDAT
    DMA_sSPI.Init.DesPeri       = MID_DMA_MEMORY_WRITE;     // Destination symbol is memory
    DMA_sSPI.Init.BSize         = DMA_BSIZE_1BYTE;          // Burst size is 1 byte
    DMA_sSPI.Init.MEMMode       = MID_DMA_MEMORY_NORMAL;    // Normal memory access mode    
    DMA_sSPI.Init.LoopMode      = DMA_LOOP_DISABLE;         // DISABLE Loop mode     
    DMA_sSPI.Parent = &sSPI0;
    
    // Run 3 times
    for(CNT=3; CNT>0; CNT--)
    {
        for(CLR_CNT=0; CLR_CNT<48; CLR_CNT++)                       // Clear TST_RXData_Table
            TST_RXData_Table[CLR_CNT] = 0;
        
        MID_DMA_Init(&DMA_uSPI);                                    // uSPI TX DMA init
        MID_DMA_Init(&DMA_sSPI);                                    // sSPI RX DMA init
        
        NVIC_EnableIRQ(DMA_IRQn);                                   // Enable DMA interrupt
        
        uSPI0.mDMATX = &DMA_uSPI;                                   // Set uSPI0 DMATX
        sSPI0.mDMARX = &DMA_sSPI;                                   // Set sSPI0 DMARX
        
        MID_SPI_Receive_DMA(&sSPI0, &TST_RXData_Table[0] , 48);     // Start sSPI DMA RX
        
        PB0 = CLR;                                                  // NSS pin active
        MID_SPI_Transmit_DMA(&uSPI0, &TST_TXData_Table[0] , 48);    // Start sSPI DMA TX
        while(uSPI0.State != MID_SPI_STATE_READY);                  // Waiting uSPI transfer complete
        PB0 = SET;                                                  // NSS pin idle
    }
}



