/**
  ******************************************************************************
 *
 * @file        Sample_SPI_Master_StandardSPI_DMA_P2M.c
 *
 * @brief       Standard-SPI master mode DMA SPI-to-SRAM sample code.
 *
 * @par         Project
 *              MG32
 * @version     V2.16
 * @date        2022/11/25
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2021 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
/* Includes ------------------------------------------------------------------*/
#include "MG32_GPIO_DRV.h"
#include "MG32_SPI_DRV.h"
#include "MG32_DMA_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define MYBINARYIMAGE1_LENGTH   11
#define SPI_NSS                 PB0

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static uint8_t MstDest[MYBINARYIMAGE1_LENGTH];

/* Private function prototypes -----------------------------------------------*/
void Sample_SPI_Master_StandardSPI_DMA_P2M (SPI_Struct* SPIx);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief       Sample SPI Master Standard SPI use DMA SPI to memory code
 * @details     1. Enable CSC to SPI clock
 *      \n      2. Default Initial SPI
 *      \n      3. Configure clock divider
 *      \n      4. Configure SPI data line, mode and data size...
 *      \n      5. Config SPI0 IO
 *      \n      6. Enable SPI
 *      \n      7. Configure memory to SPI
 *      \n      8. Trigger DMA
 *      \n      9. Disable SPI
 * @param[in]   SPIx
 * 	@arg\b			SPI0
 * @return      None
 * @note
 * @par         Example
 * @code
    Sample_SPI_Master_StandardSPI_DMA_P2M(SPI0);
 * @endcode
 *******************************************************************************
 **/
void Sample_SPI_Master_StandardSPI_DMA_P2M (SPI_Struct* SPIx)
{
    uint8_t             CNT;                    // Loop counter
    uint8_t             DLY_CNT;                //
    DMA_BaseInitTypeDef DMATestPattern;
    PIN_InitTypeDef     PINX_InitStruct;          

    
    
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12Mz
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    
    /*=== 1. Enable CSC to SPI clock ===*/
    //[A] When Use Wizard
    //Configure Peripheral On Mode Clock->SPI0 = Enable and Select SPI0_PR Source = CK_APB
    //Configure Peripheral On Mode Clock->Port B = Enable
    //[B] When Use Driver
//    UnProtectModuleReg(CSCprotect);                             // Unprotect CSC module
//    CSC_PeriphOnModeClock_Config(CSC_ON_SPI0, ENABLE);          // Enable SPI0 module clock
//    CSC_PeriphOnModeClock_Config(CSC_ON_PortB, ENABLE);		  // Enable PortB clock
//    CSC_PeriphProcessClockSource_Config(CSC_SPI0_CKS, CK_APB);  // CK_SPIx_PR = CK_APB = 12MHz
//    ProtectModuleReg(CSCprotect);                               // protect CSC module

    /*=== 2. Default Initial SPI ===*/
    SPI_DeInit(SPIx);  

    /*=== 3. Configure clock divider ===*/
    SPI_PreDivider_Select(SPIx, SPI_PDIV_1);
    SPI_Prescaler_Select(SPIx, SPI_PSC_3);
    SPI_Divider_Select(SPIx, SPI_DIV_4);
    SPI_Clock_Select(SPIx, SPI_CK_SPIx_PR);
    
    /*=== 4. Configure SPI data line, mode and data size... ===*/
    SPI_DataLine_Select(SPIx, SPI_Standard);
    SPI_ModeAndNss_Select(SPIx, SPI_Master);         
    SPI_FirstBit_Select(SPIx, SPI_LSB);
    // SPI transfer data size 8-bits
    SPI_DataSize_Select(SPIx, SPI_8bits);
    // Data bidirectional input
    SPI_BidirectionalOutput_Cmd(SPIx, DISABLE);
    // Enable SPI precatch function
    SPI_RXDMAPreCatch_Cmd(SPI0, ENABLE);
    
    /*=== 5. Config SPI0 IO ===*/
    SPI_NSS = 1; 
    PINX_InitStruct.PINX_Mode				= PINX_Mode_PushPull_O;         // Pin select pusu pull mode
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Disable;     // Enable pull up resistor
    PINX_InitStruct.PINX_Speed              = PINX_Speed_High;              // Output high speed mode enable
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;         // Pin output driver full strength.
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;    // Pin input deglitch filter clock divider bypass
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;         // Pin input data not inverse
    PINX_InitStruct.PINX_Alternate_Function = 2;                            // AFS = 2
    GPIO_PinMode_Config(PINB(2),&PINX_InitStruct);                          // CLK
    GPIO_PinMode_Config(PINB(3),&PINX_InitStruct);                          // MOSI
    
    PINX_InitStruct.PINX_Alternate_Function = 0;                            // AFS = 0
    GPIO_PinMode_Config(PINB(0),&PINX_InitStruct);                          // NSS
    
    PINX_InitStruct.PINX_Mode				= PINX_Mode_Digital_I;          // Setting digital  mode
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;      // Disable pull up resistor
    PINX_InitStruct.PINX_Alternate_Function = 2;                            // AFS = 2
    GPIO_PinMode_Config(PINB(1),&PINX_InitStruct);                          // MISO
    
    /*=== 6. Enable SPI ===*/
    SPI_Cmd(SPIx, ENABLE);

    /*=== 7. Configure memory to SPI ===*/
    for (CNT=0; CNT<MYBINARYIMAGE1_LENGTH; CNT++)
        MstDest[CNT] = 0x00;
        
    // Enable DMA
    DMA_Cmd(ENABLE);
    // Enable Channel0
    DMA_Channel_Cmd(DMAChannel0, ENABLE);
    // initial parameter
    DMA_BaseInitStructure_Init(&DMATestPattern);
    // modify parameter
    {   
        // external trigger pin and mode config
        DMATestPattern.ExtTrgPin = DMA_ExtTRG0;
        DMATestPattern.ExtTrgMode = DMA_DisableExtTrg;
        // DMA channel select
        DMATestPattern.DMAChx = DMAChannel0;
        // channel x loop mode config
        DMATestPattern.DMALoopCmd = DISABLE;
        // channel x source/destination auto increase address
        DMATestPattern.SrcSINCSel = DISABLE;        
        DMATestPattern.DestDINCSel = ENABLE;
        // DMA Burst size config
        DMATestPattern.BurstDataSize = DMA_BurstSize_1Byte;
        // DMA transfer data count initial number (? Bytes ???)
        DMATestPattern.DMATransferNUM = MYBINARYIMAGE1_LENGTH;
        // DMA source peripheral config
        DMATestPattern.SrcSymSel = DMA_SPI0_RX;
        // DMA destination peripheral config
        DMATestPattern.DestSymSel = DMA_MEM_Write;
        // source config
        DMATestPattern.DMADestinationAddr = (uint8_t *)&MstDest;
    }
    // M2P(CH0) simple parameter
    DMA_Base_Init(&DMATestPattern);
    
    /*=== 8. Trigger DMA ===*/
    for(CNT=3; CNT>0; CNT--)
    {   // Enable chip
        SPI_NSS = 0;
        // start request
        DMA_StartRequest(DMAChannel0);
        // Enable SPI RX DMA
        SPI_RXDMA_Cmd(SPIx, ENABLE);
        // Waiting DMA transmit to SPI complete
        while(DMA_GetSingleFlagStatus(DMA, DMA_FLAG_CH0_TCF) == DRV_UnHappened);
        // Clear CH0_TCF and CH0_THF
        DMA_ClearFlag(DMA, DMA_FLAG_CH0_TCF | DMA_FLAG_CH0_THF);
        // Wait SPI transfer  complete.
        while(SPI_GetSingleFlagStatus(SPI0,SPI_TCF) == DRV_UnHappened);
        // Disable chip
        SPI_NSS = 1;                                                       
        // Clear SPI all flag
        SPI_ClearFlag(SPIx, SPI_ALLF);
        
//        // Delay 
//        for(DLY_CNT=100; DLY_CNT>0; DLY_CNT--)
//            __NOP();
        
        // prepare destination area
        for (DLY_CNT=0; DLY_CNT<MYBINARYIMAGE1_LENGTH; DLY_CNT++)
            MstDest[DLY_CNT] = 0x00;
    }
    
    /*=== 9. Disable SPI ===*/
    SPI_Cmd(SPIx, DISABLE);
}



