

/**
 ******************************************************************************
 *
 * @file        Sample_LCD_GDC05720_Display.c
 * @brief       Screen GDC05720 display C file
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2024/09/18
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_ChipInit.h"
#include "Sample_LCD_GDC05720_Display.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
    
/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define LCD_PIN_PIN0        0
#define LCD_PIN_PIN1        1
#define LCD_PIN_PIN2        2
#define LCD_PIN_PIN3        3
#define LCD_PIN_PIN4        4
#define LCD_PIN_PIN5        5
#define LCD_PIN_PIN6        6
#define LCD_PIN_PIN7        7
#define LCD_PIN_PIN8        8
#define LCD_PIN_PIN9        9
#define LCD_PIN_PIN10       10
#define LCD_PIN_PIN11       11
#define LCD_PIN_PIN12       12
#define LCD_PIN_PIN13       13
#define LCD_PIN_PIN14       14
#define LCD_PIN_PIN15       15
#define LCD_PIN_PIN16       16
#define LCD_PIN_PIN17       17
#define LCD_PIN_PIN18       18
#define LCD_PIN_PIN19       19
#define LCD_PIN_PIN20       20
#define LCD_PIN_PIN21       21
#define LCD_PIN_PIN22       22
#define LCD_PIN_PIN23       23
#define LCD_PIN_PIN24       24
#define LCD_PIN_PIN25       25
#define LCD_PIN_PIN26       26
#define LCD_PIN_PIN27       27
#define LCD_PIN_PIN28       28
#define LCD_PIN_PIN29       29
#define LCD_PIN_PIN30       30
#define LCD_PIN_PIN31       31
#define LCD_PIN_PIN32       32
#define LCD_PIN_PIN33       33
#define LCD_PIN_PIN34       34
#define LCD_PIN_PIN35       35
#define LCD_PIN_PIN36       36
#define LCD_PIN_PIN37       37
#define LCD_PIN_PIN38       38
#define LCD_PIN_PIN39       39
#define LCD_PIN_PIN40       40
#define LCD_PIN_PIN41       41
#define LCD_PIN_PIN42       42
#define LCD_PIN_PIN43       43


#define SMP_LCD_COM_TOTAL   8


#define SEG16_A1            0x00000010
#define SEG16_A             0x00001000
#define SEG16_J             0x00100000
#define SEG16_B             0x10000000
#define SEG16_F             0x00000020
#define SEG16_H             0x00002000
#define SEG16_I             0x00200000
#define SEG16_G1            0x20000000
#define SEG16_G             0x00000040
#define SEG16_M             0x00004000
#define SEG16_L             0x00400000
#define SEG16_K             0x40000000
#define SEG16_E             0x00000080
#define SEG16_D1            0x00008000
#define SEG16_D             0x00800000
#define SEG16_C             0x80000000

/* Private typedef -----------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

static const uint8_t SMP_LCD_COMTable[SMP_LCD_COM_TOTAL] = 
{
    LCD_PIN_PIN24,     /*!< COM0 */
    LCD_PIN_PIN25,     /*!< COM1 */
    LCD_PIN_PIN26,     /*!< COM2 */
    LCD_PIN_PIN27,     /*!< COM3 */
    LCD_PIN_PIN28,     /*!< COM4 */
    LCD_PIN_PIN29,     /*!< COM5 */
    LCD_PIN_PIN30,     /*!< COM6 */
    LCD_PIN_PIN31      /*!< COM7 */
};


COM_TypeDef LCD_PIN1_Tmp;
COM_TypeDef LCD_PIN2_Tmp;
COM_TypeDef LCD_PIN3_Tmp;
COM_TypeDef LCD_PIN4_Tmp;
COM_TypeDef LCD_PIN5_Tmp;
COM_TypeDef LCD_PIN6_Tmp;
COM_TypeDef LCD_PIN7_Tmp;
COM_TypeDef LCD_PIN8_Tmp;
COM_TypeDef LCD_PIN9_Tmp;
COM_TypeDef LCD_PIN10_Tmp;
COM_TypeDef LCD_PIN11_Tmp;
COM_TypeDef LCD_PIN12_Tmp;
COM_TypeDef LCD_PIN13_Tmp;
COM_TypeDef LCD_PIN14_Tmp;
COM_TypeDef LCD_PIN15_Tmp;
COM_TypeDef LCD_PIN16_Tmp;
COM_TypeDef LCD_PIN17_Tmp;
COM_TypeDef LCD_PIN18_Tmp;
COM_TypeDef LCD_PIN19_Tmp;
COM_TypeDef LCD_PIN20_Tmp;
COM_TypeDef LCD_PIN21_Tmp;
COM_TypeDef LCD_PIN22_Tmp;
COM_TypeDef LCD_PIN23_Tmp;
COM_TypeDef LCD_PIN24_Tmp;


static uint32_t LCD_MD0_Tmp;
static uint32_t LCD_MD1_Tmp;
static uint32_t LCD_MD2_Tmp;
static uint32_t LCD_MD3_Tmp;
static uint32_t LCD_MD4_Tmp;
static uint32_t LCD_MD5_Tmp;
//static uint32_t LCD_MD6_Tmp;
//static uint32_t LCD_MD7_Tmp;
//static uint32_t LCD_MD8_Tmp;
//static uint32_t LCD_MD9_Tmp;
//static uint32_t LCD_MD10_Tmp;

static ctype  LCD_MDn_COM[11];

       uint8_t  SMP_LCD_Busy;


/* Private function ----------------------------------------------------------*/
void LCD_IRQHandler(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	   LCD data update function.  
 * @details    Use UDCF interrupt to update. 
 * @return     
 * @exception  No
 * @note       No          
 *******************************************************************************
 */
void Sample_LCD_main(void)
{
    /*Initial LCD*/
    Sample_LCD_Init();
    
    /*Display Segment-7 display*/
    Sample_LCD_ClearDataBuffer();
    Sample_LCD_7Segment( SMP_LCD_SEG7_1,0);
    Sample_LCD_7Segment( SMP_LCD_SEG7_2,1);
    Sample_LCD_7Segment( SMP_LCD_SEG7_3,2);
    Sample_LCD_7Segment( SMP_LCD_SEG7_4,3);
    Sample_LCD_7Segment( SMP_LCD_SEG7_5,4);
    Sample_LCD_7Segment( SMP_LCD_SEG7_6,5);
    Sample_LCD_7Segment( SMP_LCD_SEG7_7,6);
    Sample_LCD_7Segment( SMP_LCD_SEG7_8,7);
    Sample_LCD_7Segment( SMP_LCD_SEG7_9,8);
    Sample_LCD_7Segment(SMP_LCD_SEG7_10,9);
    Sample_LCD_TriggerUpdate();
}

/**
 *******************************************************************************
 * @brief	   LCD data update function.  
 * @details    Use UDCF interrupt to update. 
 * @return     
 * @exception  No
 * @note       No          
 *******************************************************************************
 */
void LCD_IRQHandler(void)
{
    LCD->MD0.W  = LCD_MD0_Tmp;
    LCD->MD1.W  = LCD_MD1_Tmp;
    LCD->MD2.W  = LCD_MD2_Tmp;
    LCD->MD3.W  = LCD_MD3_Tmp;
    LCD->MD4.W  = LCD_MD4_Tmp;
    LCD->MD5.W  = LCD_MD5_Tmp;
    LCD->MD5.W  = LCD_MD5_Tmp;
    /*No data need update*/
//    LCD->MD6.W  = LCD_MD6_Tmp;
//    LCD->MD7.W  = LCD_MD7_Tmp;
//    LCD->MD8.W  = LCD_MD8_Tmp;
//    LCD->MD9.W  = LCD_MD9_Tmp;
//    LCD->MD10.W = LCD_MD10_Tmp;
    
    LCD_ClearFlag(LCD,LCD_FLAG_UDCF);
    LCD_IT_Config(LCD, LCD_IT_UDC,DISABLE);
    
    SMP_LCD_Busy = 0;
    
}

/**
 *******************************************************************************
 * @brief	   LCD initial  
 * @details        
 * @return     
 * @exception  No
 * @note       No          
 *******************************************************************************
 */
void Sample_LCD_Init(void)
{
    PIN_InitTypeDef  LCD_Pin;
    uint8_t          LCD_InitTmp;
    
    /*
        Pin configure
    */
    LCD_Pin.PINX_Pin                = (PX_Pin_1  | PX_Pin_2  | PX_Pin_3  | PX_Pin_4  | PX_Pin_5  | PX_Pin_6  | \
                                       PX_Pin_7  | PX_Pin_8  | PX_Pin_9  | PX_Pin_10 | PX_Pin_11 | PX_Pin_12 | \
                                       PX_Pin_13 | PX_Pin_14 | PX_Pin_15);
    LCD_Pin.PINX_Mode               = PINX_Mode_Analog_IO;
    LCD_Pin.PINX_PUResistant        = PINX_PUResistant_Disable;
    LCD_Pin.PINX_Speed              = PINX_Speed_High;
    LCD_Pin.PINX_Inverse            = PINX_Inverse_Disable;
    LCD_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    LCD_Pin.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    LCD_Pin.PINX_Alternate_Function = 12;                       
    GPIO_PortMode_Config(IOMA,&LCD_Pin);
    
    LCD_Pin.PINX_Pin                = (PX_Pin_0  | PX_Pin_1  | PX_Pin_2  | PX_Pin_3  | PX_Pin_4  | PX_Pin_5  | \
                                       PX_Pin_6  | PX_Pin_7  | PX_Pin_10 | PX_Pin_11 | PX_Pin_12 | PX_Pin_13 | \
                                       PX_Pin_14 | PX_Pin_15);
    GPIO_PortMode_Config(IOMB,&LCD_Pin);
    
    LCD_Pin.PINX_Pin                = (PX_Pin_1  | PX_Pin_2  | PX_Pin_3  | PX_Pin_7  | PX_Pin_8  | PX_Pin_9  | \
                                       PX_Pin_10 | PX_Pin_11 | PX_Pin_12);
    GPIO_PortMode_Config(IOMC,&LCD_Pin);
    
    LCD_Pin.PINX_Pin                = (PX_Pin_4  | PX_Pin_5 );
    GPIO_PortMode_Config(IOMD,&LCD_Pin);
    
    //LCD C1, C2, VT, V1, V2, V3
    LCD_Pin.PINX_Alternate_Function = 0;  
    LCD_Pin.PINX_Pin                = (PX_Pin_0  | PX_Pin_1  | PX_Pin_2  | PX_Pin_3  | PX_Pin_7  | PX_Pin_9);
    GPIO_PortMode_Config(IOMD,&LCD_Pin);
    
    /*
        LCD initial
    */
    /*LCD mode to default*/
    LCD->CR0.W = 0;                                  
    
    /*LCD_Pn as common or segment line select and COM order related configure*/
    for( LCD_InitTmp = 0; LCD_InitTmp < 10; LCD_InitTmp++)
    {
        LCD_MDn_COM[LCD_InitTmp].W = 0U;
    }
    LCD->MUX0.W = 0;
    LCD->MUX1.W = 0;
    for( LCD_InitTmp = 0; LCD_InitTmp < SMP_LCD_COM_TOTAL; LCD_InitTmp++)
    {
        if( SMP_LCD_COMTable[LCD_InitTmp] > 31)
        {
            LCD->MUX1.W |= (uint32_t)(1 << (SMP_LCD_COMTable[LCD_InitTmp]-32)); 
        }
        else
        {
            LCD->MUX0.W |= (uint32_t)(1 << SMP_LCD_COMTable[LCD_InitTmp]);
        }
        LCD_MDn_COM[((SMP_LCD_COMTable[LCD_InitTmp])>>2)].B[(SMP_LCD_COMTable[LCD_InitTmp]&0x03)] = (uint8_t)(1<<LCD_InitTmp);
    }
                                                                                               
    LCD->MD0.W   = LCD_MDn_COM[0].W;                        
    LCD->MD1.W   = LCD_MDn_COM[1].W;                        
    LCD->MD2.W   = LCD_MDn_COM[2].W;                        
    LCD->MD3.W   = LCD_MDn_COM[3].W;                        
    LCD->MD4.W   = LCD_MDn_COM[4].W;                        
    LCD->MD5.W   = LCD_MDn_COM[5].W;                        
    LCD->MD6.W   = LCD_MDn_COM[6].W;  
    LCD->MD7.W   = LCD_MDn_COM[7].W;
    LCD->MD8.W   = LCD_MDn_COM[8].W;
    LCD->MD9.W   = LCD_MDn_COM[9].W;
    LCD->MD10.W  = LCD_MDn_COM[10].W;

    /*Clock related configure*/                        
    LCD_SetClockSource( LCD, ClcokSource_CK_LS);            /*!< LCD input clock source select*/
    LCD_SetClockPreDivider( LCD, PreDivider_8);             /*!< LCD input clock divider.*/
    LCD_SetClockDivider( LCD, Divider_1);                   /*!< LCD interanl clock input divider.*/
    LCD_SetChargePumpDivider( LCD, CPClock_Div1);           /*!< LCD charge pump clock selection.*/
    LCD_SetClockPrescaler( LCD, 31);                        /*!< LCD internal clock prescaler*/
    LCD_SetBlinkDivider( LCD, BDivider_4);                  /*!< LCD blinking clock divider*/
                                                            
    /*LCD mode configure*/                                  
    LCD->CR0.MBIT.DT_PW = 1;                                /*!< LCD output dead time period width is 1 unit*/
    LCD_DutyCycle(LCD, DC18Duty);                           /*!< LCD duty cycle select 1/8.*/
    LCD_BiasVoltege(LCD, BIAS_B14);                         /*!< LCD bias voltage is 1/4.*/
    LCD_WaveformInverse(LCD, LCD_CR0_CS_INV_normal_w);      /*!< LCD waveform inverse disable.*/
    LCD_DeadTimeMode(LCD, DTFrame);                         /*!< LCD output dead time mode is Frame
                                                                ( insert dead time between two Frame cycle)*/
    LCD_FrameMode(LCD,FrameTypeB);                          /*!< LCD frame control mode is Type-B frame control mode.*/                                                        
    LCD_VTBiasPowerSource(LCD,PowerSource_CP);              /*!< LCD bias power source is internal chare pump.*/
    LCD->CR0.MBIT.EN = 1;                                   /*!< LCD module enable.*/
                                                            
    LCD->CR1.MBIT.DRV_MDS = 1;                         


    LCD_SetChargePumpCount(LCD, 0);                         /*!< LCD charge pump charging counter value is 0*/
    LCD_SetBlinkCount(LCD, 0);                              /*!< LCD blinking maximum frame counter value is 0*/
    LCD_DisplayCTL(LCD,Segment_ON);                         /*!< LCD segments all lines are turning on.*/
    LCD_BlinkMode_Config(LCD,BlinkDisable);                 /*!< LCD blinking mode is disable.*/
                                                            
                                                            
    LCD->CR1.MBIT.DRV_PW = 7;                               /*!< LCD drive high strength pulse width select.*/
    LCD_ChargePumpConstrast_Config(LCD,CPContrast_Level12); /*!< LCD charge pump constrast level is 3.56 Volt.*/
    LCD_VxPinControl(LCD,(V3Pin_Pin1 | 
                          V2Pin_Pin1 | 
                          V1Pin_Pin1 |
                          VTPin_Pin1));                     /*!< LCD LCD_V3/LCD_V2/LCD_V1/LCD_VT connectoin to pin.*/
    
    LCD_ChargePumpSync_Cmd(LCD,DISABLE);                    /*!< LCD charge pump counter synchronizatoin disable.*/
    LCD->CR1.MBIT.CP_EN = 0;                                /*!< LCD built-in charge pump disable.*/
     
    /*BLK initial*/ 
    LCD_SetBlinkDivider(LCD,BDivider_4);
    LCD_SetBlinkCount(LCD, 9);
    
    
    /**/        
    LCD_ClearFlag(LCD,LCD_FLAG_CPRF);
    LCD->CR1.W = LCD->CR1.W | LCD_CR1_CP_EN_enable_w; __ISB();
    LCD->CR1.W = LCD->CR1.W | LCD_CR1_DRV_MDS_high_w; __ISB();
    while(LCD_GetAllFlagStatus(LCD)&LCD_FLAG_CPRF);
    
    LCD_DriveStrengthMode(LCD,DSNormal);
    
    /*Interrupt configure*/
    LCD_ClearFlag(LCD , LCD_FLAG_UDCF);
    LCD_IT_Config( LCD, LCD_IT_UDC,DISABLE);
    LCD_ITEA_Cmd( LCD, ENABLE);
    NVIC_EnableIRQ(LCD_IRQn);
    
    /*Parameter initial*/
    SMP_LCD_Busy = 0;
}

/**
 *******************************************************************************
 * @brief	   Trigger LCD data update. 
 * @details     
 * @return     
 * @exception  No
 * @note       No            
 *******************************************************************************
 */
uint8_t Sample_LCD_TriggerUpdate(void)
{
    if( SMP_LCD_Busy == 1)
    {
        return(SMP_LCD_FAILURE);
    }
    SMP_LCD_Busy = 1;
    
    /*
       Prepare update data 
    */
    LCD_MD0_Tmp  = (((uint32_t)(LCD_PIN17_Tmp.B << 0)  | (uint32_t)(LCD_PIN18_Tmp.B << 8)  | (uint32_t)(LCD_PIN19_Tmp.B << 16)  | (uint32_t)(LCD_PIN20_Tmp.B << 24)) | LCD_MDn_COM[0].W);
    LCD_MD1_Tmp  = (((uint32_t)(LCD_PIN21_Tmp.B << 0)  | (uint32_t)(LCD_PIN22_Tmp.B << 8)  | (uint32_t)(LCD_PIN23_Tmp.B << 16)  | (uint32_t)(LCD_PIN24_Tmp.B << 24)) | LCD_MDn_COM[1].W);
    LCD_MD2_Tmp  = (((uint32_t)(LCD_PIN1_Tmp.B  << 0)  | (uint32_t)(LCD_PIN2_Tmp.B  << 8)  | (uint32_t)(LCD_PIN3_Tmp.B << 16) | (uint32_t)(LCD_PIN4_Tmp.B << 24))    | LCD_MDn_COM[2].W);
    LCD_MD3_Tmp  = (((uint32_t)(LCD_PIN5_Tmp.B  << 0)  | (uint32_t)(LCD_PIN6_Tmp.B  << 8) | (uint32_t)(LCD_PIN7_Tmp.B  << 16) | (uint32_t)(LCD_PIN8_Tmp.B  << 24))   | LCD_MDn_COM[3].W); 
    LCD_MD4_Tmp  = (((uint32_t)(LCD_PIN9_Tmp.B << 0)  | (uint32_t)(LCD_PIN10_Tmp.B << 8) | (uint32_t)(LCD_PIN11_Tmp.B << 16) | (uint32_t)(LCD_PIN12_Tmp.B << 24))    | LCD_MDn_COM[4].W);
    LCD_MD5_Tmp  = (((uint32_t)(LCD_PIN13_Tmp.B << 0)  | (uint32_t)(LCD_PIN14_Tmp.B << 8) | (uint32_t)(LCD_PIN15_Tmp.B << 16) | (uint32_t)(LCD_PIN16_Tmp.B << 24))   | LCD_MDn_COM[5].W);
    /*No data need update*/
//    LCD_MD6_Tmp  = 0 | LCD_MDn_COM[6].W ;
//    LCD_MD7_Tmp  = 0 | LCD_MDn_COM[7].W ;
//    LCD_MD8_Tmp  = 0 | LCD_MDn_COM[8].W ;
//    LCD_MD9_Tmp  = 0 | LCD_MDn_COM[9].W ;
//    LCD_MD10_Tmp = 0 | LCD_MDn_COM[10].W;
    
    /*
       Trigger Update LCD 
    */
    LCD_ClearFlag(LCD,LCD_FLAG_UDCF);
    LCD_IT_Config( LCD, LCD_IT_UDC,ENABLE);
    
    
    return(SMP_LCD_SUCCESS);
}

/**
 *******************************************************************************
 * @brief	   Display percentage in this damo LCD glass.  
 * @details     
 * @param[in]  Percentage_number:  0 ~ 150 ( 0% ~ 150%)
 * @return     
 * @exception  No
 * @note       No     
 *******************************************************************************
 */
static uint8_t LCD_Percetage_Table0[16] = {0x00,0x10,0x30,0x70,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0};
static uint8_t LCD_Percetage_Table1[16] = {0x00,0x00,0x00,0x00,0x00,0x80,0xC0,0xE0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0};
static uint8_t LCD_Percetage_Table2[16] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x30,0x70,0x70,0x70,0x70,0x70};
static uint8_t LCD_Percetage_Table3[16] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x30,0x70,0xF0};
    
static uint32_t LCD_wPercentage_QuotientTmp;

void Sample_LCD_Percentage(uint8_t Percentage_number)
{
    /*outer frame*/
    LCD_PIN19_Tmp.MBIT.COM7 = 1;
    
    if( Percentage_number < 151)
    {
        LCD_wPercentage_QuotientTmp = Percentage_number / 10;
    }
        
    LCD_PIN17_Tmp.B |= LCD_Percetage_Table0[LCD_wPercentage_QuotientTmp];
    LCD_PIN18_Tmp.B |= LCD_Percetage_Table1[LCD_wPercentage_QuotientTmp];
    LCD_PIN19_Tmp.B |= LCD_Percetage_Table2[LCD_wPercentage_QuotientTmp];
    LCD_PIN20_Tmp.B |= LCD_Percetage_Table3[LCD_wPercentage_QuotientTmp];   
}



/**
 *******************************************************************************
 * @brief	   Display battery status in this damo LCD glass.  
 * @details     
 * @param[in]  BatterLevel: 
 *  @arg\b     BSP_LCD_BATTERY_LEVEL0:
 *  @arg\b     BSP_LCD_BATTERY_LEVEL1:
 *  @arg\b     BSP_LCD_BATTERY_LEVEL2:
 *  @arg\b     BSP_LCD_BATTERY_LEVEL3:
 * @return     
 * @exception  No
 * @note       No          
 *******************************************************************************
 */
void Sample_LCD_Battery(uint8_t BatterLevel)
{
    /*outer frame*/
    LCD_PIN18_Tmp.MBIT.COM3 = 1;
    
    /*battery level*/
    switch(BatterLevel)
    {
        case SMP_LCD_BATTERY_LEVEL1:
                                    LCD_PIN18_Tmp.MBIT.COM0 = 1;
                                    break;
        case SMP_LCD_BATTERY_LEVEL2:
                                    LCD_PIN18_Tmp.MBIT.COM0 = 1;
                                    LCD_PIN18_Tmp.MBIT.COM1 = 1;
                                    break;
        case SMP_LCD_BATTERY_LEVEL3:
                                    LCD_PIN18_Tmp.MBIT.COM0 = 1;
                                    LCD_PIN18_Tmp.MBIT.COM1 = 1;
                                    LCD_PIN18_Tmp.MBIT.COM2 = 1;
                                    break;
        case 0:        
        default:
                                    break; 
    }
}

/**
 *******************************************************************************
 * @brief	   Display signal strength status in this damo LCD glass.       
 * @details     
 * @param[in]  SignalStrength_Level: 
 *  @arg\b     SMP_LCD_SINGSTRENGTH_LEVLE0: Signal Strength is level 0. (S7)
 *  @arg\b     SMP_LCD_SINGSTRENGTH_LEVLE1: Signal Strength is level 1. (S7 + S8)
 *  @arg\b     SMP_LCD_SINGSTRENGTH_LEVLE2: Signal Strength is level 2. (S7 + S8 + S9)
 * @return     
 * @exception  No
 * @note       No            
 *******************************************************************************
 */
void Sample_LCD_SignalStrength(uint8_t SignalStrength_Level)
{
    /*Symbol*/
    LCD_PIN17_Tmp.MBIT.COM1 = 1;
    
    /*Signal Strength level*/
    switch(SignalStrength_Level)
    {
        case SMP_LCD_SINGSTRENGTH_LEVLE1:
                                         LCD_PIN17_Tmp.MBIT.COM0 = 1;
                                         break;
        case SMP_LCD_SINGSTRENGTH_LEVLE2:
                                         LCD_PIN17_Tmp.MBIT.COM0 = 1;
                                         LCD_PIN23_Tmp.MBIT.COM0 = 1;
                                         break;
        case 0:        
        default:
                break; 
    }
}


/**
 *******************************************************************************
 * @brief	   Clear LCD software data buffer.  
 * @details      
 * @return     
 * @exception  No
 * @note       No           
 *******************************************************************************
 */
void Sample_LCD_ClearDataBuffer(void)
{
    LCD_PIN1_Tmp.B = 0x00UL;
    LCD_PIN2_Tmp.B = 0x00UL;
    LCD_PIN3_Tmp.B = 0x00UL;
    LCD_PIN4_Tmp.B = 0x00UL;
    LCD_PIN5_Tmp.B = 0x00UL;
    LCD_PIN6_Tmp.B = 0x00UL;
    LCD_PIN7_Tmp.B = 0x00UL;
    LCD_PIN8_Tmp.B = 0x00UL;
    LCD_PIN9_Tmp.B = 0x00UL;
    LCD_PIN10_Tmp.B = 0x00UL;
    LCD_PIN11_Tmp.B = 0x00UL;
    LCD_PIN12_Tmp.B = 0x00UL;
    LCD_PIN13_Tmp.B = 0x00UL;
    LCD_PIN14_Tmp.B = 0x00UL;
    LCD_PIN15_Tmp.B = 0x00UL;
    LCD_PIN16_Tmp.B = 0x00UL;
    LCD_PIN17_Tmp.B = 0x00UL;
    LCD_PIN18_Tmp.B = 0x00UL;
    LCD_PIN19_Tmp.B = 0x00UL;
    LCD_PIN20_Tmp.B = 0x00UL;
    LCD_PIN21_Tmp.B = 0x00UL;
    LCD_PIN22_Tmp.B = 0x00UL;
    LCD_PIN23_Tmp.B = 0x00UL;
    LCD_PIN24_Tmp.B = 0x00UL;  
}


/**
 *******************************************************************************
 * @brief	   Display seven-segment in this damo LCD glass.   
 * @details     
 * @param[in]  SEG7_Number: 
 *  @arg\b     SMP_LCD_SEG7_1:  7 Segment display serial number 1.
 *  @arg\b     SMP_LCD_SEG7_2:  7 Segment display serial number 2. 
 *  @arg\b     SMP_LCD_SEG7_3:  7 Segment display serial number 3.
 *  @arg\b     SMP_LCD_SEG7_4:  7 Segment display serial number 4.
 *  @arg\b     SMP_LCD_SEG7_5:  7 Segment display serial number 5.
 *  @arg\b     SMP_LCD_SEG7_6:  7 Segment display serial number 6.
 *  @arg\b     SMP_LCD_SEG7_7:  7 Segment display serial number 7.
 *  @arg\b     SMP_LCD_SEG7_8:  7 Segment display serial number 8.
 *  @arg\b     SMP_LCD_SEG7_9:  7 Segment display serial number 9.
 *  @arg\b     SMP_LCD_SEG7_10: 7 Segment display serial number 10.
 * @param[in]  diplay_number : 0 ~ 9
 * @return     
 * @exception  No
 * @note       No       
 *******************************************************************************
 */
static const uint8_t LCD_SEG7_AtoD_Table0[10] = { 0x0F, 0x06, 0x0B, 0x0F, 0x06, 0x0D, 0x0D, 0x07, 0x0F, 0x07};
static const uint8_t LCD_SEG7_EtoF_Table0[10] = { 0x05, 0x00, 0x06, 0x02, 0x03, 0x03, 0x07, 0x00, 0x07, 0x03};

static const uint8_t LCD_SEG7_AtoD_Table1[10] = { 0x0F, 0x06, 0x0B, 0x0F, 0x06, 0x0D, 0x0D, 0x07, 0x0F, 0x07};
static const uint8_t LCD_SEG7_EtoF_Table1[10] = { 0x0A, 0x00, 0x0C, 0x04, 0x06, 0x06, 0x0E, 0x00, 0x0E, 0x06};
    

void Sample_LCD_7Segment(uint8_t SEG7_Number, uint8_t diplay_number)  
{
    uint8_t      LCD_SEG7_Table_Index;
    COM_TypeDef* LCD_SEG7_AtoD_PIN;
    COM_TypeDef* LCD_SEG7_EtoF_PIN;
          
    
    /*Decide to data table*/
    switch(SEG7_Number)
    {
        case SMP_LCD_SEG7_1:
        case SMP_LCD_SEG7_4:
        case SMP_LCD_SEG7_5:
        case SMP_LCD_SEG7_6:
                            LCD_SEG7_Table_Index = 1;
                            break;
        default:
                            LCD_SEG7_Table_Index = 0;
                            break;
    }
    /*Decide to output pin*/
    switch(SEG7_Number)
    {
        case SMP_LCD_SEG7_10:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN8_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN7_Tmp;
                             break;    
        case SMP_LCD_SEG7_9:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN6_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN5_Tmp;
                             break;  
        case SMP_LCD_SEG7_8:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN4_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN3_Tmp;
                             break;  
        case SMP_LCD_SEG7_7:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN2_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN1_Tmp;
                             break;  
        case SMP_LCD_SEG7_6:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN20_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN19_Tmp;
                             break;  
        case SMP_LCD_SEG7_5:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN24_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN23_Tmp;
                             break;  
        case SMP_LCD_SEG7_4:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN22_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN21_Tmp;
                             break;  
        case SMP_LCD_SEG7_3:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN14_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN13_Tmp;
                             break;  
        case SMP_LCD_SEG7_2:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN12_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN11_Tmp;
                             break;  
        case SMP_LCD_SEG7_1:
        default:
                             LCD_SEG7_AtoD_PIN = &LCD_PIN10_Tmp;
                             LCD_SEG7_EtoF_PIN = &LCD_PIN9_Tmp;
                             break;  
    }
    
    if( LCD_SEG7_Table_Index == 0)
    {
        LCD_SEG7_AtoD_PIN->B |= LCD_SEG7_AtoD_Table0[diplay_number];
        LCD_SEG7_EtoF_PIN->B |= LCD_SEG7_EtoF_Table0[diplay_number];
    }
    else
    {
        LCD_SEG7_AtoD_PIN->B |= LCD_SEG7_AtoD_Table1[diplay_number];
        LCD_SEG7_EtoF_PIN->B |= LCD_SEG7_EtoF_Table1[diplay_number];
    }  
}

/**
 *******************************************************************************
 * @brief	   Display seven-segment6 ~ 10 pointer in this damo LCD glass.  
 * @details     
 * @param[in]  SEG7_6to10_point: 
 *  @arg\b     SMP_LCD_SEG7_6to10_S14:
 *  @arg\b     SMP_LCD_SEG7_6to10_S15:
 *  @arg\b     SMP_LCD_SEG7_6to10_S16:
 *  @arg\b     SMP_LCD_SEG7_6to10_S17:
 * @return     
 * @exception  No
 * @note       No             
 *******************************************************************************
 */
void Sample_LCD_7Segment_6_10_Point(uint8_t SEG7_6to10_point) 
{
    switch(SEG7_6to10_point)
    {
        case SMP_LCD_SEG7_6to10_S14:
                                    LCD_PIN1_Tmp.MBIT.COM3 = 1;
                                    break;
        case SMP_LCD_SEG7_6to10_S15:
                                    LCD_PIN3_Tmp.MBIT.COM3 = 1;
                                    break;
        case SMP_LCD_SEG7_6to10_S16:
                                    LCD_PIN5_Tmp.MBIT.COM3 = 1;
                                    break;
        case SMP_LCD_SEG7_6to10_S17:
                                    LCD_PIN7_Tmp.MBIT.COM3 = 1;
                                    break;
        default:
                                    break;
    }
}

/**
 *******************************************************************************
 * @brief	   Display seven-segment 6 ~ 10 negative symbol in this damo LCD glass.   
 * @details     
 * @param[in]  SEG7_6to10_SigFigs: ( 1 ~ 4)
 *  @arg\b     1: negative symbol in 9 of 7segment.
 *  @arg\b     2: negative symbol in 8 of 7segment.
 *  @arg\b     3: negative symbol in 7 of 7segment.
 *  @arg\b     4: negative symbol in 6 of 7segment.
 * @return     
 * @exception  No
 * @note       No         
 *******************************************************************************
 */
void Sample_LCD_7Segment_6_10_NegativeSymbol(uint8_t SEG7_6to10_SigFigs) 
{
    switch(SEG7_6to10_SigFigs)
    {
        case 1:
                LCD_PIN5_Tmp.MBIT.COM1 = 1;
                break;
        case 2:
                LCD_PIN3_Tmp.MBIT.COM1 = 1;
                break;
        case 3:
                LCD_PIN1_Tmp.MBIT.COM1 = 1;
                break;
        case 4:
                LCD_PIN19_Tmp.MBIT.COM2 = 1;
                break;
        default:
                break;
    }
}

/**
 *******************************************************************************
 * @brief	    16 segment display A ~ Z
 * @details  
 * @param[in] SEG16_Number:
 *  @arg\b    SMP_LCD_SEG16_11:
 *  @arg\b    SMP_LCD_SEG16_12:
 *  @arg\b    SMP_LCD_SEG16_13:
 *  @arg\b    SMP_LCD_SEG16_14:
 *  @arg\b    SMP_LCD_SEG16_15:
 * @param[in] SEG16_display:  A ~ Z
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
static const uint32_t LCD_SEG16_0to9_Table[10] = 
{        
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_E  | SEG16_F),                         /*!< '0'*/
    (SEG16_B  | SEG16_C),                                                                                           /*!< '1'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_G1 | SEG16_G  | SEG16_E  | SEG16_D1 | SEG16_D),                         /*!< '2'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_G1 | SEG16_G),                         /*!< '3'*/
    (SEG16_F  | SEG16_G  | SEG16_G1 | SEG16_B  | SEG16_C),                                                          /*!< '4'*/
    (SEG16_A1 | SEG16_A  | SEG16_F  | SEG16_G  | SEG16_G1 | SEG16_C  | SEG16_D  | SEG16_D1),                        /*!< '5'*/
    (SEG16_A  | SEG16_A1 | SEG16_F  | SEG16_E  | SEG16_G  | SEG16_G1 | SEG16_C  | SEG16_D  | SEG16_D1),             /*!< '6'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C),                                                                     /*!< '7'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_E  | SEG16_F  | SEG16_G  | SEG16_G1),  /*!< '8'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_F  | SEG16_G  | SEG16_G1)              /*!< '9'*/
};


static const uint32_t LCD_SEG16_AtoZ_Table[26] = 
{
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_E  | SEG16_F  | SEG16_G  | SEG16_G1),                         /*!< 'A'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_I  | SEG16_L  | SEG16_G1),              /*!< 'B'*/
    (SEG16_A1 | SEG16_A  | SEG16_F  | SEG16_E  | SEG16_D1 | SEG16_D),                                                /*!< 'C'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_I  | SEG16_L),                          /*!< 'D'*/
    (SEG16_A1 | SEG16_A  | SEG16_D1 | SEG16_D  | SEG16_E  | SEG16_F  | SEG16_G  | SEG16_G1),                         /*!< 'E'*/
    (SEG16_A1 | SEG16_A  | SEG16_F  | SEG16_E  | SEG16_G  | SEG16_G1),                                               /*!< 'F'*/
    (SEG16_A1 | SEG16_A  | SEG16_F  | SEG16_E  | SEG16_D1 | SEG16_D  | SEG16_C  | SEG16_G1),                         /*!< 'G'*/
    (SEG16_B  | SEG16_C  | SEG16_F  | SEG16_E  | SEG16_G  | SEG16_G1),                                               /*!< 'H'*/
    (SEG16_A1 | SEG16_A  | SEG16_I  | SEG16_L  | SEG16_D1 | SEG16_D),                                                /*!< 'I'*/
    (SEG16_A1 | SEG16_A  | SEG16_I  | SEG16_L  | SEG16_D1 | SEG16_E),                                                /*!< 'J'*/
    (SEG16_F  | SEG16_E  | SEG16_G  | SEG16_J  | SEG16_K),                                                           /*!< 'K'*/
    (SEG16_F  | SEG16_E  | SEG16_D1 | SEG16_D),                                                                      /*!< 'L'*/
    (SEG16_H  | SEG16_J  | SEG16_B  | SEG16_C  | SEG16_E  | SEG16_F  | SEG16_L),                                     /*!< 'M'*/
    (SEG16_B  | SEG16_C  | SEG16_H  | SEG16_K  | SEG16_F  | SEG16_E),                                                /*!< 'N'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_F  | SEG16_E),                          /*!< 'O'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_G1 | SEG16_G  | SEG16_F  | SEG16_E),                                     /*!< 'P'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_F  | SEG16_E  | SEG16_K),               /*!< 'Q'*/
    (SEG16_A1 | SEG16_A  | SEG16_B  | SEG16_G1 | SEG16_G  | SEG16_F  | SEG16_E  | SEG16_K),                          /*!< 'R'*/
    (SEG16_A1 | SEG16_A  | SEG16_F  | SEG16_G  | SEG16_G1 | SEG16_C  | SEG16_D  | SEG16_D1),                         /*!< 'S'*/
    (SEG16_A1 | SEG16_A  | SEG16_I  | SEG16_L),                                                                      /*!< 'T'*/
    (SEG16_B  | SEG16_C  | SEG16_D  | SEG16_D1 | SEG16_F  | SEG16_E),                                                /*!< 'U'*/
    (SEG16_J  | SEG16_M  | SEG16_F  | SEG16_E),                                                                      /*!< 'V'*/
    (SEG16_F  | SEG16_E  | SEG16_M  | SEG16_K  | SEG16_B  | SEG16_C),                                                /*!< 'W'*/
    (SEG16_H  | SEG16_J  | SEG16_M  | SEG16_K),                                                                      /*!< 'X'*/
    (SEG16_F  | SEG16_G  | SEG16_G1 | SEG16_B  | SEG16_L),                                                           /*!< 'Y'*/
    (SEG16_A1 | SEG16_A  | SEG16_J  | SEG16_M  | SEG16_D1 | SEG16_D),                                                /*!< 'Z'*/
};

void Sample_LCD_16SegmentChar( uint8_t SEG16_Number, uint8_t SEG16_display)
{
    COM_TypeDef* LCD_SEG16_A1toE_PIN;
    COM_TypeDef* LCD_SEG16_AtoD1_PIN;
    COM_TypeDef* LCD_SEG16_JtoD_PIN;
    COM_TypeDef* LCD_SEG16_BtoC_PIN;
    ctype        LCD_SEG16_Tmp;
    
    switch(SEG16_Number)
    {
        case SMP_LCD_SEG16_11:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN1_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN2_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN3_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN4_Tmp;
                              break;
        case SMP_LCD_SEG16_12:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN5_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN6_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN7_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN8_Tmp;  
                              break;
        case SMP_LCD_SEG16_13:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN9_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN10_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN11_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN12_Tmp;
                              break;
        case SMP_LCD_SEG16_14: 
                              LCD_SEG16_A1toE_PIN = &LCD_PIN13_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN14_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN15_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN16_Tmp;
                              break;
        case SMP_LCD_SEG16_15:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN21_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN22_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN23_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN24_Tmp; 
                              break;
        default:
                              return;
    }
    
    /* 0 ~ 9*/
    if( SEG16_display > '/' && SEG16_display < ':')
    {
        LCD_SEG16_Tmp.W = LCD_SEG16_0to9_Table[SEG16_display - '0'];
    }
    else if( SEG16_display > '@' && SEG16_display < '[')
    {
        LCD_SEG16_Tmp.W = LCD_SEG16_AtoZ_Table[SEG16_display - 'A'];
    }
    else
    {
        return;
    }
    
    LCD_SEG16_A1toE_PIN->B |= LCD_SEG16_Tmp.B[0];
    LCD_SEG16_AtoD1_PIN->B |= LCD_SEG16_Tmp.B[1];
    LCD_SEG16_JtoD_PIN->B  |= LCD_SEG16_Tmp.B[2]; 
    LCD_SEG16_BtoC_PIN->B  |= LCD_SEG16_Tmp.B[3];      
}

/**
 *******************************************************************************
 * @brief	    16 segment display 0 ~ 9
 * @details  
 * @param[in] SEG16_Number:
 *  @arg\b    SMP_LCD_SEG16_11:
 *  @arg\b    SMP_LCD_SEG16_12:
 *  @arg\b    SMP_LCD_SEG16_13:
 *  @arg\b    SMP_LCD_SEG16_14:
 *  @arg\b    SMP_LCD_SEG16_15:
 * @param[in] SEG16_display:  0 ~ 9
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_16Segment0to9( uint8_t SEG16_Number, uint8_t SEG16_display)
{
    COM_TypeDef* LCD_SEG16_A1toE_PIN;
    COM_TypeDef* LCD_SEG16_AtoD1_PIN;
    COM_TypeDef* LCD_SEG16_JtoD_PIN;
    COM_TypeDef* LCD_SEG16_BtoC_PIN;
    ctype        LCD_SEG16_Tmp;
    
    switch(SEG16_Number)
    {
        case SMP_LCD_SEG16_11:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN1_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN2_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN3_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN4_Tmp;
                              break;
        case SMP_LCD_SEG16_12:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN5_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN6_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN7_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN8_Tmp;  
                              break;
        case SMP_LCD_SEG16_13:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN9_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN10_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN11_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN12_Tmp;
                              break;
        case SMP_LCD_SEG16_14: 
                              LCD_SEG16_A1toE_PIN = &LCD_PIN13_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN14_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN15_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN16_Tmp;
                              break;
        case SMP_LCD_SEG16_15:
                              LCD_SEG16_A1toE_PIN = &LCD_PIN21_Tmp;
                              LCD_SEG16_AtoD1_PIN = &LCD_PIN22_Tmp;
                              LCD_SEG16_JtoD_PIN  = &LCD_PIN23_Tmp;
                              LCD_SEG16_BtoC_PIN  = &LCD_PIN24_Tmp; 
                              break;
        default:
                              return;
    }
    
    /* 0 ~ 9*/
    if( SEG16_display > 9)
    {
        LCD_SEG16_Tmp.W = LCD_SEG16_0to9_Table[SEG16_display];
    }
    
    LCD_SEG16_A1toE_PIN->B |= LCD_SEG16_Tmp.B[0];
    LCD_SEG16_AtoD1_PIN->B |= LCD_SEG16_Tmp.B[1];
    LCD_SEG16_JtoD_PIN->B  |= LCD_SEG16_Tmp.B[2]; 
    LCD_SEG16_BtoC_PIN->B  |= LCD_SEG16_Tmp.B[3];     
}


/**
 *******************************************************************************
 * @brief	   Display LCD T symbol 
 * @details  
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_T(void)
{
    LCD_PIN17_Tmp.MBIT.COM3 = 1;
}    


/**
 *******************************************************************************
 * @brief	    Display LCD P symbol 
 * @details
 * @return       
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_P(void)
{
    LCD_PIN17_Tmp.MBIT.COM2 = 1;
}  

/**
 *******************************************************************************
 * @brief	   Display LCD Degree symbol  
 * @details   
 * @return    
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_Degree(void)
{
    LCD_PIN16_Tmp.MBIT.COM1 = 1;
}
    
/**
 *******************************************************************************
 * @brief	  Display LCD percentage symbol   
 * @details    
 * @return   
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_Percentage(void)
{
    LCD_PIN15_Tmp.MBIT.COM0 = 1;
}

/**
 *******************************************************************************
 * @brief	  Display LCD V symbol   
 * @details  
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_V(void)
{
    LCD_PIN16_Tmp.MBIT.COM2 = 1;
}

/**
 *******************************************************************************
 * @brief	  Display LCD A symbol   
 * @details 
 * @return      
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_A(void)
{
    LCD_PIN16_Tmp.MBIT.COM3 = 1;
}
/**
 *******************************************************************************
 * @brief	  Display LCD m symbol   
 * @details
 * @return       
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_m(void)
{
    LCD_PIN15_Tmp.MBIT.COM3 = 1;
}
/**
 *******************************************************************************
 * @brief	  Display LCD KPa symbol   
 * @details 
 * @return      
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_KPa(void)
{
    LCD_PIN15_Tmp.MBIT.COM1 = 1;
}
/**
 *******************************************************************************
 * @brief	  Display LCD MPa symbol   
 * @details     
 * @return  
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_Symbol_MPa(void)
{
    LCD_PIN15_Tmp.MBIT.COM2 = 1;
}

/**
 *******************************************************************************
 * @brief	  Control LCD display ON / OFF  
 * @details  
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_LCD_OFF_Cmd( FunctionalState LCD_OFF_Cmd)
{
    if( LCD_OFF_Cmd == ENABLE)
    {
        LCD_DisplayCTL(LCD,Segment_OFF);
    }
    else
    {
        LCD_DisplayCTL(LCD,Segment_ON);
    }
}
