
/**
  ******************************************************************************
 *
 * @file        Sample_CMP_WakeupFromSLEEPMode.c
 *
 * @brief       Initial COMP recipe for SLEEP mode. 
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2025/06/20
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
 
/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Sample_CMP_WakeupFromSLEEPMode(void);
void COMP_IRQHandler(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/
 
 
/**
 *******************************************************************************
 * @brief       Initial COMP recipe for SLEEP mode. 
 * @details     1. COMP & GPIOA Clock initial
 *      \n      2. GPIO initial
 *      \n      3. ACx(0~3) initial
 *      \n      4. Interrupt and NVIC configuration
 *      \n      5. COMP sleep mode continuous run configure.
 * @return	    None	
 * @note        
 * @par         Example
 * @code
    Sample_CMP_WakeupFromSLEEPMode();
    // To do ...
    {
    
    }
    SLEEP_WFI();
 * @endcode
 *******************************************************************************
*/
void Sample_CMP_WakeupFromSLEEPMode(void)
{
    PIN_InitTypeDef PINX_InitStruct;

    // ===CSC Initialize Set Description====
    // MG32_CSC_Init.h(Configuration Wizard)
    // Select CK_HS source = CK_IHRCO
    // Select IHRCO = 12Mz
    // Select CK_MAIN Source = CK_HS
    // Select CK_LS Source = CK_ILRCO
    // Configure Select APB Prescaler = CK_MAIN/1
    // Configure Peripheral On Mode Clock->RTC = Enable
    
    // (GPIO Config)
    // AC0 needs PA8/PA9 or PB0/PB1 pin.
    // AC1 needs PA10/PA11 or PB0/PB1 pin.
    // MG32_GPIO_Init.H
    // Select GPIO Initial -> GPIOA configuration -> PA8/PA9/PA10/PA11 configuration
    // Select GPIO Initial -> GPIOB configuration -> PB0/PB1 configuration
    // Select Pin mode = AIO
    // Select Pin alternate function = GPA8/GPA9/GPA10/GPA11
    // Other selection keep default value

    // ------------------------------------------------------------------------
    // 1. COMP & GPIOA Clock initial (User can config in MG32_CSC_Init.h.)
    // ------------------------------------------------------------------------
    UnProtectModuleReg(CSCprotect);
    CSC_PeriphONModeClock_Config(CSC_ON_PortA, ENABLE);         // Enable PortA clock    
    
    CSC_PeriphONModeClock_Config(CSC_ON_CMP, ENABLE);           // Enable CMP clock
    CSC_PeriphProcessClockSource_Config(CSC_CMP_CKS, CK_APB);   // CMP select AHB
    
    CSC_PeriphSLEEPModeClock_Config(CSC_SLP_CMP, ENABLE);       // *** Enable CMP sleep clock ***
    ProtectModuleReg(CSCprotect);
    
    // ------------------------------------------------------------------------
    // 2. GPIO initial (User can config in MG32_GPIO_LQFP80_Init.h.)
    // ------------------------------------------------------------------------
    // for AC0
    PINX_InitStruct.PINX_Pin                = (PX_Pin_8 | PX_Pin_9);
    PINX_InitStruct.PINX_Mode				= PINX_Mode_Analog_IO;
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Disable;
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;
    PINX_InitStruct.PINX_Alternate_Function = 0;
    
    GPIO_PortMode_Config(IOMA, &PINX_InitStruct);
    
    // for AC1
    PINX_InitStruct.PINX_Pin                = (PX_Pin_10 | PX_Pin_11);
    GPIO_PortMode_Config(IOMA, &PINX_InitStruct);
        
    // ------------------------------------------------------------------------
    // 3. AC0 initial (User can config in MG32_CMP_Init.h.)
    // ------------------------------------------------------------------------
    // CMPAC0 config
    CMP_Cmd(CMPAC0, ENABLE);                                // Enable CMPAC0
    #if !defined(CMP_CR0_AC0_PMUX_op0_p0_w)
        CMP_PositivePin_Select(CMPAC0, CMP_ACz_I0);         // PA8    
        CMP_NegativePin_Select(CMPAC0, CMP_ACz_I1);         // PA9
    #else                                                   // for MG32_5TH
        CMP_PositivePin_Select(CMPAC0, CMP_ACP_I0);         // PA8    
        CMP_NegativePin_Select(CMPAC0, CMP_ACN_I0);         // PA9
    #endif
    CMP_Power_Select(CMPAC0, CMP_Normal);                   // Normal Power level
    
    // ------------------------------------------------------------------------
    // AC1 initial 
    // ------------------------------------------------------------------------
    // CMPAC1 config
    CMP_Cmd(CMPAC1, ENABLE);                                // Enable CMPAC1
    #if !defined(CMP_CR1_AC1_PMUX_op0_p0_w)                 
        CMP_PositivePin_Select(CMPAC1, CMP_ACz_I0);         // PA10    
        CMP_NegativePin_Select(CMPAC1, CMP_ACz_I1);         // PA11
    #else                                                   // for MG32_5TH
        CMP_PositivePin_Select(CMPAC1, CMP_ACP_I0);         // PA10    
        CMP_NegativePin_Select(CMPAC1, CMP_ACN_I1);         // PA11
    #endif
    CMP_Power_Select(CMPAC1, CMP_Normal);                   // Normal Power level
        
    // ------------------------------------------------------------------------
    // 4. Interrupt and NVIC configuration
    // ------------------------------------------------------------------------
    CMP_ClearFlag (CMP, AC0_RisingEdge_Flag | AC0_FallingEdge_Flag);
    CMP_ClearFlag (CMP, AC1_RisingEdge_Flag | AC1_FallingEdge_Flag);
    
    CMP_IT_Config(CMP, (AC0_RisingEdge_IE | AC0_FallingEdge_IE), ENABLE);
    CMP_IT_Config(CMP, (AC1_RisingEdge_IE | AC1_FallingEdge_IE), ENABLE);
        
    //
    CMP_ITEA_Cmd(CMP, ENABLE);
    
    NVIC_EnableIRQ(COMP_IRQn);
    
    // ------------------------------------------------------------------------
    // 5. COMP sleep mode continuous run configure.
    // ------------------------------------------------------------------------
    UnProtectModuleReg(PWprotect);
    PW_PeriphSLEEPModeContinuous_Config(PW_SLPPO_CMP0, ENABLE); // Enable CMP in SLEEP
    PW_PeriphSLEEPModeContinuous_Config(PW_SLPPO_CMP1, ENABLE); // Enable CMP in SLEEP
    ProtectModuleReg(PWprotect);

    // ------------------------------------------------------------------------
    // SLEEP test
    // ------------------------------------------------------------------------
//    SLEEP_WFI();
    
    
}


/**
 *******************************************************************************
 * @brief       COMP: An interrupt, or IRQ, is an exception signalled by a 
 *              peripheral, or generated by a software request. 
 * @details     COMP handles port CMP0~3 interrupt.
 * @return      No
 *******************************************************************************
 */
#ifndef MG32_COMP_IRQ_
void COMP_IRQHandler(void)
{
    // ------------------------------------------------------------------------
    // COMP - AC0 ISR
    // ------------------------------------------------------------------------
    // AC0  rising / falling edge flag
    if (CMP_GetSingleFlagStatus(CMP, AC0_RisingEdge_Flag) == DRV_Happened)
    {
        // Rising edge interrupt flag 
        // To do...
        
        CMP_ClearFlag (CMP, AC0_RisingEdge_Flag);
    }
    if (CMP_GetSingleFlagStatus(CMP, AC0_FallingEdge_Flag) == DRV_Happened)
    {
        // Falling edge interrupt flag 
        // To do...
        
        CMP_ClearFlag (CMP, AC0_FallingEdge_Flag);
    }

    // ------------------------------------------------------------------------
    // COMP - AC1 ISR
    // ------------------------------------------------------------------------
    // AC1  rising / falling edge flag
    if (CMP_GetSingleFlagStatus(CMP, AC1_RisingEdge_Flag) == DRV_Happened)
    {
        // Rising edge interrupt flag 
        // To do...
        
        CMP_ClearFlag (CMP, AC1_RisingEdge_Flag);
    }
    if (CMP_GetSingleFlagStatus(CMP, AC1_FallingEdge_Flag) == DRV_Happened)
    {
        // Falling edge interrupt flag 
        // To do...
        
        CMP_ClearFlag (CMP, AC1_FallingEdge_Flag);
    }
}
#endif


