
/**
 ******************************************************************************
 *
 * @file        Sample_ADC_WindowDetet.C
 * @brief       Demo ADC window detect function with interrupt support. 
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2025/11/10
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */



/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
//*** <<< Use Configuration Wizard in Context Menu >>> ***

//  <h> ADC Window detect channel recipe 

//      <o0> ADC Voltage window detect lower threshold <0-4095>
#define ADC_WINDTH_WIND_LT          100

//      <o0> ADC Voltage window detect higher threshold <0-4095>
#define ADC_WINDTH_WIND_HT          3800
//      <h> Window detect interrupt 
//          <q0.10> ADC voltage window detect outside high event interrupt enable.
//              <i> When the ADC conversion data is more than window detect higher threshold.
//          <q0.9> ADC voltage window detect inside event interrupt enable.
//              <i> When the ADC conversion data is in the middle of the window threshold.
//          <q0.8> ADC voltage window detect outside low event interrupt enable.
//              <i> When the ADC conversion data is low than window detect lower threshold.
//      </h>
#define ADC_INT_WDx_IE              0x00000200

//  </h>

//*** <<< end of configuration section >>>    ***

#if ADC_WINDTH_WIND_LT >= ADC_WINDTH_WIND_HT
    #error "Erroe setting : ADC Voltage window detect lower threshold >= higher threshold value."
#endif


/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/
/**
 * @name    Function announce
 * @brief   addtogroup ADC_Exported_Functions_Group
 */ 
///@{
void Sample_ADC_WindowDetet(void);
void ADC_IRQHandler(void);
//!@}


/**
 *******************************************************************************
 * @brief	    ADC Window detect initial
 * @details     1. Config ADC base parameter
 *    \n        2. One shot conversion mode
 *    \n        3. ADC calibration
 *    \n        4. Config Window detect 
 *    \n        6. Enable ADC interrupt 
 * @return 	    None
 *******************************************************************************
 */
void Sample_ADC_WindowDetet(void) 
{
    ADC_InitTypeDef ADC_Base;
    
    // make sure :
	
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->ADC = Enable
    //Configure Peripheral On Mode Clock->Port A = Enable 

    // --------------------------------------------------------------
    // 1. Config ADC base parameter    
    // --------------------------------------------------------------
    ADC_BaseStructure_Init(&ADC_Base);
    {   // modify parameter
        ADC_Base.ADCMainClockSelect = ADC_CKADC;    
            ADC_Base.ADC_IntCK_Div = ADC_IntDIV2;   // for internal clock divider
    
        // ADC data alignment mode (Right or Left)
        ADC_Base.ADC_DataAlign = ADC_RightJustified;
        
        // ADC conversion resolution 8, 10 or 12 bit
        ADC_Base.ADC_ResolutionSel = ADC_12BitData;
        
        // ADC overwritten data or keep data
        ADC_Base.ADC_DataOverrunEvent = ADC_DataOverWritten;
        
    }
    ADC_Base_Init(ADC0, &ADC_Base);
    
    ADC_Cmd(ADC0, ENABLE);
    
    // ------------------------------------------------------------------------
    // 2. One shot conversion mode
    // ------------------------------------------------------------------------
    ADC_ConversionMode_Select(ADC0, ADCMode);
    
    // ------------------------------------------------------------------------
    // 3. ADC calibration
    // ------------------------------------------------------------------------
    ADC_StartCalibration(ADC0, ENABLE);         // ADC calibration  
            
    // ------------------------------------------------------------------------
    // 4. Config Window detect 
    // ------------------------------------------------------------------------
    // Window detect threshold value:
    //  Driver level:
    //      ADC_SetHigherThreshold(ADC0, ADC_WINDTH_WIND_HT);
    //      ADC_SetLowerThreshold(ADC0, ADC_WINDTH_WIND_LT);
    //  Register write:
    ADC0->WINDTH.W = (ADC_WINDTH_WIND_HT << ADC_WINDTH_WIND_HT_shift_w) |
                     (ADC_WINDTH_WIND_LT);
    
    // Window detect interurpt:
    //  Driver level:
    //      ADC_IT_Config(ADC0, (ADC_WDH_IE | ADC_WDH_IE | ADC_WDH_IE), DISABLE);
    //      ADC_IT_Config(ADC0, (ADC_INT_WDx_IE), ENABLE);
    //  Register write:
    ADC0->INT.W &= ~(ADC_INT_WDH_IE_mask_w  |
                     ADC_INT_WDI_IE_mask_w  |
                     ADC_INT_WDL_IE_mask_w);
    ADC0->INT.W |= (ADC_INT_WDx_IE);
    
    // Enable ADC window detect function
    ADC_WindowDetect_Cmd(ADC0, ENABLE);
    
    // ------------------------------------------------------------------------
    // 6. Enable ADC interrupt 
    // ------------------------------------------------------------------------
    ADC_ITEA_Cmd(ADC0, ENABLE);
        
    // Enable ADC NVIC
    NVIC_EnableIRQ(ADC_IRQn);
    
    // ------------------------------------------------------------------------
    // Sample code : Conversion a channel
    // if user choice WDH event, then ADC conversion data is more than higher threshold.
    // Firmware will enter the ADC_IRQHandler routine.
    ADC_ExternalChannel_Select(ADC0, ADC_ExtAIN3);
    ADC_SoftwareConversion_Cmd(ADC0, ENABLE);
    while((ADC_GetAllFlagStatus(ADC0) & ADC_E1CNVF) == 0);
    ADC_ClearFlag(ADC0, ADC_E1CNVF);
    
    return;
}

/**
 *******************************************************************************
 * @brief       ADC interrupt function. (Only for window detect)
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void ADC_IRQHandler(void)
{
    // to do ...
    
    
    // ------------------------------------------------------------------------
    // window detect (WDH event)
    // ------------------------------------------------------------------------
    if (((ADC0->STA.W & ADC_STA_WDHF_happened_w) != 0) && ((ADC0->INT.W & ADC_INT_WDH_IE_mask_w) != 0))
    {
        // ADC0 voltage window detect outside high event flag
        // To do...
    
        // Clear WDH flag
        ADC0->STA.W = ADC_STA_WDHF_happened_w;
    }

    // ------------------------------------------------------------------------
    // window detect (WDI event)
    // ------------------------------------------------------------------------
    if (((ADC0->STA.W & ADC_STA_WDIF_happened_w) != 0) && ((ADC0->INT.W & ADC_INT_WDI_IE_mask_w) != 0))
    {
        // ADC0 voltage window detect inside event flag 
        // To do...

        // Clear WDI flag
        ADC0->STA.W = ADC_STA_WDIF_happened_w;
    }

    // ------------------------------------------------------------------------
    // window detect (WDL event)
    // ------------------------------------------------------------------------
    if (((ADC0->STA.W & ADC_STA_WDLF_happened_w) != 0) && ((ADC0->INT.W & ADC_INT_WDL_IE_mask_w) != 0))
    {
        // ADC0 voltage window detect outside low event flag  
        // To do...

        // Clear WDL flag
        ADC0->STA.W = ADC_STA_WDLF_happened_w;
    }


}    

