
/**
 ******************************************************************************
 *
 * @file        BSP_EEPROM.c
 * @brief       This is EEPROM (24C16) C file.
 
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2023/02/06
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
/*==============================================================================
                                 User Notes
How To use this function:
-----------------------
   1. Use BSP_EEPROM_Init() function to initial.
   2. EEPROM page write by BSP_EEPROM_PageWrite() fumction.
   3. EEPROM read by BSP_EEPROM_Read() function.
   
Driver architecture:
--------------------
   + MG32_GPIO_DRV
   + MG32_I2C_DRV
   
Known Limitations:
------------------

Require parameter
------------------
    Require module : CSC / GPIO / TM00 / I2C0
    
    GPIO pin configuration : 
        Pin / IO mode / AFS
        ---  --------  -----
        PA8 / ODO     / I2C0_SCL
        PA10/ ODO     / I2C0_SDA

    TM00 Module :
        Mode              : Cascade mode
        Overflow time     : 1ms (in TM00 clock source is 48Mhz)

    I2C Module :
        Baud-rate         : 400KHz (in I2C clock source is 48MHz)
         
Example codes:
------------------

==============================================================================*/ 
 
/* Includes ------------------------------------------------------------------*/ 
#include "BSP_14_EEPROM.h"
 
/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

// ----------------------------------------------------------------------------
// GPIO
// ----------------------------------------------------------------------------
#define EEPROM_SCL_PIN         PX_Pin_8
#define EEPROM_SDA_PIN         PX_Pin_10
// GPIO I/O Mode                               
#define EEPROM_SCL_PIN_MODE    PINX_Mode_OpenDrain_O  
#define EEPROM_SDA_PIN_MODE    PINX_Mode_OpenDrain_O 
// Pin AFS
#define EEPROM_SCL_PIN_AFS     3
#define EEPROM_SDA_PIN_AFS     3
                               
#define EEPROM_IOM             IOMA

// ----------------------------------------------------------------------------
// Timer module (TM00)
// ----------------------------------------------------------------------------
#define EEPROM_TM              TM00
// Timer counter setting (prescaler counter & main counter setting)
#define EEPROM_TM_PRESCALER    240
#define EEPROM_TM_MAINCOUNTER  200

// ----------------------------------------------------------------------------
// I2C module (I2C0)                
// ----------------------------------------------------------------------------
#define EEPROM_I2Cn            I2C0
                               
//I2C Acknowledge              
#define I2C_ACK                1
#define I2C_NACK               0


/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void BSP_EEPROM_Delay(uint32_t EEPROM_Delay);
static uint8_t I2C_BusStart(void);
static uint8_t I2C_Master_BusStop(void);
static uint8_t I2C_SendData(uint8_t TxData);
static uint8_t I2C_ReceiveData(uint8_t *RxData, uint8_t Acknowledge);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief	    EEPROM initial.
 * @details     1. Initial I2C0 GPIO.
 *              2. Initial TM00 for delay function
 *              3. Initial I2C0.
 * @return      None
 *******************************************************************************
 */
void BSP_EEPROM_Init(void)
{
    PIN_InitTypeDef        EEPROM_Pin;
    TM_TimeBaseInitTypeDef TM_TimeBase_InitStruct;
    
    // ------------------------------------------------------------------------
    // GPIO Initial
    //      PA8 = SCL0 (OpenDrain with Pull-up)
    //      PA10 = SDA0 (OpenDrain with Pull-up)
    // ------------------------------------------------------------------------
    EEPROM_Pin.PINX_Pin                = (EEPROM_SCL_PIN | EEPROM_SDA_PIN );
    EEPROM_Pin.PINX_Mode               = EEPROM_SCL_PIN_MODE;
    EEPROM_Pin.PINX_PUResistant        = PINX_PUResistant_Enable;
    EEPROM_Pin.PINX_Speed              = PINX_Speed_Low;
    EEPROM_Pin.PINX_Inverse            = PINX_Inverse_Disable;
    EEPROM_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    EEPROM_Pin.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    EEPROM_Pin.PINX_Alternate_Function = EEPROM_SCL_PIN_AFS;   
    
    GPIO_PortMode_Config(EEPROM_IOM ,&EEPROM_Pin);
    
    // ------------------------------------------------------------------------
    // Timer Initial (for delay) - TM00
    // ------------------------------------------------------------------------
    TM_TimeBaseStruct_Init(&TM_TimeBase_InitStruct);            // Initial TimeBase structure 
    
    // modify parameter
    TM_TimeBase_InitStruct.TM_Prescaler   = EEPROM_TM_PRESCALER - 1;    // 240
    TM_TimeBase_InitStruct.TM_Period      = EEPROM_TM_MAINCOUNTER - 1;  // 200
    TM_TimeBase_InitStruct.TM_CounterMode = Cascade;            // Cascade mode    
    
    TM_TimeBase_Init(EEPROM_TM, &TM_TimeBase_InitStruct);       // Initial TM00 with TM_TimeBase_InitStruct
    
    // Clear TOF flag
    TM_ClearFlag(EEPROM_TM, TMx_TOF);
    TM_Timer_Cmd(EEPROM_TM, DISABLE);                           // Don't start TM00.
    
    // ------------------------------------------------------------------------
    // I2C Initial (I2C0)
    //  - I2C0 Output Clock Config (Master mode)
    //  - SCL Clock = CK_I2C_PR / Prescaler / Divider / (HT + LT), Must HT >=5 LT >= 4.
    // ------------------------------------------------------------------------
    I2C_Cmd(EEPROM_I2Cn, DISABLE);
    
    I2C_SetClockSource(EEPROM_I2Cn, I2C_CLK_SRC_PROC);          // CK_I2C0_PR
    I2C_SetClockPrescaler(EEPROM_I2Cn, I2C_CLK_PSC_2);         
    I2C_SetClockDivider(EEPROM_I2Cn, I2C_CLK_DIV_4);       
    I2C_SetSCLHighTime(EEPROM_I2Cn, 7);                         // High Time Must >= 5CLK, 
    I2C_SetSCLLowTime(EEPROM_I2Cn, 6);                          // Low Time Must >= 4CLK
                                                               
    I2C_SetPreDriveTime(EEPROM_I2Cn, I2C_PDRV_0T);

    // ------------------------------------------------------------------------
    // (Do not use) I2C OwnAddress Config
    // ------------------------------------------------------------------------
    I2C_SlaveAddressDetect_Cmd(EEPROM_I2Cn, (I2C_SADR_0 | I2C_SADR_1 | I2C_SADR_2), DISABLE);
    
    // ------------------------------------------------------------------------
    // (Do not use) I2C Interrupt Config 
    // ------------------------------------------------------------------------
    I2C_IT_Config(EEPROM_I2Cn, I2C_IT_EVENT, DISABLE);
    I2C_ITEA_Cmd(EEPROM_I2Cn, DISABLE);

    // ------------------------------------------------------------------------
    // I2C Enable
    // ------------------------------------------------------------------------
    I2C_Cmd(EEPROM_I2Cn, ENABLE);
    
}
/**
 *******************************************************************************
 * @brief	    EEPROM (24C16) page write
 * @param[in]   Address : Program address.
 * @param[in]   pData : Program data source.
 * @param[in]   Size : Program data size.
 * @return      None
 *******************************************************************************
 */
void BSP_EEPROM_PageWrite(uint8_t Address, uint8_t* pData, uint16_t Size)
{
    uint8_t lAddress = Address;
    uint8_t lByteCount = 0;
    uint16_t lCount = 0;

    //I2C0_BusStart();                      // Output Bus Start
    //I2C0_Transmiter(0xA0);                // Output Address 0xA0, Transmiter Slave Device Addree and Read / Write
    //I2C0_Transmiter(Address);             // Output 24C Memory Address 0x00

    do
    {
        // --------------------------------------------------------------------
        // START + SLAW (ACK) + Address (ACK) + Data0 (ACK) + Data1 (ACK) + .... + Datan (ACK) + STOP
        // --------------------------------------------------------------------
        I2C_BusStart();                     // Output Bus Start
        I2C_SendData(0xA0);                 // Output Address 0xA0, Transmiter Slave Device Addree and Read / Write
        I2C_SendData(lAddress);             // Output 24C Memory Address 0x00
        
        // --------------------------------------------------------------------
        // Only accept 16 bytes in once.
        // --------------------------------------------------------------------
        lByteCount = 0;
        do{
            I2C_SendData(*pData);           // I2C0 output data.
            
            pData ++;                       // Next data          
            lCount ++;
            lByteCount ++;
        }while((lByteCount < 16) && (lCount < Size));
        
        // --------------------------------------------------------------------
        lAddress += 16;                     // pass to next address
        
        I2C_Master_BusStop();               // Output Bus STOP
        BSP_EEPROM_Delay(5);                // Delay 5ms for EEPROM
    } while(lCount < Size);
}
/**
 *******************************************************************************
 * @brief	    EEPROM (24C16) read
 * @param[in]   Address : Read address.
 * @param[in]   pData : Data store destination.
 * @return      Size : Read data size.
 *******************************************************************************
 */
void BSP_EEPROM_Read(uint8_t Address, uint8_t *pData, uint16_t Size)
{
    uint16_t lCount = 0;

    // ------------------------------------------------------------------------
    // START + SLAW (ACK) + Address (ACK) 
    // ------------------------------------------------------------------------
    I2C_BusStart();                         // Output Bus Start
    I2C_SendData(0xA0);                     // Output Address 0xA0, Transmiter Slave Device Addree and Read / Write
    I2C_SendData(Address);                  // Output 24C Memory Address 0x00
                                             
    // ------------------------------------------------------------------------
    // R-START + SLAR (ACK) + R-Data0 (ACK) + R-Data1 (ACK) + ...R-Datan (ACK) + STOP
    // ------------------------------------------------------------------------
    I2C_BusStart();                         // Output Repeat Start
    I2C_SendData(0xA1);                     // Output Address 0xA1, Transmiter Slave Device Addree and Read / Write
    
    do
    {
        if(lCount < (Size - 1))
            I2C_ReceiveData(pData, I2C_ACK); 
        else
            I2C_ReceiveData(pData, I2C_NACK);
        pData ++;
        lCount ++;
    } while(lCount < Size);
    
    I2C_Master_BusStop();                       // Output Bus Stop
}
/**
 *******************************************************************************
 * @brief	    Delay X ms.
 * @details     The delay is polling delay by using Timer.
 * @param[in]   EEPROM_Delay : (uint is ms)
 * @return      None
 *******************************************************************************
 */
static void BSP_EEPROM_Delay(uint32_t EEPROM_Delay)
{
    uint32_t BSP_EEPROM_DelayCount = 0;
    
    // ------------------------------------------------------------------------
    // Enable TM00 (period)
    // ------------------------------------------------------------------------
    TM_Timer_Cmd(EEPROM_TM, ENABLE);
    
    do
    {
        // unit : 1ms  (Assert TOF)
        TM_ClearFlag(EEPROM_TM, TMx_TOF);       // clear TOF flag
        while(TM_GetSingleFlagStatus(EEPROM_TM, TMx_TOF) == DRV_UnHappened);
        
        // delay counter 
        BSP_EEPROM_DelayCount = BSP_EEPROM_DelayCount + 1;
    
    } while(BSP_EEPROM_DelayCount != EEPROM_Delay);
    
    // ------------------------------------------------------------------------
    // Disable TM00
    TM_Timer_Cmd(EEPROM_TM, DISABLE);
}
/**
 *******************************************************************************
 * @brief	    I2C Bus start.
 * @details     Generate a I2C Start signal.    
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
static uint8_t I2C_BusStart(void)
{
    uint8_t lEventCode;
    
    // ------------------------------------------------------------------------
    // Get I2C Event code
    lEventCode = I2C_GetEventCode(EEPROM_I2Cn);
    
    // ------------------------------------------------------------------------
    // until I2C0 Bus fine.
    if(lEventCode != 0xF8)
    {
        // polling the I2C Event flag 
        while(I2C_GetEventFlag(EEPROM_I2Cn) != 0);
        
        // Get I2C Event code again
        lEventCode = I2C_GetEventCode(EEPROM_I2Cn);
    }
    
    // ------------------------------------------------------------------------
    switch (lEventCode)
    {
    case 0x08:                          // START transmitted
    case 0x10:                          // Repeated START transmitted
        // Sends START or R-START complete.
        return EEPROM_STATUS_SUCCESS;
    case 0xF8:                          // STOP or bus is released
    default:
        Set_STOP_CLR(EEPROM_I2Cn);      // Clear Stop event.    
        Set_START_SET(EEPROM_I2Cn);     // Generate a START event in next state.
        I2C_ClearEventFlag(EEPROM_I2Cn);// I2C start trigger event.
    
        while(I2C_GetEventFlag(EEPROM_I2Cn) != 0);

        return EEPROM_STATUS_FAILURE;
    }
}
/**
 *******************************************************************************
 * @brief	    I2C master bus stop
 * @details     Generate a I2C Stop signal.    
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
static uint8_t I2C_Master_BusStop(void)
{
    uint8_t lEventCode;

    // ------------------------------------------------------------------------
    while(I2C_GetEventFlag(EEPROM_I2Cn) != 0);
    
    lEventCode = I2C_GetEventCode(EEPROM_I2Cn);
    
    // ------------------------------------------------------------------------
    switch (lEventCode)
    {
    case 0xF8:                          // STOP or bus is released
    case 0xA0:                          // STOP or Repeated START received
        return EEPROM_STATUS_SUCCESS;

    default:
        Set_STA_STO_AA_010(EEPROM_I2Cn);// Set I2C status
        I2C_ClearEventFlag(EEPROM_I2Cn);// Active
        __I2C_WaitSTOClear(EEPROM_I2Cn);// until complete
    
        return EEPROM_STATUS_SUCCESS;
    }
}
/**
 *******************************************************************************
 * @brief	    I2C send data
 * @param[in]   TxData : transmit data.
 * @return      None 
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
static uint8_t I2C_SendData(uint8_t TxData)
{
    uint8_t lEventCode;

    // ------------------------------------------------------------------------
    while(I2C_GetEventFlag(EEPROM_I2Cn) != 0);

    lEventCode = I2C_GetEventCode(EEPROM_I2Cn);

    // ------------------------------------------------------------------------
    switch (lEventCode)
    {
    case 0x08:                          // START transmitted
    case 0x10:                          // Repeated START transmitted
        Set_START_CLR(EEPROM_I2Cn);
        break;
    
    case 0x18:                          // SLA+W transmitted and ACK received
    case 0x20:                          // SLA+W transmitted and NoACK received
    case 0x28:                          // DAT transmitted and ACK received
    case 0x30:                          // DAT transmitted and NoACK received
    case 0xB8:                          // Data byte in DAT has been transmitted
        break;
    
    default:
        return EEPROM_STATUS_FAILURE;
    }
    
    // ------------------------------------------------------------------------
    Set_ASSERT_ACKNOWLEDGE_SET(EEPROM_I2Cn);    // Assert ACK for next trnasmit cycle
    I2C_SendSBUF(EEPROM_I2Cn, TxData);  // Prepare data on I2C Bus.
    I2C_ClearEventFlag(EEPROM_I2Cn);    // Active
    
    return EEPROM_STATUS_SUCCESS;
}
/**
 *******************************************************************************
 * @brief	    I2C receive data.
 * @param[in]   RxData : Receive data store pointer.
 * @param[in]   Acknowledge : 
 *   @arg\b     I2C_ACK : Acknowledge is ACK after receive data.
 *   @arg\b     I2C_NACK : Acknowledge is NACK after receive data.
 * @return      Receive data.
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
static uint8_t I2C_ReceiveData(uint8_t *RxData, uint8_t Acknowledge)
{
    uint8_t lEventCode;

    // ------------------------------------------------------------------------
    while(I2C_GetEventFlag(EEPROM_I2Cn) != 0);

    lEventCode = I2C_GetEventCode(EEPROM_I2Cn);

    // ------------------------------------------------------------------------
    switch (lEventCode)
    {
    case 0x40:                          // SLA+R transmitted and ACK received
    case 0x50:                          // DAT received and ACK received
    case 0x80:                          // Previously addressed with own SLA, DAT received and ACK returned
    case 0x90:                          // DAT received and ACK returned (General Call)
        if(Acknowledge != 0)
            Set_ASSERT_ACKNOWLEDGE_SET(EEPROM_I2Cn);    // I2C_ACK
        else
            Set_ASSERT_ACKNOWLEDGE_CLR(EEPROM_I2Cn);    // I2C_NAK

        // --------------------------------------------------------------------
        I2C_ClearEventFlag(EEPROM_I2Cn);
        while(I2C_GetEventFlag(EEPROM_I2Cn) != 0);      // complete?
        
        // Read data 
        *RxData = I2C_ReceiveSBUF(EEPROM_I2Cn);

        return EEPROM_STATUS_SUCCESS;
        
    default:
        return EEPROM_STATUS_FAILURE;
    }
}


