/**
 ******************************************************************************
 *
 * @file        MG32_IEC60730_Flash.c
 *
 * @brief       This is the C code format driver head file.
 *
 * @par         Project
 *              MG32x02z
 * @version     V2.01
 * @date        2025/07/11
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2024 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer 
 *      The Demo software is provided "AS IS"  without any warranty, either 
 *      expressed or implied, including, but not limited to, the implied warranties 
 *      of merchantability and fitness for a particular purpose.  The author will 
 *      not be liable for any special, incidental, consequential or indirect 
 *      damages due to loss of data or any other reason. 
 *      These statements agree with the world wide and local dictated laws about 
 *      authorship and violence against these laws. 
 ******************************************************************************
 ******************************************************************************
 */ 


/* Includes -----------------------------------------------------------------*/
#include "MG32_IEC60730_Common.h"
#include "MG32_IEC60730_Manager.h"

/* Private typedef ----------------------------------------------------------*/

/* Private define -----------------------------------------------------------*/

/* Private macro ------------------------------------------------------------*/

/* Private variables --------------------------------------------------------*/

//    const uint32_t CRCChecksumSize __attribute__((at(0x00000024)));
//    const uint32_t CRCChecksumValue __attribute__((at(0x00000028)));
#define APCRCChecksumSize (*(uint32_t *) 0x0000024)
#define APCRCChecksumValue (*(uint32_t *) 0x00000028)

#if IAP_Planning
  #define IAPCRCChecksumValue (*(uint32_t *)(0x1A000000 + (IAP_Size - 4)))
#else
  #define IAPCRCChecksumValue (*(uint32_t *)(0x1A000000 + ((MEM->IAPSZ.W << 2) - 4)))
#endif



/* This table corresponds to the polynomial 0x11021 */
static const uint16_t crc16_table[256] =
  {
    0x0000u, 0x1021u, 0x2042u, 0x3063u, 0x4084u, 0x50a5u, 0x60c6u, 0x70e7u, 0x8108u,
    0x9129u, 0xa14au, 0xb16bu, 0xc18cu, 0xd1adu, 0xe1ceu, 0xf1efu, 0x1231u, 0x0210u,
    0x3273u, 0x2252u, 0x52b5u, 0x4294u, 0x72f7u, 0x62d6u, 0x9339u, 0x8318u, 0xb37bu,
    0xa35au, 0xd3bdu, 0xc39cu, 0xf3ffu, 0xe3deu, 0x2462u, 0x3443u, 0x0420u, 0x1401u,
    0x64e6u, 0x74c7u, 0x44a4u, 0x5485u, 0xa56au, 0xb54bu, 0x8528u, 0x9509u, 0xe5eeu,
    0xf5cfu, 0xc5acu, 0xd58du, 0x3653u, 0x2672u, 0x1611u, 0x0630u, 0x76d7u, 0x66f6u,
    0x5695u, 0x46b4u, 0xb75bu, 0xa77au, 0x9719u, 0x8738u, 0xf7dfu, 0xe7feu, 0xd79du,
    0xc7bcu, 0x48c4u, 0x58e5u, 0x6886u, 0x78a7u, 0x0840u, 0x1861u, 0x2802u, 0x3823u,
    0xc9ccu, 0xd9edu, 0xe98eu, 0xf9afu, 0x8948u, 0x9969u, 0xa90au, 0xb92bu, 0x5af5u,
    0x4ad4u, 0x7ab7u, 0x6a96u, 0x1a71u, 0x0a50u, 0x3a33u, 0x2a12u, 0xdbfdu, 0xcbdcu,
    0xfbbfu, 0xeb9eu, 0x9b79u, 0x8b58u, 0xbb3bu, 0xab1au, 0x6ca6u, 0x7c87u, 0x4ce4u,
    0x5cc5u, 0x2c22u, 0x3c03u, 0x0c60u, 0x1c41u, 0xedaeu, 0xfd8fu, 0xcdecu, 0xddcdu,
    0xad2au, 0xbd0bu, 0x8d68u, 0x9d49u, 0x7e97u, 0x6eb6u, 0x5ed5u, 0x4ef4u, 0x3e13u,
    0x2e32u, 0x1e51u, 0x0e70u, 0xff9fu, 0xefbeu, 0xdfddu, 0xcffcu, 0xbf1bu, 0xaf3au,
    0x9f59u, 0x8f78u, 0x9188u, 0x81a9u, 0xb1cau, 0xa1ebu, 0xd10cu, 0xc12du, 0xf14eu,
    0xe16fu, 0x1080u, 0x00a1u, 0x30c2u, 0x20e3u, 0x5004u, 0x4025u, 0x7046u, 0x6067u,
    0x83b9u, 0x9398u, 0xa3fbu, 0xb3dau, 0xc33du, 0xd31cu, 0xe37fu, 0xf35eu, 0x02b1u,
    0x1290u, 0x22f3u, 0x32d2u, 0x4235u, 0x5214u, 0x6277u, 0x7256u, 0xb5eau, 0xa5cbu,
    0x95a8u, 0x8589u, 0xf56eu, 0xe54fu, 0xd52cu, 0xc50du, 0x34e2u, 0x24c3u, 0x14a0u,
    0x0481u, 0x7466u, 0x6447u, 0x5424u, 0x4405u, 0xa7dbu, 0xb7fau, 0x8799u, 0x97b8u,
    0xe75fu, 0xf77eu, 0xc71du, 0xd73cu, 0x26d3u, 0x36f2u, 0x0691u, 0x16b0u, 0x6657u,
    0x7676u, 0x4615u, 0x5634u, 0xd94cu, 0xc96du, 0xf90eu, 0xe92fu, 0x99c8u, 0x89e9u,
    0xb98au, 0xa9abu, 0x5844u, 0x4865u, 0x7806u, 0x6827u, 0x18c0u, 0x08e1u, 0x3882u,
    0x28a3u, 0xcb7du, 0xdb5cu, 0xeb3fu, 0xfb1eu, 0x8bf9u, 0x9bd8u, 0xabbbu, 0xbb9au,
    0x4a75u, 0x5a54u, 0x6a37u, 0x7a16u, 0x0af1u, 0x1ad0u, 0x2ab3u, 0x3a92u, 0xfd2eu,
    0xed0fu, 0xdd6cu, 0xcd4du, 0xbdaau, 0xad8bu, 0x9de8u, 0x8dc9u, 0x7c26u, 0x6c07u,
    0x5c64u, 0x4c45u, 0x3ca2u, 0x2c83u, 0x1ce0u, 0x0cc1u, 0xef1fu, 0xff3eu, 0xcf5du,
    0xdf7cu, 0xaf9bu, 0xbfbau, 0x8fd9u, 0x9ff8u, 0x6e17u, 0x7e36u, 0x4e55u, 0x5e74u,
    0x2e93u, 0x3eb2u, 0x0ed1u, 0x1ef0u
  };

static const uint32_t crc32_table[256] = {
    0x00000000, 0x04c11db7, 0x09823b6e, 0x0d4326d9, 0x130476dc, 0x17c56b6b, 0x1a864db2, 0x1e475005,
    0x2608edb8, 0x22c9f00f, 0x2f8ad6d6, 0x2b4bcb61, 0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd,
    0x4c11db70, 0x48d0c6c7, 0x4593e01e, 0x4152fda9, 0x5f15adac, 0x5bd4b01b, 0x569796c2, 0x52568b75,
    0x6a1936c8, 0x6ed82b7f, 0x639b0da6, 0x675a1011, 0x791d4014, 0x7ddc5da3, 0x709f7b7a, 0x745e66cd,
    0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039, 0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5,
    0xbe2b5b58, 0xbaea46ef, 0xb7a96036, 0xb3687d81, 0xad2f2d84, 0xa9ee3033, 0xa4ad16ea, 0xa06c0b5d,
    0xd4326d90, 0xd0f37027, 0xddb056fe, 0xd9714b49, 0xc7361b4c, 0xc3f706fb, 0xceb42022, 0xca753d95,
    0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1, 0xe13ef6f4, 0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d,
    0x34867077, 0x30476dc0, 0x3d044b19, 0x39c556ae, 0x278206ab, 0x23431b1c, 0x2e003dc5, 0x2ac12072,
    0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16, 0x018aeb13, 0x054bf6a4, 0x0808d07d, 0x0cc9cdca,
    0x7897ab07, 0x7c56b6b0, 0x71159069, 0x75d48dde, 0x6b93dddb, 0x6f52c06c, 0x6211e6b5, 0x66d0fb02,
    0x5e9f46bf, 0x5a5e5b08, 0x571d7dd1, 0x53dc6066, 0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
    0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e, 0xbfa1b04b, 0xbb60adfc, 0xb6238b25, 0xb2e29692,
    0x8aad2b2f, 0x8e6c3698, 0x832f1041, 0x87ee0df6, 0x99a95df3, 0x9d684044, 0x902b669d, 0x94ea7b2a,
    0xe0b41de7, 0xe4750050, 0xe9362689, 0xedf73b3e, 0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2,
    0xc6bcf05f, 0xc27dede8, 0xcf3ecb31, 0xcbffd686, 0xd5b88683, 0xd1799b34, 0xdc3abded, 0xd8fba05a,
    0x690ce0ee, 0x6dcdfd59, 0x608edb80, 0x644fc637, 0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb,
    0x4f040d56, 0x4bc510e1, 0x46863638, 0x42472b8f, 0x5c007b8a, 0x58c1663d, 0x558240e4, 0x51435d53,
    0x251d3b9e, 0x21dc2629, 0x2c9f00f0, 0x285e1d47, 0x36194d42, 0x32d850f5, 0x3f9b762c, 0x3b5a6b9b,
    0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff, 0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623,
    0xf12f560e, 0xf5ee4bb9, 0xf8ad6d60, 0xfc6c70d7, 0xe22b20d2, 0xe6ea3d65, 0xeba91bbc, 0xef68060b,
    0xd727bbb6, 0xd3e6a601, 0xdea580d8, 0xda649d6f, 0xc423cd6a, 0xc0e2d0dd, 0xcda1f604, 0xc960ebb3,
    0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7, 0xae3afba2, 0xaafbe615, 0xa7b8c0cc, 0xa379dd7b,
    0x9b3660c6, 0x9ff77d71, 0x92b45ba8, 0x9675461f, 0x8832161a, 0x8cf30bad, 0x81b02d74, 0x857130c3,
    0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640, 0x4e8ee645, 0x4a4ffbf2, 0x470cdd2b, 0x43cdc09c,
    0x7b827d21, 0x7f436096, 0x7200464f, 0x76c15bf8, 0x68860bfd, 0x6c47164a, 0x61043093, 0x65c52d24,
    0x119b4be9, 0x155a565e, 0x18197087, 0x1cd86d30, 0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
    0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088, 0x2497d08d, 0x2056cd3a, 0x2d15ebe3, 0x29d4f654,
    0xc5a92679, 0xc1683bce, 0xcc2b1d17, 0xc8ea00a0, 0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb, 0xdbee767c,
    0xe3a1cbc1, 0xe760d676, 0xea23f0af, 0xeee2ed18, 0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4,
    0x89b8fd09, 0x8d79e0be, 0x803ac667, 0x84fbdbd0, 0x9abc8bd5, 0x9e7d9662, 0x933eb0bb, 0x97ffad0c,
    0xafb010b1, 0xab710d06, 0xa6322bdf, 0xa2f33668, 0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4,
  };

/* Private function prototypes ----------------------------------------------*/

/* Private functions --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint16_t IEC60730_SWCRC16_8Bit(uint16_t InitVlaue, uint8_t *StartAddr, uint32_t Length)
{
    uint16_t SumValue = InitVlaue;
    uint8_t *lpAddr = StartAddr;
    while (Length--)
    {
        SumValue = crc16_table[(uint16_t)(SumValue >> 8u) ^ *lpAddr++] ^ (uint16_t)(SumValue << 8u);
    }
    return SumValue;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint32_t IEC60730_SWCRC32_8Bit(uint32_t InitVlaue, uint8_t *StartAddr, uint32_t Length)
{
    uint32_t SumValue = InitVlaue;
    uint8_t *lpAddr = StartAddr;
    while (Length--)
    {
        SumValue = crc32_table[((SumValue >> 24) & 0xff) ^ *lpAddr++] ^ ((SumValue << 8) & 0xFFFFFF00);
    }
    return SumValue ^ 0xFFFFFFFF;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint16_t IEC60730_CRC16_8Bit(uint16_t InitVlaue, uint8_t *StartAddr, uint32_t Length)
{
    uint8_t * lpAddr = (uint8_t*)StartAddr;
    uint32_t lCount = Length;
    uint16_t  CRC16Value;

    // width=16 poly=0x1021 init=0x0000 refin=false refout=false xorout=0x0000 residue=0x0000
    // Initial GPL CRC
    __GPL_CRC_Disable();
    __GPL_Inverse_Disable();
    __GPL_ByteOrderChange_Disable();
    __GPL_BeforeBitOrderChange_Select(GPL_BEFORE_BIT_ORDER_DISABLE);
    __GPL_CRC_Mode_Select(GPL_CRC_MODE_CCITT16);
    __GPL_CRC_Data_Size_Select(GPL_CRC_DATA_SIZE_8BITS);
    __GPL_CRC_SetInitialValue((uint32_t)InitVlaue);
    __GPL_AfterBitOrderChange_Select(GPL_AFTER_BIT_ORDER_DISABLE);
    __GPL_CRC_Enable();

    do{
        GPL->DIN.B[0] = *lpAddr;
        __ISB();
        lpAddr ++;
    }while(-- lCount != 0);

    CRC16Value = GPL->DOUT.H[0];
    __GPL_CRC_Disable();

    return CRC16Value;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint16_t IEC60730_CRC16_32Bit(uint16_t InitVlaue, uint32_t *StartAddr, uint32_t Length)
{
    uint32_t * lpAddr = (uint32_t *)StartAddr;
    uint32_t lCount = (Length >> 2);
    uint16_t  CRC16Value;

    // width=16 poly=0x1021 init=0x0000 refin=false refout=false xorout=0x0000
    // Initial GPL CRC
    __GPL_CRC_Disable();
    __GPL_Inverse_Disable();
    __GPL_ByteOrderChange_Enable();
    __GPL_BeforeBitOrderChange_Select(GPL_BEFORE_BIT_ORDER_DISABLE);
    __GPL_CRC_Mode_Select(GPL_CRC_MODE_CCITT16);
    __GPL_CRC_Data_Size_Select(GPL_CRC_DATA_SIZE_32BITS);
    __GPL_CRC_SetInitialValue((uint32_t)InitVlaue);
    __GPL_AfterBitOrderChange_Select(GPL_AFTER_BIT_ORDER_DISABLE);
    __GPL_CRC_Enable();

    do{
        GPL->DIN.W = *lpAddr;
        __ISB();
        lpAddr ++;
    }while(-- lCount != 0);

    CRC16Value = GPL->DOUT.H[0];
    __GPL_CRC_Disable();

    return CRC16Value;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint32_t IEC60730_CRC32_8Bit(uint32_t InitVlaue, uint8_t *StartAddr, uint32_t Length)
{
    uint8_t * lpAddr = (uint8_t*)StartAddr;
    uint32_t lCount = Length;
    uint32_t  CRC32Value;

    // width=32 poly=0x04c11db7 init=0x00000000 refin=false refout=false
    // Initial GPL CRC
    __GPL_CRC_Disable();
    __GPL_Inverse_Disable();
    __GPL_ByteOrderChange_Disable();
    __GPL_BeforeBitOrderChange_Select(GPL_BEFORE_BIT_ORDER_DISABLE);
    __GPL_CRC_Mode_Select(GPL_CRC_MODE_CRC32);
    __GPL_CRC_Data_Size_Select(GPL_CRC_DATA_SIZE_8BITS);
    __GPL_CRC_SetInitialValue((uint32_t)InitVlaue);
    __GPL_AfterBitOrderChange_Select(GPL_AFTER_BIT_ORDER_DISABLE);
    __GPL_DataInput(0);
    __GPL_CRC_Enable();

    do{
        GPL->DIN.B[0] = *lpAddr;
        __ISB();
        lpAddr ++;
    }while(-- lCount != 0);

    CRC32Value = GPL->DOUT.W;
    __GPL_CRC_Disable();

    return CRC32Value;
}

#if !defined(MG32_1ST) && !defined(MG32_2ND) && !defined(MG32_3RD)
/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint32_t IEC60730_CRC32_8Bit_DMA(uint32_t InitVlaue, uint8_t *StartAddr, uint32_t Length)
{
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
    #pragma clang diagnostic push
    #pragma clang diagnostic ignored "-Wcast-align"
        uint32_t * lpAddr = (uint32_t *)StartAddr;
    #pragma clang diagnostic pop
#else
    uint32_t * lpAddr = (uint32_t *)StartAddr;
#endif
    uint32_t lCount = Length;
    uint32_t  CRC32Value;

// DMA Initial
    DMA->CR0.W &= ~(DMA_CR0_EN_mask_b0 | DMA_CR0_CH0_ENB_mask_w | DMA_CR0_GPL_CHS_mask_b0);
    DMA->CR0.W &= ~DMA_CR0_FGBUS_SEL_mask_w;
    DMA->CR0.W |= DMA_CR0_EN_mask_h0;

// DMA Channel Config and Calculation Enable
    DMAChannel0->A.W &= ~(DMA_CH0A_CH0_REQ_mask_w |\
                          DMA_CH0A_CH0_EN_mask_w |\
                          DMA_CH0A_CH0_CIE_mask_w |\
                          DMA_CH0A_CH0_HIE_mask_w |\
                          DMA_CH0A_CH0_EIE_mask_w);

    DMAChannel0->A.W = (DMA_CH0A_CH0_ERR2F_mask_w |\
                        DMA_CH0A_CH0_TH2F_mask_w |\
                        DMA_CH0A_CH0_TC2F_mask_w |\
                        DMA_CH0A_CH0_CIE_disable_w |\
                        DMA_CH0A_CH0_HIE_disable_w |\
                        DMA_CH0A_CH0_EIE_disable_w |\
                        DMA_CH0A_CH0_BSIZE_one_w |\
                        DMA_CH0A_CH0_XMDS_disable_w |\
                        DMA_CH0A_CH0_LAST_not_w |\
                        DMA_CH0A_CH0_ADSEL_normal_w |\
                        DMA_CH0A_CH0_LOOP_disable_w |\
                        DMA_CH0A_CH0_HOLD_disable_w |\
                        0);

    DMAChannel0->B.W = (DMA_CH0B_CH0_DSYNC_disable_w |\
                        DMA_CH0B_CH0_SSYNC_disable_w |\
                        DMA_CH0B_CH0_DINC_disable_w |\
                        DMA_CH0B_CH0_SINC_enable_w |\
                        (DMA_GPL_Write << DMA_CH0B_CH0_DET_shift_w) |\
                        (DMA_MEM_Read << DMA_CH0B_CH0_SRC_shift_w) |\
                        0);

    DMAChannel0->SSA.W = ((uint32_t)lpAddr);
    DMAChannel0->DSA.W = 0;
    DMAChannel0->NUM.H[0] = (uint16_t)lCount;

    DMAChannel0->A.W |= DMA_CH0A_CH0_EN_mask_w;
    DMAChannel0->A.W |= DMA_CH0A_CH0_REQ_mask_w;

// GPL CRC Initial
    // width=32 poly=0x04c11db7 init=0x00000000 refin=false refout=false xorout=0xffffffff check=0x765e7680 residue=0xc704dd7b name="CRC-32/POSIX"
    GPL->CR0.W &= ~GPL_CR0_DMA_EN_mask_w;
    GPL->CR1.W &= ~GPL_CR1_CRC_EN_mask_w;
    GPL->CR0.W = GPL_CR0_PAR_POL_even_w | \
                 GPL_CR0_IN_INV_disable_w | \
                 GPL_CR0_BEND_EN_disable_w | \
                 GPL_CR0_BEND16_EN_disable_w | \
                 GPL_CR0_BREV_MDS_disable_w | \
                 0;

    GPL->CR1.W = GPL_CR1_CRC_DSIZE_8bit_w | \
                 GPL_CR1_CRC_MDS_crc32_w | \
                 GPL_CR1_CRC_BREV_disable_w | \
                 0;

    GPL->CRCINIT.W = (uint32_t)InitVlaue;
    GPL->DIN.W = 0;

    GPL->CR1.W |= GPL_CR1_CRC_EN_mask_w;
    GPL->CR0.W |= GPL_CR0_DMA_EN_mask_w;

    while((DMA->STA.W & (DMA_STA_CH0_ERRF_mask_w | DMA_STA_CH0_TCF_mask_w)) == 0);

    CRC32Value = GPL->DOUT.W;

    GPL->CR0.W &= ~GPL_CR0_DMA_EN_mask_w;
    GPL->CR1.W &= ~GPL_CR1_CRC_EN_mask_w;

    DMAChannel0->A.W &= ~DMA_CH0A_CH0_EN_mask_w;

    return CRC32Value;
}
#endif

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint32_t IEC60730_CRC32_32Bit(uint32_t InitVlaue, uint32_t *StartAddr, uint32_t Length)
{
    uint32_t * lpAddr = (uint32_t *)StartAddr;
    uint32_t lCount = (Length >> 2);
    uint32_t  CRC32Value;

    // width=32 poly=0x04c11db7 init=0x00000000 refin=false refout=false xorout=0xffffffff check=0x765e7680 residue=0xc704dd7b name="CRC-32/POSIX"
    // Initial GPL CRC
    __GPL_CRC_Disable();
    __GPL_Inverse_Disable();
    __GPL_ByteOrderChange_Enable();
    __GPL_BeforeBitOrderChange_Select(GPL_BEFORE_BIT_ORDER_DISABLE);
    __GPL_CRC_Mode_Select(GPL_CRC_MODE_CRC32);
    __GPL_CRC_Data_Size_Select(GPL_CRC_DATA_SIZE_32BITS);
    __GPL_CRC_SetInitialValue((uint32_t)InitVlaue);
    __GPL_AfterBitOrderChange_Select(GPL_AFTER_BIT_ORDER_DISABLE);
    __GPL_DataInput(0);
    __GPL_CRC_Enable();

    do{
        GPL->DIN.W = *lpAddr;
        __ISB();
        lpAddr ++;
    }while(-- lCount != 0);

    CRC32Value = GPL->DOUT.W;
    __GPL_CRC_Disable();

    return CRC32Value;
}

#if !defined(MG32_1ST) && !defined(MG32_2ND) && !defined(MG32_3RD)
/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
uint32_t IEC60730_CRC32_32Bit_DMA(uint32_t InitVlaue, uint32_t *StartAddr, uint32_t Length)
{
    uint32_t * lpAddr = (uint32_t *)StartAddr;
    uint32_t lCount = (Length >> 2);
    uint32_t  CRC32Value;

// DMA Initial
    DMA->CR0.W &= ~(DMA_CR0_EN_mask_b0 | DMA_CR0_CH0_ENB_mask_w | DMA_CR0_GPL_CHS_mask_b0);
    DMA->CR0.W |= DMA_CR0_FGBUS_SEL_mask_w;
    DMA->CR0.W |= DMA_CR0_EN_mask_h0;

// DMA Channel Config and Calculation Enable
    DMAChannel0->A.W &= ~(DMA_CH0A_CH0_REQ_mask_w |\
                          DMA_CH0A_CH0_EN_mask_w |\
                          DMA_CH0A_CH0_CIE_mask_w |\
                          DMA_CH0A_CH0_HIE_mask_w |\
                          DMA_CH0A_CH0_EIE_mask_w);

    DMAChannel0->A.W = (DMA_CH0A_CH0_ERR2F_mask_w |\
                        DMA_CH0A_CH0_TH2F_mask_w |\
                        DMA_CH0A_CH0_TC2F_mask_w |\
                        DMA_CH0A_CH0_CIE_disable_w |\
                        DMA_CH0A_CH0_HIE_disable_w |\
                        DMA_CH0A_CH0_EIE_disable_w |\
                        DMA_CH0A_CH0_BSIZE_one_w |\
                        DMA_CH0A_CH0_XMDS_disable_w |\
                        DMA_CH0A_CH0_LAST_not_w |\
                        DMA_CH0A_CH0_ADSEL_normal_w |\
                        DMA_CH0A_CH0_LOOP_disable_w |\
                        DMA_CH0A_CH0_HOLD_disable_w |\
                        0);

    DMAChannel0->B.W = (DMA_CH0B_CH0_DSYNC_disable_w |\
                        DMA_CH0B_CH0_SSYNC_disable_w |\
                        DMA_CH0B_CH0_DINC_disable_w |\
                        DMA_CH0B_CH0_SINC_enable_w |\
                        (DMA_GPL_Write << DMA_CH0B_CH0_DET_shift_w) |\
                        (DMA_MEM_Read << DMA_CH0B_CH0_SRC_shift_w) |\
                        0);

    DMAChannel0->SSA.W = ((uint32_t)lpAddr);
    DMAChannel0->DSA.W = 0;
    DMAChannel0->NUM.H[0] = (uint16_t)lCount;

    DMAChannel0->A.W |= DMA_CH0A_CH0_EN_mask_w;
    DMAChannel0->A.W |= DMA_CH0A_CH0_REQ_mask_w;

// GPL CRC Initial
    // width=32 poly=0x04c11db7 init=0x00000000 refin=false refout=false xorout=0xffffffff check=0x765e7680 residue=0xc704dd7b name="CRC-32/POSIX"
    GPL->CR0.W &= ~GPL_CR0_DMA_EN_mask_w;
    GPL->CR1.W &= ~GPL_CR1_CRC_EN_mask_w;
    GPL->CR0.W = GPL_CR0_PAR_POL_even_w | \
                 GPL_CR0_IN_INV_disable_w | \
                 GPL_CR0_BEND_EN_enable_w | \
                 GPL_CR0_BEND16_EN_disable_w | \
                 GPL_CR0_BREV_MDS_disable_w | \
                 0;

    GPL->CR1.W = GPL_CR1_CRC_DSIZE_32bit_w | \
                 GPL_CR1_CRC_MDS_crc32_w | \
                 GPL_CR1_CRC_BREV_disable_w | \
                 0;

    GPL->CRCINIT.W = (uint32_t)InitVlaue;
    GPL->DIN.W = 0;

    GPL->CR1.W |= GPL_CR1_CRC_EN_mask_w;
    GPL->CR0.W |= GPL_CR0_DMA_EN_mask_w;

    while((DMA->STA.W & (DMA_STA_CH0_ERRF_mask_w | DMA_STA_CH0_TCF_mask_w)) == 0);

    CRC32Value = GPL->DOUT.W;

    GPL->CR0.W &= ~GPL_CR0_DMA_EN_mask_w;
    GPL->CR1.W &= ~GPL_CR1_CRC_EN_mask_w;

    DMAChannel0->A.W &= ~DMA_CH0A_CH0_EN_mask_w;

    return CRC32Value;
}
#endif

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_CRC16_8Bit(uint32_t StartAddr, uint32_t Length, uint16_t CRCValue)
{
    IEC60730_Ret ReturnState = IEC60730_TEST_SUCCESS;
    // Check Length
    if(Length == 0)
        return IEC60730_TEST_FAILURE;

    if(IEC60730_CRC16_8Bit(0, (uint8_t *)StartAddr, Length) != CRCValue)
        ReturnState = IEC60730_TEST_FAILURE;

    if(IEC60730_SWCRC16_8Bit(0, (uint8_t *)StartAddr, Length) != CRCValue)
        ReturnState = IEC60730_TEST_FAILURE;

    return ReturnState;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_CRC16_32Bit(uint32_t StartAddr, uint32_t Length, uint16_t CRCValue)
{
    IEC60730_Ret ReturnState = IEC60730_TEST_SUCCESS;
    // Check Address and Length
    if(((StartAddr & 0x00000003) != 0) || \
       ((Length & 0x00000003) != 0) || \
        (Length == 0))
        return IEC60730_TEST_FAILURE;

    if(IEC60730_CRC16_32Bit(0, (uint32_t *)StartAddr, Length) != CRCValue)
        ReturnState = IEC60730_TEST_FAILURE;

    if(IEC60730_SWCRC16_8Bit(0, (uint8_t *)StartAddr, Length) != CRCValue)
        ReturnState = IEC60730_TEST_FAILURE;

    return ReturnState;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_CRC32_8Bit(uint32_t StartAddr, uint32_t Length, uint32_t CRCValue)
{
    IEC60730_Ret ReturnState = IEC60730_TEST_SUCCESS;
    // Check Length
    if(Length == 0)
        return IEC60730_TEST_FAILURE;

    if(IEC60730_CRC32_8Bit(0, (uint8_t *)StartAddr, Length) != CRCValue)
        ReturnState = IEC60730_TEST_FAILURE;

    if(IEC60730_SWCRC32_8Bit(0, (uint8_t *)StartAddr, Length) != CRCValue)
        ReturnState = IEC60730_TEST_FAILURE;

    return ReturnState;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_CRC32_32Bit(uint32_t StartAddr, uint32_t Length, uint32_t CRCValue)
{
    IEC60730_Ret ReturnState = IEC60730_TEST_SUCCESS;
    // Check Address and Length
    if(((StartAddr & 0x00000003) != 0) || \
       ((Length & 0x00000003) != 0) || \
        (Length == 0))
        return IEC60730_TEST_FAILURE;

  #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
    if(IEC60730_CRC32_32Bit(0, (uint32_t *)StartAddr, Length) != CRCValue)
  #else
    if(IEC60730_CRC32_32Bit_DMA(0, (uint32_t *)StartAddr, Length) != CRCValue)
  #endif
        ReturnState = IEC60730_TEST_FAILURE;

    if(IEC60730_SWCRC32_8Bit(0, (uint8_t *)StartAddr, Length) != CRCValue)
        ReturnState = IEC60730_TEST_FAILURE;

    return ReturnState;
}


static uint32_t APCRCCalcCount;
static uint32_t APCRCChecksum;
static uint32_t IAPCRCCalcCount;
static uint32_t IAPCRCChecksum;

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_Initial_IAP(void)
{
    IAPCRCCalcCount = 0;
    IAPCRCChecksum = 0;

  #if(IAP_Planning && IAP_CRCCheckEnable && IAP_SelfCalculation)
    #if((IAP_CheckStartAddress + IAP_CheckSize) > (IAP_Size - 4))
      #error "IAP CRC Check Size Over"
    #endif
    if(IAPCRCChecksumValue == 0xFFFFFFFF)
    {
      #if IAP_CRC_Mode == 1
        IAPCRCChecksum = IEC60730_CRC16_32Bit(0, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize);
      #endif
      #if IAP_CRC_Mode == 2
        #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
        IAPCRCChecksum = IEC60730_CRC32_32Bit(0, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize);
        #else
        IAPCRCChecksum = IEC60730_CRC32_32Bit_DMA(0, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize);
        #endif
      #endif
      #if IAP_CRC_Mode == 3
        IAPCRCChecksum = IEC60730_CRC16_8Bit(0, (uint8_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize);
      #endif
      #if IAP_CRC_Mode == 4
        IAPCRCChecksum = IEC60730_CRC32_8Bit(0, (uint8_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize);
      #endif

        __MEM_UnProtect();
        __MEM_Enable();
        __MEM_IAPWrite_Enable();
        __MEM_SetWriteMode(IAPProgram);
        __MEM_SingleWriteUnProtect();
        IAPCRCChecksumValue = IAPCRCChecksum;
        __MEM_Protect();
    }
  #endif
    return IEC60730_TEST_SUCCESS;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_Initial(void)
{
    volatile uint32_t lChecksum;
    volatile uint32_t lIAPSizeTemp;

    APCRCCalcCount = 0;
    APCRCChecksum = 0;

  #if IAP_Planning
    if(((uint8_t)(IAP_Size >> 10)) != MEM->IAPSZ.B[1])
    {
//        printf(" Error IAP Size Not Match\n\r");
//        printf(" IAP Size Modify to %d \n\r", IAP_Size);
        __MEM_UnProtect();
        //__MEM_Enable();
        __MEM_SetIAPSize(IAP_Size);
        __MEM_Protect();
    }
  #endif

  #if(AP_SelfCalculation)
    if(APCRCChecksumSize == 0xFFFFFFFF)
    {
        lIAPSizeTemp = (uint32_t)Flash_Size - (((uint32_t)CFG->OR02.W) << 2);
        __MEM_UnProtect();
        __MEM_SetIAPSize(lIAPSizeTemp);
        __MEM_Enable();
        __MEM_IAPWrite_Enable();
        __MEM_SetWriteMode(IAPProgram);
        __MEM_SingleWriteUnProtect();
        //*(uint32_t *)APCRCChecksumSize = ((uint32_t)(AP_CheckSize));
        *(uint32_t *)0x1A000024 = ((uint32_t)(AP_CheckSize));
    #if IAP_Planning
        lIAPSizeTemp = IAP_Size;
    #else
        lIAPSizeTemp = (((uint32_t)CFG->OR01.W) << 2);
    #endif
        __MEM_SetIAPSize(lIAPSizeTemp);
        __MEM_Protect();
    }

    if(APCRCChecksumValue == 0xFFFFFFFF)
    {
      #if AP_CRC_Mode == 1
        lChecksum = IEC60730_CRC16_32Bit(0, (uint32_t *)0x0, 0x28);
        lChecksum = IEC60730_CRC16_32Bit(lChecksum, ((uint32_t *)0x2C), (APCRCChecksumSize - 0x2C));
      #endif
      #if AP_CRC_Mode == 2
        #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
        lChecksum = IEC60730_CRC32_32Bit(0, (uint32_t *)0x0, 0x28);
        lChecksum = IEC60730_CRC32_32Bit(lChecksum, ((uint32_t *)0x2C), (APCRCChecksumSize - 0x2C));
        #else
        lChecksum = IEC60730_CRC32_32Bit_DMA(0, (uint32_t *)0x0, 0x28);
        lChecksum = IEC60730_CRC32_32Bit_DMA(lChecksum, ((uint32_t *)0x2C), (APCRCChecksumSize - 0x2C));
        #endif
      #endif
      #if AP_CRC_Mode == 3
        lChecksum = IEC60730_CRC16_8Bit(0, (uint8_t *)0x0, 0x28);
        lChecksum = IEC60730_CRC16_8Bit(lChecksum, ((uint8_t *)0x2C), (APCRCChecksumSize - 0x2C));
      #endif
      #if AP_CRC_Mode == 4
        lChecksum = IEC60730_CRC32_8Bit(0, (uint8_t *)0x0, 0x28);
        lChecksum = IEC60730_CRC32_8Bit(lChecksum, ((uint8_t *)0x2C), (APCRCChecksumSize - 0x2C));
      #endif

        lIAPSizeTemp = (uint32_t)Flash_Size - (((uint32_t)CFG->OR02.W) << 2);
        __MEM_UnProtect();
        __MEM_SetIAPSize(lIAPSizeTemp);
        __MEM_Enable();
        __MEM_IAPWrite_Enable();
        __MEM_SetWriteMode(IAPProgram);
        __MEM_SingleWriteUnProtect();
        //MEM_Write((uint32_t)&__AP_CRC_Check_Sum, lChecksum);
        //APCRCChecksumValue = ((uint32_t)(lChecksum));
        *(uint32_t *)0x1A000028 = ((uint32_t)lChecksum);
    #if IAP_Planning
        lIAPSizeTemp = IAP_Size;
    #else
        lIAPSizeTemp = (((uint32_t)CFG->OR01.W) << 2);
    #endif
        __MEM_SetIAPSize(lIAPSizeTemp);
        __MEM_Protect();
    }
  #endif

    IEC60730_Flash_Initial_IAP();

    return IEC60730_TEST_SUCCESS;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_StartUp_IAP(void)
{
    volatile IEC60730_Ret ReturnState = IEC60730_TEST_SUCCESS;

  #if(IAP_Planning && IAP_CRCCheckEnable)
    #if((IAP_CheckStartAddress + IAP_CheckSize) > (IAP_Size - 4))
      #error "IAP CRC Check Size Over"
    #endif
    #if IAP_CRC_Mode == 1
      if(IEC60730_CRC16_32Bit(0, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize) != (*(uint16_t *)((0x1A000000 + IAP_Size) - 4)))
    #endif
    #if IAP_CRC_Mode == 2
      #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
      if(IEC60730_CRC32_32Bit(0, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize) != (*(uint32_t *)((0x1A000000 + IAP_Size) - 4)))
      #else
      if(IEC60730_CRC32_32Bit_DMA(0, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize) != (*(uint32_t *)((0x1A000000 + IAP_Size) - 4)))
      #endif
    #endif
    #if IAP_CRC_Mode == 3
      if(IEC60730_CRC16_8Bit(0, (uint8_t *)(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize) != (*(uint16_t *)((0x1A000000 + IAP_Size) - 4)))
    #endif
    #if IAP_CRC_Mode == 4
      if(IEC60730_CRC32_8Bit(0, (uint8_t *)I(0x1A000000 + IAP_CheckStartAddress), IAP_CheckSize) != (*(uint32_t *)((0x1A000000 + IAP_Size) - 4)))
    #endif
          ReturnState = IEC60730_TEST_FAILURE;
  #endif

    return ReturnState;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_StartUp(void)
{
    volatile uint32_t lChecksum = 0;
    volatile IEC60730_Ret ReturnState = IEC60730_TEST_SUCCESS;

    #if AP_CheckSize > AP_Size
      #error "AP CRC check over range"
    #endif
  #if AP_CRC_Mode == 1
    lChecksum = IEC60730_CRC16_32Bit(0, (uint32_t *)0x0, 0x28);
    lChecksum = IEC60730_CRC16_32Bit(lChecksum, ((uint32_t *)0x2C), (APCRCChecksumSize - 0x2C));
  #endif
  #if AP_CRC_Mode == 2
    #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
    lChecksum = IEC60730_CRC32_32Bit(0, (uint32_t *)0x0, 0x28);
    lChecksum = IEC60730_CRC32_32Bit(lChecksum, ((uint32_t *)0x2C), (APCRCChecksumSize - 0x2C));
    #else
    lChecksum = IEC60730_CRC32_32Bit_DMA(0, (uint32_t *)0x0, 0x28);
    lChecksum = IEC60730_CRC32_32Bit_DMA(lChecksum, ((uint32_t *)0x2C), (APCRCChecksumSize - 0x2C));
    #endif
  #endif
  #if AP_CRC_Mode == 3
    lChecksum = IEC60730_CRC16_8Bit(0, (uint8_t *)0x0, 0x28);
    lChecksum = IEC60730_CRC16_8Bit(lChecksum, ((uint8_t *)0x2C), (APCRCChecksumSize - 0x2C));
  #endif
  #if AP_CRC_Mode == 4
    lChecksum = IEC60730_CRC32_8Bit(0, (uint8_t *)0x0, 0x28);
    lChecksum = IEC60730_CRC32_8Bit(lChecksum, ((uint8_t *)0x2C), (APCRCChecksumSize - 0x2C));
  #endif
    if(lChecksum != (*(uint32_t *)0x00000028))
        ReturnState = IEC60730_TEST_FAILURE;

    ReturnState |= IEC60730_Flash_StartUp_IAP();
    return ReturnState;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_RunTime_AP(void)
{
    if((APCRCCalcCount & 0xFFFFFFC0) == 0)
    {
      #if IAP_CRC_Mode == 1
        APCRCChecksum = IEC60730_CRC16_32Bit(0, (uint32_t *)0x0, 0x28);
        APCRCChecksum = IEC60730_CRC16_32Bit(APCRCChecksum, ((uint32_t *)0x2C), (AP_RuntimeCheckSize - 0x2C));
      #endif
      #if IAP_CRC_Mode == 2
        #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
        APCRCChecksum = IEC60730_CRC32_32Bit(0, (uint32_t *)0x0, 0x28);
        APCRCChecksum = IEC60730_CRC32_32Bit(APCRCChecksum, ((uint32_t *)0x2C), (AP_RuntimeCheckSize - 0x2C));
        #else
        APCRCChecksum = IEC60730_CRC32_32Bit_DMA(0, (uint32_t *)0x0, 0x28);
        APCRCChecksum = IEC60730_CRC32_32Bit_DMA(APCRCChecksum, ((uint32_t *)0x2C), (AP_RuntimeCheckSize - 0x2C));
        #endif
      #endif
      #if IAP_CRC_Mode == 3
        APCRCChecksum = IEC60730_CRC16_8Bit(0, (uint8_t *)0x0, 0x28);
        APCRCChecksum = IEC60730_CRC16_8Bit(APCRCChecksum, ((uint32_t *)0x2C), (AP_RuntimeCheckSize - 0x2C));
      #endif
      #if IAP_CRC_Mode == 4
        APCRCChecksum = IEC60730_CRC32_8Bit(0, (uint8_t *)0x0, 0x28);
        APCRCChecksum = IEC60730_CRC32_8Bit(APCRCChecksum, ((uint32_t *)0x2C), (AP_RuntimeCheckSize - 0x2C));
      #endif
      APCRCCalcCount += AP_RuntimeCheckSize;
    }
    else
    {
        if((APCRCChecksumSize - APCRCCalcCount) > AP_RuntimeCheckSize)
        {
          #if IAP_CRC_Mode == 1
            APCRCChecksum = IEC60730_CRC16_32Bit(APCRCChecksum, (uint32_t *)APCRCCalcCount, AP_RuntimeCheckSize);
          #endif
          #if IAP_CRC_Mode == 2
            #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
            APCRCChecksum = IEC60730_CRC32_32Bit(APCRCChecksum, (uint32_t *)APCRCCalcCount, AP_RuntimeCheckSize);
            #else
            APCRCChecksum = IEC60730_CRC32_32Bit_DMA(APCRCChecksum, (uint32_t *)APCRCCalcCount, AP_RuntimeCheckSize);
            #endif
          #endif
          #if IAP_CRC_Mode == 3
            APCRCChecksum = IEC60730_CRC16_8Bit(APCRCChecksum, (uint8_t *)APCRCCalcCount, AP_RuntimeCheckSize);
          #endif
          #if IAP_CRC_Mode == 4
            APCRCChecksum = IEC60730_CRC32_8Bit(APCRCChecksum, (uint8_t *)APCRCCalcCount, AP_RuntimeCheckSize);
          #endif
          APCRCCalcCount += AP_RuntimeCheckSize;
        }
        else
        {
          #if IAP_CRC_Mode == 1
            APCRCChecksum = IEC60730_CRC16_32Bit(APCRCChecksum, (uint32_t *)APCRCCalcCount, (APCRCChecksumSize - APCRCCalcCount));
          #endif
          #if IAP_CRC_Mode == 2
            #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
            APCRCChecksum = IEC60730_CRC32_32Bit(APCRCChecksum, (uint32_t *)APCRCCalcCount, (APCRCChecksumSize - APCRCCalcCount));
            #else
            APCRCChecksum = IEC60730_CRC32_32Bit_DMA(APCRCChecksum, (uint32_t *)APCRCCalcCount, (APCRCChecksumSize - APCRCCalcCount));
            #endif
          #endif
          #if IAP_CRC_Mode == 3
            APCRCChecksum = IEC60730_CRC16_8Bit(APCRCChecksum, (uint8_t *)APCRCCalcCount, (APCRCChecksumSize - APCRCCalcCount));
          #endif
          #if IAP_CRC_Mode == 4
            APCRCChecksum = IEC60730_CRC32_8Bit(APCRCChecksum, (uint8_t *)APCRCCalcCount, (APCRCChecksumSize - APCRCCalcCount));
          #endif
            APCRCCalcCount = 0;
            if(APCRCChecksumValue != APCRCChecksum)
                return IEC60730_TEST_FAILURE;
        }
    }
    return IEC60730_TEST_SUCCESS;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Flash_RunTime_IAP(void)
{
  #if(IAP_Planning && IAP_CRCCheckEnable)
    if((IAP_CheckSize - IAPCRCCalcCount) > IAP_RuntimeCheckSize)
    {
      #if IAP_CRC_Mode == 1
        IAPCRCChecksum = IEC60730_CRC16_32Bit(IAPCRCChecksum, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), IAP_RuntimeCheckSize);
      #endif
      #if IAP_CRC_Mode == 2
        #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
        IAPCRCChecksum = IEC60730_CRC32_32Bit(IAPCRCChecksum, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), IAP_RuntimeCheckSize);
        #else
        IAPCRCChecksum = IEC60730_CRC32_32Bit_DMA(IAPCRCChecksum, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), IAP_RuntimeCheckSize);
        #endif
      #endif
      #if IAP_CRC_Mode == 3
        IAPCRCChecksum = IEC60730_CRC16_8Bit(IAPCRCChecksum, (uint8_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), IAP_RuntimeCheckSize);
      #endif
      #if IAP_CRC_Mode == 4
        IAPCRCChecksum = IEC60730_CRC32_8Bit(IAPCRCChecksum, (uint8_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), IAP_RuntimeCheckSize);
      #endif
       IAPCRCCalcCount += IAP_RuntimeCheckSize;
    }
    else
    {
      #if IAP_CRC_Mode == 1
        IAPCRCChecksum = IEC60730_CRC16_32Bit(IAPCRCChecksum, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), (IAP_CheckSize - IAPCRCCalcCount));
      #endif
      #if IAP_CRC_Mode == 2
        #if defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD)
        IAPCRCChecksum = IEC60730_CRC32_32Bit(IAPCRCChecksum, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), (IAP_CheckSize - IAPCRCCalcCount));
        #else
        IAPCRCChecksum = IEC60730_CRC32_32Bit_DMA(IAPCRCChecksum, (uint32_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), (IAP_CheckSize - IAPCRCCalcCount));
        #endif
      #endif
      #if IAP_CRC_Mode == 3
        IAPCRCChecksum = IEC60730_CRC16_8Bit(IAPCRCChecksum, (uint8_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), (IAP_CheckSize - IAPCRCCalcCount));
      #endif
      #if IAP_CRC_Mode == 4
        IAPCRCChecksum = IEC60730_CRC32_8Bit(IAPCRCChecksum, (uint8_t *)(0x1A000000 + IAP_CheckStartAddress + IAPCRCCalcCount), (IAP_CheckSize - IAPCRCCalcCount));
      #endif
        if(IAPCRCChecksumValue != IAPCRCChecksum)
            return IEC60730_TEST_FAILURE;
        IAPCRCCalcCount = 0;
        IAPCRCChecksum = 0;
    }
    return IEC60730_TEST_SUCCESS;
  #else
    return IEC60730_TEST_FAILURE;
  #endif
}


