/**
 ******************************************************************************
 *
 * @file        MG32_IEC60730_CSC_RunTime.c
 *
 * @brief       IEC60730 Class-C : CSC test in Run Time state.
 *
 * @par         Project
 *              MG32
 * @version     V1.04
 * @date        2025/07/30
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par     Disclaimer
 *    The Demo software is provided "AS IS"  without any warranty, either
 *    expressed or implied, including, but not limited to, the implied warranties
 *    of merchantability and fitness for a particular purpose.  The author will
 *    not be liable for any special, incidental, consequential or indirect
 *    damages due to loss of data or any other reason.
 *    These statements agree with the world wide and local dictated laws about
 *    authorship and violence against these laws.
 ******************************************************************************
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_IEC60730_Common.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define Module_Unprotect_Key    0xA217
#define Module_Protect_Key      0x0000
#define XOSC_Clock              12000000
#if XOSC_Clock <= 32768
#error "Cannot Calculation low speed frequency"
#endif

// SysTick Define
//#define TICK_CR                   0xE000E010      // System Tick Configure Register Address Define
//#define MG32_SYSTICK_EN_w         0x00000001      // System Tick Function Enable
//#define MG32_SYSTICk_CLKSEL_w     0x00000004      // System Tick Clock Source Select
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
uint32_t Get_CK_APB_Count_In_CK_LS (void);
IEC60730_Ret IEC60730_Clock_RunTime(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief           Get the number of CK_APB clocks in one CK_LS cycle.
 * @details         
 * @param[in]    
 * @return      
 * @exception      
 * @note        
 *******************************************************************************
 */
uint32_t Get_CK_APB_Count_In_CK_LS (void)
{
    // If defined TM20 module 
    // Use TM20 to get the number of CK_APB clocks in one CK_LS cycle.
    #if defined(TM20_Base)
    
    CSC->KEY.H[0] = 0xA217;
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_main_b0;                 // Internal clock output source select (0x0 = CK_MAIN).
    CSC->CKS2.B[2] &= ~CSC_CKS2_TM20_CKS_mask_b2;               // TM20 process clock source select (0x0 = CK_APB).
    CSC->APB1.B[1] |= CSC_APB1_TM20_EN_enable_b1;               // TM20 module clock source enable.
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_ls_b0;                   // Internal clock output source select (0x4 = CK_LS).
    CSC->KEY.H[0] = 0x0000;

    // Deinitial
    TM20->CR0.W = 0x00000000;
    TM20->STA.W = 0xFFFFFFFF;
    TM20->INT.W = 0x00000000;
    TM20->CLK.W = 0x00000000;
    TM20->CNT.W = 0x00000000;
    TM20->PSCNT.W = 0x00000000;
    TM20->OSCR.W = 0x0000F0F0;
    TM20->OCCR0.W = 0x00000000;
    TM20->OCCR1.W = 0x00000000;
    TM20->PWM.W = 0x00000000;
    TM20->BS.W = 0x00000000;
    TM20->CC0A.W = 0x00000000;
    TM20->CC0B.W = 0x00000000;

    // Initial
    TM20->ARR.H[0] = 0xFFFF;                                    // Main timer/counter auto-reload value register (0xFFFF).
    TM20->PSARR.H[0] = 0xFFFF;                                  // Timer prescaler or 2nd timer/counter auto-reload value register (0xFFFF).

    TM20->CR0.W = TM_CR0_MDS_separate_w |                       // Timer operation mode select (0x2 = Full-Counter : 32-bit counter Mode).
                  TM_CR0_DIR_up_w;                              // Main Timer counting direction bit (Up (Up Counting)).

    TM20->CLK.W = TM_CLK_CKI_SEL_proc_w;                        // Timer input clock CK_TM36 source select (PROC : CK_TM36_PR process clock from CSC).

    TM20->CR1.W = TM_CR1_OVR0_MDS_keep_w;                       // Timer channel 0 capture data buffer overrun mode select (Overwritten (Keep (Preserved old data)).
    TM20->CCMDS.W = TM_CCMDS_CC0_MDS_16bit_ic_w;                // Timer channel 0 capture and compare mode select (0x1 = 16bit_IC (Input capture)).
    APB->CR2.B[0] = APB_CR2_ITR7_MUX_trg8_b0;                   // APB ITR7 = ICKO_INT
    TM20->TRG.W = TM_TRG_ITR_MUX_itr7_w |                       // Trigger source ITR7
                  TM_TRG_TRG_MUX_in0_w |                        // Timer trigger source TRGI select IN0 (TM20_IN0).
                  TM_TRG_TRGI_MDS_trigger_f_w |                 // Timer trigger input mode select (Trigger-F (TRGI falling)).
                  TM_TRG_TRGI2_MDS_trigger_f_w;                 // 2nd Timer trigger input mode select (Trigger-F (TRGI falling)).
    TM20->ICCR.W = TM_ICCR_IC0_MUX_ic01_w |                     // Timer channel 0 input Mux select for input capture (TM36_ITR).
                   TM_ICCR_IC0_TRGS_falling_edge_w;             // Timer channel 0 input trigger edge select (Falling edge).
                   
    // Clear flag
    TM20->STA.W = TM_STA_CF0A_mask_w |                          //  Clear flag CFOA and CF0B
                  TM_STA_CF0B_mask_w;  
              
    // Enable TM36
    TM20->CR0.W |= TM_CR0_EN_enable_w |                         // Enable main Timer/Counter and 2nd Timer/Counter.
                   TM_CR0_EN2_enable_w;    
    
    // Wait flag CF0A and CF0B happened.
    while((TM20->STA.W & (TM_STA_CF0A_mask_w | TM_STA_CF0B_mask_w)) != (TM_STA_CF0A_happened_w | TM_STA_CF0B_happened_w));
    
    TM20->CR0.W &= ~(TM_CR0_EN_enable_w |                       // Enable main Timer/Counter and 2nd Timer/Counter.
                     TM_CR0_EN2_enable_w);    
    
    return (((uint32_t) TM20->CC0A.H[0]) * 32000);


    // If not defined TM20 module 
    // Use TM36 to get the number of CK_APB clocks in one CK_LS cycle.
    #else

    CSC->KEY.H[0] = 0xA217;
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_main_b0;                 // Internal clock output source select (0x0 = CK_MAIN).
    CSC->CKS2.B[3] &= ~CSC_CKS2_TM36_CKS_mask_b3;               // TM36 process clock source select (0x0 = CK_APB).
    CSC->APB1.B[1] |= CSC_APB1_TM36_EN_enable_b1;               // TM36 module clock source enable.
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_ls_b0;                   // Internal clock output source select (0x4 = CK_LS).
    CSC->KEY.H[0] = 0x0000;

    // Deinitial
    TM36->CR0.W = 0x00000000;
    TM36->STA.W = 0xFFFFFFFF;
    TM36->INT.W = 0x00000000;
    TM36->CKO.W = 0x00000008;
    TM36->CNT.W = 0x00000000;
    TM36->PSCNT.W = 0x00000000;
    TM36->OSCR.W = 0x0000F0F0;
    TM36->OCCR0.W = 0x00000000;
    TM36->OCCR1.W = 0x00000000;
    TM36->PWM.W = 0x00000000;
    TM36->BS.W = 0x00000000;
    TM36->CC0A.W = 0x00000000;
    TM36->CC0B.W = 0x00000000;
    
    // Intial
    TM36->ARR.H[0] = 0xFFFF;                                    // Main timer/counter auto-reload value register (0xFFFF).
    TM36->PSARR.H[0] = 0xFFFF;                                  // Timer prescaler or 2nd timer/counter auto-reload value register (0xFFFF).
    TM36->CR0.W = TM_CR0_MDS_separate_w |                       // Timer operation mode select (Separate : Separated two 16-bit counters Mode).
                  TM_CR0_DIR_up_w;                              // Main Timer counting direction bit (Up (Up Counting)).
    TM36->CLK.W = TM_CLK_CKI_SEL_proc_w;                        // Timer input clock CK_TM36 source select (PROC : CK_TM36_PR process clock from CSC).
    TM36->CR1.W = TM_CR1_OVR0_MDS_keep_w;                       // Timer channel 0 capture data buffer overrun mode select (Overwritten (Keep (Preserved old data)).
    TM36->CCMDS.W = TM_CCMDS_CC0_MDS_16bit_ic_w;                // Timer channel 0 capture and compare mode select (0x1 = 16bit_IC (Input capture)).
    APB->CR2.B[0] = APB_CR2_ITR7_MUX_trg8_b0;                   // APB ITR7 = ICKO_INT
    TM36->TRG.W = TM_TRG_ITR_MUX_itr7_w |                       // Trigger source ITR7
                  TM_TRG_TRG_MUX_in0_w |                        // Timer trigger source TRGI select IN0 (TM36_IN0).
                  TM_TRG_TRGI_MDS_trigger_f_w |                 // Timer trigger input mode select (Trigger-F (TRGI falling)).
                  TM_TRG_TRGI2_MDS_trigger_f_w;                 // 2nd Timer trigger input mode select (Trigger-F (TRGI falling)).
                  
    TM36->ICCR.W = TM_ICCR_IC0_MUX_ic01_w |                     // Timer channel 0 input Mux select for input capture (TM36_ITR).
                   TM_ICCR_IC0_TRGS_falling_edge_w;             // Timer channel 0 input trigger edge select (Falling edge).
                   
    // Clear flag
    TM36->STA.W = TM_STA_CF0A_mask_w |                          //  Clear flag CFOA and CF0B
                  TM_STA_CF0B_mask_w;  
                  
    // Enable TM36
    TM36->CR0.W |= TM_CR0_EN_enable_w |                         // Enable main Timer/Counter and 2nd Timer/Counter.
                   TM_CR0_EN2_enable_w;    
    
    // Wait flag CF0A and CF0B happened.
    while((TM36->STA.W & (TM_STA_CF0A_mask_w | TM_STA_CF0B_mask_w)) != (TM_STA_CF0A_happened_w | TM_STA_CF0B_happened_w));
    
    TM36->CR0.W &= ~(TM_CR0_EN_enable_w |                       // Enable main Timer/Counter and 2nd Timer/Counter.
                     TM_CR0_EN2_enable_w);  
    
    return (((uint32_t) TM36->CC0A.H[0]) * 32000);
    #endif
}


/**
 *******************************************************************************
 * @brief           MG32_IEC60730_CSC_RunTime
 * @details         
 * @param[in]    
 * @return      
 * @exception      
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Clock_RunTime(void)
{
    uint32_t SYS_CLK = 0;
    uint32_t Real_CLK = 0;
    
    
    
    CSC->KEY.H[0] = Module_Unprotect_Key;             // CSC Module Unprotect
    // System Clock Calculate
    if((CSC->CR0.B[1]&CSC_CR0_HS_SEL_mask_b1) == CSC_CR0_HS_SEL_ihrco_b1)                 // CK_HS Select Check(When Select CK_IHRCO)
    {
        if(CSC->CR0.MBIT.IHRCO_EN == ENABLE)
        {
            if((CSC->CR0.B[2]&CSC_CR0_IHRCO_SEL_mask_b2) == CSC_CR0_IHRCO_SEL_12_b2)
            {
                SYS_CLK = 12000000;
            }
            else if((CSC->CR0.B[2]&CSC_CR0_IHRCO_SEL_mask_b2) == CSC_CR0_IHRCO_SEL_11_b2)
            {
                SYS_CLK = 11059200;
            }
        }
    }
    
    if((CSC->CR0.B[1]&CSC_CR0_HS_SEL_mask_b1) == CSC_CR0_HS_SEL_xosc_b1)
    {
        if((CSC->PLL.B[2]&CSC_PLL_XOSC_GN_mask_b2) == CSC_PLL_XOSC_GN_medium_b2)            // XOSC gain check(When select high speed define SYS_CLK).
        {
            SYS_CLK = XOSC_Clock;                   // Please define XOSC_Clock value.(Will affect the system clock calculation result).
        }
        
        //else if((CSC->PLL.B[2]&CSC_PLL_XOSC_GN_mask_b2) == CSC_PLL_XOSC_GN_medium_b2)         // Not Support
            
        //else if((CSC->PLL.B[2]&CSC_PLL_XOSC_GN_mask_b2) == CSC_PLL_XOSC_GN_32k_normal_b2)     // Not Support
            
    }
    
    //if((CSC->CR0.B[0]&CSC_CR0_HS_SEL_mask_b1) == CSC_CR0_HS_SEL_ilrco_b1)               // Not Support
    //{
        //return IEC60730_TEST_SUCCESS;
    //}
    
    if((CSC->CR0.B[1]&CSC_CR0_HS_SEL_mask_b1) == CSC_CR0_HS_SEL_ck_ext_b1)
    {
        SYS_CLK = XOSC_Clock;                       // When used EXTCK please define frequency(Will affect the system clock calculation result).
    }

    // When CK_MAIN = CK_PLLI or CK_PLLO
    if(((CSC->CR0.B[1] & CSC_CR0_MAIN_SEL_mask_b1) == CSC_CR0_MAIN_SEL_ck_plli_b1) || 
       ((CSC->CR0.B[1] & CSC_CR0_MAIN_SEL_mask_b1) == CSC_CR0_MAIN_SEL_ck_pllo_b1))
    {
        // PLL Input Divider Check
        if((CSC->DIV.B[0]&CSC_DIV_PLLI_DIV_mask_b0) == CSC_DIV_PLLI_DIV_div1_b0)
        {
            SYS_CLK = SYS_CLK/1;
        }
        if((CSC->DIV.B[0]&CSC_DIV_PLLI_DIV_mask_b0) == CSC_DIV_PLLI_DIV_div2_b0)
        {
            SYS_CLK = SYS_CLK/2;
        }
        if((CSC->DIV.B[0]&CSC_DIV_PLLI_DIV_mask_b0) == CSC_DIV_PLLI_DIV_div4_b0)
        {
            SYS_CLK = SYS_CLK/4;
        }
        if((CSC->DIV.B[0]&CSC_DIV_PLLI_DIV_mask_b0) == CSC_DIV_PLLI_DIV_div6_b0)
        {
            SYS_CLK = SYS_CLK/6;
        }
    }
    
    // CK_MAIN Clock Check
    if((CSC->CR0.B[1] & CSC_CR0_MAIN_SEL_mask_b1) == CSC_CR0_MAIN_SEL_ck_pllo_b1)
    {
        //  CSC PLL Configure Check
        if((CSC->CR0.B[0]&CSC_CR0_PLL_EN_mask_b0) == CSC_CR0_PLL_EN_enable_b0)
        {
        // PLL Multiplier Check
        #if defined(CSC_PLL_PLL_MDS_mask_w)
            #if defined(CSC_PLL_PLL_MULS_mask_w)
                SYS_CLK = SYS_CLK*(2*CSC->PLL.MBIT.PLL_MULX+CSC->PLL.MBIT.PLL_MULS);
            #else
                SYS_CLK = SYS_CLK * (CSC->PLL.MBIT.PLL_MULX+1);
            #endif
        #else
            if((CSC->PLL.B[1]&CSC_PLL_PLL_MUL_mask_b1) == CSC_PLL_PLL_MUL_16_b1)
            {
                SYS_CLK = SYS_CLK*16;
            }
            if((CSC->PLL.B[1]&CSC_PLL_PLL_MUL_mask_b1) == CSC_PLL_PLL_MUL_24_b1)
            {
                SYS_CLK = SYS_CLK*24;
            }
        #endif
            // PLLO Divider Check
            if((CSC->DIV.B[0]&CSC_DIV_PLLO_DIV_mask_b0)==CSC_DIV_PLLO_DIV_div4_b0)
            {
                SYS_CLK = SYS_CLK/4;
            }
            if((CSC->DIV.B[0]&CSC_DIV_PLLO_DIV_mask_b0)==CSC_DIV_PLLO_DIV_div3_b0)
            {
                SYS_CLK = SYS_CLK/3;
            }
            if((CSC->DIV.B[0]&CSC_DIV_PLLO_DIV_mask_b0)==CSC_DIV_PLLO_DIV_div2_b0)
            {
                SYS_CLK = SYS_CLK/2;
            }
            if((CSC->DIV.B[0]&CSC_DIV_PLLO_DIV_mask_b0)==CSC_DIV_PLLO_DIV_div1_b0)
            {
                SYS_CLK = SYS_CLK/1;
            }
        }
    }
        
    // Peripheral Divider Set.
    if((CSC->DIV.B[2]&CSC_DIV_APB_DIV_mask_b2) == CSC_DIV_APB_DIV_div1_b2)
    {
        SYS_CLK = SYS_CLK/1;
    }
    if((CSC->DIV.B[2]&CSC_DIV_APB_DIV_mask_b2) == CSC_DIV_APB_DIV_div2_b2)
    {
        SYS_CLK = SYS_CLK/2;
    }
    if((CSC->DIV.B[2]&CSC_DIV_APB_DIV_mask_b2) == CSC_DIV_APB_DIV_div4_b2)
    {
        SYS_CLK = SYS_CLK/4;
    }
    if((CSC->DIV.B[2]&CSC_DIV_APB_DIV_mask_b2) == CSC_DIV_APB_DIV_div8_b2)
    {
        SYS_CLK = SYS_CLK/8;
    }
    if((CSC->DIV.B[2]&CSC_DIV_APB_DIV_mask_b2) == CSC_DIV_APB_DIV_div16_b2)
    {
        SYS_CLK = SYS_CLK/16;
    }
    CSC->KEY.H[0] = Module_Protect_Key;
    
    Real_CLK = Get_CK_APB_Count_In_CK_LS(); // Get real frequency.
    
    if((Real_CLK > (SYS_CLK * 1.4)) ||      // When real frequency > (ideal frequency + 40%).
      (Real_CLK < (SYS_CLK * 0.6)))         // When real frequency < ideal frequency - 40%.
    {
        return IEC60730_TEST_FAILURE;       // Return IEC60730 failure.
    }
    
    return IEC60730_TEST_SUCCESS;           // Return IEC60730 success.
}



