/**
 ******************************************************************************
 *
 * @file        MG32_IEC60730_CSC.c
 *
 * @brief       This file provides firmware functions to manage the following 
 *              functionalities of the ADC peripheral:
 *
 * @par         Project
 *              MG32
 * @version     V1.13
 * @date        2025/07/30
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2016 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par 		Disclaimer 
 *		The Demo software is provided "AS IS"  without any warranty, either 
 *		expressed or implied, including, but not limited to, the implied warranties 
 *		of merchantability and fitness for a particular purpose.  The author will 
 *		not be liable for any special, incidental, consequential or indirect 
 *		damages due to loss of data or any other reason. 
 *		These statements agree with the world wide and local dictated laws about 
 *		authorship and violence against these laws. 
 ******************************************************************************

 ******************************************************************************
 */ 
 
 /* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_IEC60730_Common.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define IHRCO0_MaxLimit 625 /* When ILRCO 32KHz - 40%, TM26 count in IHRCO0 */
#define IHRCO0_MinLimit 267 /* When ILRCO 32KHz + 40%, TM26 count in IHRCO0 */
#define IHRCO1_MaxLimit 576 /* When ILRCO 32KHz - 40%, TM26 count in IHRCO1 */
#define IHRCO1_MinLimit 246 /* When ILRCO 32KHz + 40%, TM26 count in IHRCO1 */
#define IHRCOx_MaxRatio 
#define IHRCOx_MinRatio 

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
IEC60730_Ret IEC60730_Clock_Test (void);
uint16_t Get_CK_MAIN_Count_In_CK_LS (void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief           Get the number of CK_APB clocks in one CK_LS cycle.
 * @details         
 * @param[in]    
 * @return      
 * @exception      
 * @note        
 *******************************************************************************
 */
uint16_t Get_CK_MAIN_Count_In_CK_LS (void)
{
    // If defined TM20 module 
    // Use TM20 to get the number of CK_APB clocks in one CK_LS cycle.
    #if defined(TM20_Base)
    
    CSC->KEY.H[0] = 0xA217;
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_main_b0;                 // Internal clock output source select (0x0 = CK_MAIN).
    CSC->CKS2.B[2] &= ~CSC_CKS2_TM20_CKS_mask_b2;               // TM20 process clock source select (0x0 = CK_APB).
    CSC->APB1.B[1] |= CSC_APB1_TM20_EN_enable_b1;               // TM20 module clock source enable.
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_ls_b0;                   // Internal clock output source select (0x4 = CK_LS).
    CSC->KEY.H[0] = 0x0000;

    // Deinitial
    TM20->CR0.W = 0x00000000;
    TM20->STA.W = 0xFFFFFFFF;
    TM20->INT.W = 0x00000000;
    TM20->CLK.W = 0x00000000;
    TM20->CNT.W = 0x00000000;
    TM20->PSCNT.W = 0x00000000;
    TM20->OSCR.W = 0x0000F0F0;
    TM20->OCCR0.W = 0x00000000;
    TM20->OCCR1.W = 0x00000000;
    TM20->PWM.W = 0x00000000;
    TM20->BS.W = 0x00000000;
    TM20->CC0A.W = 0x00000000;
    TM20->CC0B.W = 0x00000000;

    // Initial
    TM20->ARR.H[0] = 0xFFFF;                                    // Main timer/counter auto-reload value register (0xFFFF).
    TM20->PSARR.H[0] = 0xFFFF;                                  // Timer prescaler or 2nd timer/counter auto-reload value register (0xFFFF).

    TM20->CR0.W = TM_CR0_MDS_separate_w |                       // Timer operation mode select (0x2 = Full-Counter : 32-bit counter Mode).
                  TM_CR0_DIR_up_w;                              // Main Timer counting direction bit (Up (Up Counting)).

    TM20->CLK.W = TM_CLK_CKI_SEL_proc_w;                        // Timer input clock CK_TM36 source select (PROC : CK_TM36_PR process clock from CSC).

    TM20->CR1.W = TM_CR1_OVR0_MDS_keep_w;                       // Timer channel 0 capture data buffer overrun mode select (Overwritten (Keep (Preserved old data)).
    TM20->CCMDS.W = TM_CCMDS_CC0_MDS_16bit_ic_w;                // Timer channel 0 capture and compare mode select (0x1 = 16bit_IC (Input capture)).
    APB->CR2.B[0] = APB_CR2_ITR7_MUX_trg8_b0;                   // APB ITR7 = ICKO_INT
    TM20->TRG.W = TM_TRG_ITR_MUX_itr7_w |                       // Trigger source ITR7
                  TM_TRG_TRG_MUX_in0_w |                        // Timer trigger source TRGI select IN0 (TM20_IN0).
                  TM_TRG_TRGI_MDS_trigger_f_w |                 // Timer trigger input mode select (Trigger-F (TRGI falling)).
                  TM_TRG_TRGI2_MDS_trigger_f_w;                 // 2nd Timer trigger input mode select (Trigger-F (TRGI falling)).
    TM20->ICCR.W = TM_ICCR_IC0_MUX_ic01_w |                     // Timer channel 0 input Mux select for input capture (TM36_ITR).
                   TM_ICCR_IC0_TRGS_falling_edge_w;             // Timer channel 0 input trigger edge select (Falling edge).
                   
    // Clear flag
    TM20->STA.W = TM_STA_CF0A_mask_w |                          //  Clear flag CFOA and CF0B
                  TM_STA_CF0B_mask_w;  
              
    // Enable TM36
    TM20->CR0.W |= TM_CR0_EN_enable_w |                         // Enable main Timer/Counter and 2nd Timer/Counter.
                   TM_CR0_EN2_enable_w;    
    
    // Wait flag CF0A and CF0B happened.
//    while((TM20->STA.W & (TM_STA_CF0A_mask_w | TM_STA_CF0B_mask_w)) != (TM_STA_CF0A_happened_w | TM_STA_CF0B_happened_w));
    while((TM20->STA.W & TM_STA_CF0A_mask_w) == TM_STA_CF0A_normal_w);
    
    TM20->CR0.W &= ~(TM_CR0_EN_enable_w |                       // Enable main Timer/Counter and 2nd Timer/Counter.
                     TM_CR0_EN2_enable_w);    
    
    return TM20->CC0A.H[0];


    // If not defined TM20 module 
    // Use TM36 to get the number of CK_APB clocks in one CK_LS cycle.
    #else

    CSC->KEY.H[0] = 0xA217;
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_main_b0;                 // Internal clock output source select (0x0 = CK_MAIN).
    CSC->CKS2.B[3] &= ~CSC_CKS2_TM36_CKS_mask_b3;               // TM36 process clock source select (0x0 = CK_APB).
    CSC->APB1.B[1] |= CSC_APB1_TM36_EN_enable_b1;               // TM36 module clock source enable.
    CSC->CKO.B[0] = CSC_CKO_CKO_DIV_div1_b0 |                   // Internal clock output divider (0x0 = DIV1 : divided by 1)
                    CSC_CKO_CKO_SEL_ck_ls_b0;                   // Internal clock output source select (0x4 = CK_LS).
    CSC->KEY.H[0] = 0x0000;

    // Deinitial
    TM36->CR0.W = 0x00000000;
    TM36->STA.W = 0xFFFFFFFF;
    TM36->INT.W = 0x00000000;
    TM36->CKO.W = 0x00000008;
    TM36->CNT.W = 0x00000000;
    TM36->PSCNT.W = 0x00000000;
    TM36->OSCR.W = 0x0000F0F0;
    TM36->OCCR0.W = 0x00000000;
    TM36->OCCR1.W = 0x00000000;
    TM36->PWM.W = 0x00000000;
    TM36->BS.W = 0x00000000;
    TM36->CC0A.W = 0x00000000;
    TM36->CC0B.W = 0x00000000;
    
    // Intial
    TM36->ARR.H[0] = 0xFFFF;                                    // Main timer/counter auto-reload value register (0xFFFF).
    TM36->PSARR.H[0] = 0xFFFF;                                  // Timer prescaler or 2nd timer/counter auto-reload value register (0xFFFF).
    TM36->CR0.W = TM_CR0_MDS_separate_w |                       // Timer operation mode select (Separate : Separated two 16-bit counters Mode).
                  TM_CR0_DIR_up_w;                              // Main Timer counting direction bit (Up (Up Counting)).
    TM36->CLK.W = TM_CLK_CKI_SEL_proc_w;                        // Timer input clock CK_TM36 source select (PROC : CK_TM36_PR process clock from CSC).
    TM36->CR1.W = TM_CR1_OVR0_MDS_keep_w;                       // Timer channel 0 capture data buffer overrun mode select (Overwritten (Keep (Preserved old data)).
    TM36->CCMDS.W = TM_CCMDS_CC0_MDS_16bit_ic_w;                // Timer channel 0 capture and compare mode select (0x1 = 16bit_IC (Input capture)).
    APB->CR2.B[0] = APB_CR2_ITR7_MUX_trg8_b0;                   // APB ITR7 = ICKO_INT
    TM36->TRG.W = TM_TRG_ITR_MUX_itr7_w |                       // Trigger source ITR7
                  TM_TRG_TRG_MUX_in0_w |                        // Timer trigger source TRGI select IN0 (TM36_IN0).
                  TM_TRG_TRGI_MDS_trigger_f_w |                 // Timer trigger input mode select (Trigger-F (TRGI falling)).
                  TM_TRG_TRGI2_MDS_trigger_f_w;                 // 2nd Timer trigger input mode select (Trigger-F (TRGI falling)).
                  
    TM36->ICCR.W = TM_ICCR_IC0_MUX_ic01_w |                     // Timer channel 0 input Mux select for input capture (TM36_ITR).
                   TM_ICCR_IC0_TRGS_falling_edge_w;             // Timer channel 0 input trigger edge select (Falling edge).
                   
    // Clear flag
    TM36->STA.W = TM_STA_CF0A_mask_w |                          //  Clear flag CFOA and CF0B
                  TM_STA_CF0B_mask_w;  
                  
    // Enable TM36
    TM36->CR0.W |= TM_CR0_EN_enable_w |                         // Enable main Timer/Counter and 2nd Timer/Counter.
                   TM_CR0_EN2_enable_w;    
    
    // Wait flag CF0A and CF0B happened.
//    while((TM36->STA.W & (TM_STA_CF0A_mask_w | TM_STA_CF0B_mask_w)) != (TM_STA_CF0A_happened_w | TM_STA_CF0B_happened_w));
    while((TM36->STA.W & TM_STA_CF0A_mask_w) == TM_STA_CF0A_normal_w);
    
    TM36->CR0.W &= ~(TM_CR0_EN_enable_w |                       // Enable main Timer/Counter and 2nd Timer/Counter.
                     TM_CR0_EN2_enable_w);  
    
    return TM36->CC0A.H[0];
    #endif
}


/**
 *******************************************************************************
 * @brief       
 * @details  
 * @param[in]    
 * @return      
 * @exception      
 * @note        
 *******************************************************************************
 */
IEC60730_Ret IEC60730_Clock_Test (void)
{
    uint8_t  FailFlag;                                                  // 0x01 IHRCO0 check fail
                                                                        // 0x02 IHRCO1 check fail
    uint32_t IHRCO0_Freq;
    uint32_t IHRCO1_Freq;

    
    FailFlag = 0;                                     
    CSC->KEY.H[0] = 0xA217;                                                                 // Unprotect CSC module
    
    if((CSC->CR0.B[1] & CSC_CR0_HS_SEL_mask_b1) != CSC_CR0_HS_SEL_ihrco_b1)                 // When CK_HS not IHRCOx 
    {
        if((CSC->CR0.B[0] & CSC_CR0_IHRCO_EN_mask_b0) == CSC_CR0_IHRCO_EN_disable_b0)       // When IHRCOx not enable  
        {
            CSC->CR0.B[0] |= CSC_CR0_IHRCO_EN_enable_b0;                                    // Enable IHRCOx
            while((CSC->STA.B[0] & CSC_STA_IHRCOF_mask_b0) == CSC_STA_IHRCOF_normal_b0);
            CSC->STA.B[0] = CSC_STA_IHRCOF_mask_b0;
        }
        CSC->CR0.B[1] &= ~CSC_CR0_HS_SEL_mask_b1;                                           // CK_HS = IHRCOx
    }
     
    CSC->CR0.B[2] &= ~CSC_CR0_IHRCO_SEL_mask_b2;                        // IHRCO = 12MHz
    CSC->CR0.B[1] &= ~CSC_CR0_MAIN_SEL_mask_b1;                         // CK_MAIN = CK_HS
    CSC->DIV.W &= ~(CSC_DIV_APB_DIV_mask_w |                            // CK_APB = CK_MAIN /1
                    CSC_DIV_AHB_DIV_mask_w);                            // CK_AHB = CK_APB /1
    CSC->KEY.H[0] = 0x0000;                                             // Protect CSC module

    /* Check IHRCO0 frequency */
    IHRCO0_Freq = Get_CK_MAIN_Count_In_CK_LS();                         // Get 1 CK_LS cycle count (CK_MAIN) 
    // Check IHRCO0 is in range
    if(IHRCO0_Freq > IHRCO0_MaxLimit)                                   // When ILRCO -40%
        FailFlag |= 0x01;
    if(IHRCO0_Freq < IHRCO0_MinLimit)                                   // When ILRCO +40%
        FailFlag |= 0x01;
    
    /* Check IHRCO1 frequcncy */
    CSC->KEY.H[0] = 0xA217;                                             // Unprotect CSC module
    CSC->CR0.B[2] |= CSC_CR0_IHRCO_SEL_11_b2;                           // IHRCO = 11M0592Hz
    CSC->KEY.H[0] = 0x0000;                                             // Protect CSC module
    
    IHRCO1_Freq = Get_CK_MAIN_Count_In_CK_LS();                         // Get 1 CK_LS cycle count (CK_MAIN) 

    CSC->KEY.H[0] = 0xA217;                                             // Unprotect CSC module
    CSC->CR0.B[2] &= ~CSC_CR0_IHRCO_SEL_mask_b2;                        // IHRCO = 12MHz
    CSC->KEY.H[0] = 0x0000;                                             // Protect CSC module
    
    // Check IHRCO1 is in range
    if(IHRCO1_Freq > IHRCO1_MaxLimit)                                   // When ILRCO -40%
        FailFlag |= 0x02;
    if(IHRCO1_Freq < IHRCO1_MinLimit)                                   // When ILRCO +40%
        FailFlag |= 0x02;

    // Check IHRCO0 <= IHRCO1
    if(IHRCO0_Freq <= IHRCO1_Freq)
        FailFlag |= 0x04;
    
    if(FailFlag == 0)                                                   // When IHRCOx or ILRCO success
        return IEC60730_TEST_SUCCESS;
    else                                                                // When IHRCOx or ILRCO failure
        return IEC60730_TEST_FAILURE;                                   
}







