



/**
 ******************************************************************************
 *
 * @file        BSP_SPIFlash.c
 * @brief       This is SPI Flash (MX25LXX) C file.
 
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2022/05/23
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/*==============================================================================
                                 User Notes
How To use this function:
-----------------------
   1. Use "BSP_SPIFlash_Init" function to inital.
   2. Use "BSP_SPIFlash_ReadFlashRDID" functio to read SPI Flash RDID.
   3. Use "BSP_SPIFlash_Erase" function to erase SPI Flash.
   4. Use "BSP_SPIFlash_Read" function to read SPI Flash data.
   5. Use "BPS_SPIFlash_Program" function to program SPI Flash.

Driver architecture:
--------------------
   + MG32_GPIO_DRV
   + MG32_SPI_DRV
   
Known Limitations:
------------------

Require parameter
------------------
    Require module : CSC / GPIO / SPI0
    
    GPIO pin configuration : 
        Pin / IO mode / AFS
        ---  --------  -----
        PD0 / PPO     / GPIO
        PD1 / PPO     / SPI0_CLK
        PD2 / PPO     / SPI0_MOSI
        PD7 / DIN     / SPI0_MISO
        PD8 / PPO     / SPI0_D2
        PD3 / PPO     / SPI0_D3
        
    SPI Module : 
        Mode              : Standard or 4 data line bidirectional mode.
		Baud-rate         : 3MHz in ( SPI0 clock source is 48MHz)
        
Example codes:
------------------

==============================================================================*/

/* Includes ------------------------------------------------------------------*/
#include "BSP_13_SPIFlash.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO
#define SPIFLASH_CEN_PIN            PX_Pin_0
#define SPIFLASH_SDK_PIN            PX_Pin_1
#define SPIFLASH_DA0_PIN            PX_Pin_2
#define SPIFLASH_DA1_PIN            PX_Pin_7
#define SPIFLASH_DA2_PIN            PX_Pin_8
#define SPIFALSH_DA3_PIN            PX_Pin_3
								    
#define SPIFLASH_CEN_PIN_MODE       PINX_Mode_PushPull_O
#define SPIFLASH_SDK_PIN_MODE       PINX_Mode_PushPull_O
#define SPIFLASH_DA0_PIN_MODE       PINX_Mode_PushPull_O
#define SPIFLASH_DA1_PIN_MODE       PINX_Mode_Digital_I
#define SPIFLASH_DA2_PIN_MODE       PINX_Mode_PushPull_O
#define SPIFLASH_DA3_PIN_MODE       PINX_Mode_PushPull_O
								    
#define SPIFALSH_CEN_PIN_AFS        0
#define SPIFALSH_SDK_PIN_AFS        7
#define SPIFALSH_DA0_PIN_AFS        7
#define SPIFALSH_DA1_PIN_AFS        5
#define SPIFALSH_DA2_PIN_AFS        5
#define SPIFALSH_DA3_PIN_AFS        7

#define SPIFALSH_IOM                IOMD 

#define Flash_nCS                   PD0             /*!< Define software nCS Pin.*/

//SPI
#define Flash_SPI                   SPI0            /*!< Use the SPI = to control SPI flash.*/

//Flash 
#define FLASH_nCS_ACTIVE            0               /*!< Define polarity (nCS = Active).*/
#define FLASH_nCS_NoACTIVE          1               /*!< Define polarity (nCS = No Active).*/ 
								    
#define FLASH_MAXSIZE               64*1024         /*!< Flash max size.*/
#define FLASH_PROGRAM_MAX           256             /*!< Define Flash program max bytes  in one write enable.*/ 
								    
#define Flash_Dummy_Data            0xFFFFFFFF      /*!< Define DUMMY Data.*/

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
FlashCTR_TypeDef SPIFlash_CTR;

static uint8_t BSP_SPIFlash_QE;

/* Private function prototypes -----------------------------------------------*/
static uint8_t BSP_SPIFlash_SPIGetData(void);
static void BSP_SPIFlash_SPISetData(uint8_t SPI_DataLen , uint32_t SPI_Data);

static void BPS_SPIFlash_WriteFlash_Cmd (FunctionalState WREN);
static void BSP_SPIFlash_CheckFlashBusy(void);
static DRV_Return BSP_SPIFlash_QuadEnable_Cmd (FunctionalState QE_Cmd);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief	    SPI Flash inital.
 * @details     
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void BSP_SPIFlash_Init(void) 
{ 
    PIN_InitTypeDef  SPIFlash_Pin;
    
    /*GPIO init.*/
    SPIFlash_Pin.PINX_Pin                = ( SPIFLASH_SDK_PIN | SPIFLASH_DA0_PIN );
    SPIFlash_Pin.PINX_Mode               = SPIFLASH_SDK_PIN_MODE;
    SPIFlash_Pin.PINX_PUResistant        = PINX_PUResistant_Enable;
    SPIFlash_Pin.PINX_Speed              = PINX_Speed_High;
    SPIFlash_Pin.PINX_Inverse            = PINX_Inverse_Disable;
    SPIFlash_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    SPIFlash_Pin.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    SPIFlash_Pin.PINX_Alternate_Function = SPIFALSH_SDK_PIN_AFS;                            
    GPIO_PortMode_Config( SPIFALSH_IOM ,&SPIFlash_Pin);
    
    SPIFlash_Pin.PINX_Pin                = SPIFALSH_DA3_PIN;
    SPIFlash_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level3;
    GPIO_PortMode_Config( SPIFALSH_IOM ,&SPIFlash_Pin);
    
    SPIFlash_Pin.PINX_Pin                = SPIFLASH_CEN_PIN;
    SPIFlash_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    SPIFlash_Pin.PINX_Alternate_Function = SPIFALSH_CEN_PIN_AFS;  
    GPIO_PortMode_Config( SPIFALSH_IOM ,&SPIFlash_Pin);
    
    SPIFlash_Pin.PINX_Pin                = SPIFLASH_DA2_PIN;
    SPIFlash_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level3;
    SPIFlash_Pin.PINX_Alternate_Function = SPIFALSH_DA2_PIN_AFS;  
    GPIO_PortMode_Config( SPIFALSH_IOM ,&SPIFlash_Pin);
    
    SPIFlash_Pin.PINX_Pin                = SPIFLASH_DA1_PIN;
    SPIFlash_Pin.PINX_Mode               = SPIFLASH_DA1_PIN_MODE;
    SPIFlash_Pin.PINX_Alternate_Function = SPIFALSH_DA1_PIN_AFS;  
    GPIO_PortMode_Config( SPIFALSH_IOM ,&SPIFlash_Pin);

    /*SPI inital.*/
    SPI_DeInit(Flash_SPI);
    
    //Set BaudRate
    SPI_Clock_Select(Flash_SPI, SPI_CK_SPIx_PR);        // SPI clock source 
    SPI_PreDivider_Select(Flash_SPI, SPI_PDIV_1);       // SPI PDIV /1
    SPI_Prescaler_Select(Flash_SPI, SPI_PSC_1);         // SPI PSC /1
    SPI_Divider_Select(Flash_SPI, SPI_DIV_16);          // SPI DIV /16

    //Set Mode Select
    SPI_ModeAndNss_Select(Flash_SPI, SPI_Master);       // SPI master mode without NSS

    //Set DataLine Select
    SPI_DataLine_Select(Flash_SPI, SPI_Standard);       // Srandard SPI
    
    //Set SPI Mode = (CPHA = 0 , CPOL = 0)
    SPI_ClockPhase_Select(Flash_SPI, SPI_LeadingEdge);  // Leading edge
    SPI_ClockPolarity_Select(Flash_SPI, SPI_Low);       // Clock idle low

    Flash_nCS = FLASH_nCS_NoACTIVE;                     // nCS no active
    
    //Set Data Control
    SPI_FirstBit_Select(Flash_SPI, SPI_MSB);            // First bit MSB
    SPI_DataSize_Select(Flash_SPI, SPI_8bits);          // Data size 8bit

    //Enable URT
    SPI_Cmd(Flash_SPI, ENABLE);
	
    /*Parameter Inital.*/
    BSP_SPIFlash_QE = 0;
    
    /*Flash control inital*/
    BSP_SPIFlash_QuadEnable_Cmd(DISABLE);
}
/**
 *******************************************************************************
 * @brief	    Read SPI Flash RDID
 * @details     
 * @return      Flash RDID number
 * @exception   No
 * @note        No
 *******************************************************************************
 */
uint32_t BSP_SPIFlash_ReadFlashRDID (void)
{
    ctype RDAT;
    
    Flash_nCS = FLASH_nCS_ACTIVE;     
    
    BSP_SPIFlash_SPISetData(1,FLASH_READ_IDENTIFICATION);
    RDAT.B[0] = BSP_SPIFlash_SPIGetData();
    RDAT.B[1] = BSP_SPIFlash_SPIGetData();
    RDAT.B[2] = BSP_SPIFlash_SPIGetData();
    RDAT.B[3] = 0;
    
    Flash_nCS = FLASH_nCS_NoACTIVE;

    return(RDAT.W);
}
/**
 *******************************************************************************
 * @brief	    Read configuration register
 * @details     
 * @return      Configuration register value.
 * @exception   No
 * @note        No
 *******************************************************************************
 */
uint16_t BSP_SPIFlash_ReadFlashRDCR (void)
{
    ctype RDAT;
    
    Flash_nCS = FLASH_nCS_ACTIVE;
    
    BSP_SPIFlash_SPISetData(1,FLASH_READ_CONFIGURATION);
    RDAT.B[0] = BSP_SPIFlash_SPIGetData();
    RDAT.B[1] = BSP_SPIFlash_SPIGetData();
    
    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    return(RDAT.H[0]);
}
/**
 *******************************************************************************
 * @brief	   Erase SPI Flash 
 * @details    Use polling to 
 * @param[in]  EraseType:
 *   @arg\b    Flash_Erase_Chip    : Flash chip erase.
 *   @arg\b    Flash_Erase_Block64K: Flash block(64K) erase.
 *   @arg\b    Flash_Erase_Block32K: Flash block(32K) erase.
 *   @arg\b    Flash_Erase_Sector  : Flash sector erase.
 * @param[in]  Page_Num: Page number for erase type unit.
 * @return      
 * @exception  No
 * @note       No
 *******************************************************************************
 */
void BSP_SPIFlash_Erase(Flash_EraseypeDef EraseType , uint32_t Page_Num)
{
    ctype sFS_Erase_AddressIndex;

    switch(EraseType)
    {
        case Flash_Erase_Chip:
                                  break;
        case Flash_Erase_Block64K:
                                  sFS_Erase_AddressIndex.W = 0x10000 * Page_Num;
                                  break; 
        
        case Flash_Erase_Block32K:
                                  sFS_Erase_AddressIndex.W = 0x8000  * Page_Num;
                                  break;
        
        case Flash_Erase_Sector:
                                  sFS_Erase_AddressIndex.W = 0x1000  * Page_Num;
                                  break;    
    }
    
    /*Check erase size whether over flash size.*/
    if(sFS_Erase_AddressIndex.W > FLASH_MAXSIZE)
    {
        return;
    }
    BPS_SPIFlash_WriteFlash_Cmd(ENABLE);
    
    Flash_nCS = FLASH_nCS_ACTIVE;
    
    BSP_SPIFlash_SPISetData(1,EraseType);
    
    if(EraseType!=Flash_Erase_Chip)
    {
        BSP_SPIFlash_SPISetData(1,sFS_Erase_AddressIndex.B[2]);
        BSP_SPIFlash_SPISetData(1,sFS_Erase_AddressIndex.B[1]);
        BSP_SPIFlash_SPISetData(1,sFS_Erase_AddressIndex.B[0]);
    }
    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    /*Check erase whether complete.*/
    BSP_SPIFlash_CheckFlashBusy();
}
/**
 *******************************************************************************
 * @brief	    Read SPI flash
 * @details     Use polling to read SPI Flash.
 * @param[in]   rFlash
 *  @arg\b      Address     : Flash address.
 *  @arg\b      Total_Length: Read total byte.
 *  @arg\b      Buffer      : Data store destination memory.
 * @param[in]   rFlash_Mode
 *  @arg\b      Flash_Read    : Read data bytes at higher speed.
 *  @arg\b      Flash_QuadRead: Quad read mode.
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void BSP_SPIFlash_Read(FlashCTR_TypeDef rFlash, Flash_ReadTypeDef rFlash_Mode)
{
    uint32_t  FS_ReadTmp;
    ctype     FS_Read_Address;
    

    /*Quad whether enable or not.*/
    switch( rFlash_Mode)
    {
        case Flash_4IO_Read:
                            if(BSP_SPIFlash_QE==0)
                            {
                                //Quad Enable
                                BSP_SPIFlash_QuadEnable_Cmd(ENABLE);
                            }
                            break;
        case Flash_Read:
                            break;
    }
    
    /*Set Command + Flash Address*/
    Flash_nCS = FLASH_nCS_ACTIVE;
    FS_Read_Address.W = rFlash.Address;
    
    //Flash command.(1 line)
    BSP_SPIFlash_SPISetData(1, rFlash_Mode);
    
    /*Set PIN line*/
    if( rFlash_Mode == Flash_4IO_Read)
    {
        //Change SPI module Data is 4 lines.
        SPI_DataLine_Select(Flash_SPI, SPI_4LinesBidirection);
        SPI_BidirectionalOutput_Cmd(Flash_SPI, ENABLE);
    }
    //Flash address (1 line)
    BSP_SPIFlash_SPISetData(1, FS_Read_Address.B[2]);
    BSP_SPIFlash_SPISetData(1, FS_Read_Address.B[1]);
    BSP_SPIFlash_SPISetData(1, FS_Read_Address.B[0]);
    
    
    if( rFlash_Mode == Flash_4IO_Read)
    {
        BSP_SPIFlash_SPISetData(3, Flash_Dummy_Data);
        SPI_BidirectionalOutput_Cmd(Flash_SPI, DISABLE);
    }
    else
    {
        BSP_SPIFlash_SPISetData(1, Flash_Dummy_Data);
    }
    
    /*Read Data*/
    for(FS_ReadTmp=0;FS_ReadTmp<rFlash.Total_Length;FS_ReadTmp++)
    {
        rFlash.Buffer[FS_ReadTmp] = BSP_SPIFlash_SPIGetData();
    }

    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    SPI_DataLine_Select(Flash_SPI, SPI_Standard);
}
/**
 *******************************************************************************
 * @brief	    Program SPI Flash
 * @details     Use polling to program SPI Flash.
 * @param[in]   pFlash
 *  @arg\b      Address     : Flash address.
 *  @arg\b      Total_Length: program total byte.
 *  @arg\b      Buffer      : Data source pointer.
 * @param[in]   pFlash_Mode
 *  @arg\b      Flash_PageProgram    : The page program.
 *  @arg\b      Flash_4IO_PageProgram: 4 x IO page program.
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void BPS_SPIFlash_Program(FlashCTR_TypeDef pFlash, Flash_ProgramTypeDef pFlash_Mode)
{
    uint32_t sFS_ProgramTmp;
    uint32_t sFS_Program_TotalSize;
    uint32_t sFS_Program_BLen;
    uint32_t sFS_Program_Index;
    ctype    sFS_Program_Address;
    
    sFS_Program_Address.W = pFlash.Address;
    sFS_Program_TotalSize = pFlash.Total_Length;
    sFS_Program_Index     = 0;
    

    switch( pFlash_Mode)
    {
        case Flash_4IO_PageProgram:
                                   if(BSP_SPIFlash_QE==0)
                                   {
                                       BSP_SPIFlash_QuadEnable_Cmd(ENABLE);
                                   }
                                   break;
        case Flash_PageProgram:
                                   break;
    }
    
    do
    {
        /*Enable write*/
        BPS_SPIFlash_WriteFlash_Cmd(ENABLE);

        if(sFS_Program_TotalSize > FLASH_PROGRAM_MAX)
        {
            sFS_Program_BLen = FLASH_PROGRAM_MAX;
        }
        else
        {
            sFS_Program_BLen = sFS_Program_TotalSize;
        }

        Flash_nCS = FLASH_nCS_ACTIVE;

        /*Program command*/
        BSP_SPIFlash_SPISetData(1, pFlash_Mode);
        
        /*Set PIN line*/
        if( pFlash_Mode == Flash_4IO_PageProgram)
        {
            SPI_DataLine_Select(Flash_SPI, SPI_4LinesBidirection);
            SPI_BidirectionalOutput_Cmd(Flash_SPI, ENABLE);
        }
        /*Program Addresss*/
        BSP_SPIFlash_SPISetData(1,sFS_Program_Address.B[2]);
        BSP_SPIFlash_SPISetData(1,sFS_Program_Address.B[1]);
        BSP_SPIFlash_SPISetData(1,sFS_Program_Address.B[0]);
        
        /*Data0 ~ DataN*/
        for(sFS_ProgramTmp = 0; sFS_ProgramTmp < sFS_Program_BLen; sFS_ProgramTmp++)
        {
            BSP_SPIFlash_SPISetData(1,pFlash.Buffer[sFS_Program_Index]);
            sFS_Program_Index = sFS_Program_Index + 1;
        }
        
        sFS_Program_Address.W = sFS_Program_Address.W + sFS_Program_BLen;
        sFS_Program_TotalSize = sFS_Program_TotalSize - sFS_Program_BLen;
        
        Flash_nCS = FLASH_nCS_NoACTIVE;
        
        SPI_DataLine_Select(Flash_SPI, SPI_Standard);
        SPI_BidirectionalOutput_Cmd(Flash_SPI, DISABLE);
        
        
        BSP_SPIFlash_CheckFlashBusy(); 
    }while(sFS_Program_TotalSize!=0);
}
/**
 *******************************************************************************
 * @brief	    SPI set data.
 * @details     SPI use polling to set data.
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
static void BSP_SPIFlash_SPISetData(uint8_t SPI_DataLen , uint32_t SPI_Data)
{
    SPI_SetTxData(Flash_SPI, (Byte_Enum) (SPI_DataLen - 1), SPI_Data);
    while((SPI_GetAllFlagStatus(Flash_SPI) & SPI_INT_TC) != SPI_INT_TC);
    SPI_ClearFlag(Flash_SPI, (SPI_ALLF));
    SPI_ClearRxData(Flash_SPI);
}
/**
 *******************************************************************************
 * @brief	    SPI read data.
 * @details     SPI use polling to read data.
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
static uint8_t BSP_SPIFlash_SPIGetData(void)
{
    SPI_SetTxData(Flash_SPI, SPI_1Byte, Flash_Dummy_Data);
    while((SPI_GetAllFlagStatus(Flash_SPI) & SPI_INT_RX) != SPI_INT_RX);
    return((uint8_t)SPI_GetRxData(Flash_SPI));
}
/**
 *******************************************************************************
 * @brief	    Control Flash Write enable. 
 * @details     
 * @param[in]   WREN: Flash write control.
 *   @arg\b     ENABLE  : Flash write enable.
 *   @arg\b     DISABLE : Flash write disable.
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
static void BPS_SPIFlash_WriteFlash_Cmd(FunctionalState WREN)
{
    uint32_t RDAT;
    
    do
    {
        /* Write enable */
        Flash_nCS = FLASH_nCS_ACTIVE;
        
        if(WREN==ENABLE)
        {
            BSP_SPIFlash_SPISetData(1, FLASH_WRITE_ENABLE);
            Flash_nCS = FLASH_nCS_NoACTIVE;
            
            /* Read status */
            Flash_nCS = FLASH_nCS_ACTIVE;
            BSP_SPIFlash_SPISetData(1,FLASH_READ_STATUS);
            
            RDAT = BSP_SPIFlash_SPIGetData();
            Flash_nCS = FLASH_nCS_NoACTIVE;
        }
        else
        {
            BSP_SPIFlash_SPISetData(1,FLASH_WRITE_DISABLE);
            Flash_nCS = FLASH_nCS_NoACTIVE;
            return;
        }
        
    /* Check WEL == 1 */
    }while((RDAT & 0x02) == 0x00);
}
/**
 *******************************************************************************
 * @brief	    Check SPI flash if is busy or not.
 * @details     
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
static void BSP_SPIFlash_CheckFlashBusy(void)
{
    uint8_t RDAT;
    
    //============================================
    /* Read status register */
    Flash_nCS = FLASH_nCS_ACTIVE;
    
    BSP_SPIFlash_SPISetData(1,FLASH_READ_STATUS);

    //===========================================
    /* Check erase or write complete */
    /* Get State */
    do
    {
        RDAT = BSP_SPIFlash_SPIGetData();
    }while((RDAT & 0x01)==0x01);
    
    Flash_nCS = FLASH_nCS_NoACTIVE; 
}
/**
 *******************************************************************************
 * @brief	    Quad enable or disable
 * @details     
 * @param[in]   QE_Cmd : 
 *   @arg\b     DISABLE : Disable Quad.
 *   @arg\b     ENABLE  : Enable Quad.
 * @return      Return Quad Enable or disable whether success or not. 
 * @exception   No
 * @note        No
 *******************************************************************************
 */
static DRV_Return BSP_SPIFlash_QuadEnable_Cmd (FunctionalState QE_Cmd)
{
    DRV_Return State = DRV_Success;
    uint16_t   REG;
    ctype      RDAT;
    uint8_t    SPIFlash_QuadEnableTmp;
    uint8_t    QEEnableTmp;
    
    /* Enabl */
    BPS_SPIFlash_WriteFlash_Cmd(ENABLE);
    
    /* Read status */
    Flash_nCS = FLASH_nCS_ACTIVE;
    BSP_SPIFlash_SPISetData(1,FLASH_READ_STATUS);
    RDAT.B[0] = BSP_SPIFlash_SPIGetData();
    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    /* Read Configuration Register */
    REG = BSP_SPIFlash_ReadFlashRDCR();
    RDAT.B[1] = (uint8_t) REG;
    RDAT.B[2] = (uint8_t) (REG >> 8);
    
    /* Setting Status Register QE */
    if(QE_Cmd == DISABLE)
    {
        RDAT.B[0] &= ~0x40;
        QEEnableTmp = 0;
    }
    else
    {
        RDAT.B[0] |= 0x40;
        QEEnableTmp = 1;
    }
    /* Write WDSR */
    Flash_nCS = FLASH_nCS_ACTIVE;
    BSP_SPIFlash_SPISetData(1, FLASH_WRITE_STATUS);
    BSP_SPIFlash_SPISetData(3, RDAT.W);
    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    /* Wait WIP == 0 */
    do
    {
        Flash_nCS = FLASH_nCS_ACTIVE;
        BSP_SPIFlash_SPISetData(1,FLASH_READ_STATUS);
        RDAT.B[0] = BSP_SPIFlash_SPIGetData();
        Flash_nCS = FLASH_nCS_NoACTIVE;
    }while((RDAT.B[0] & 0x01) != 0);
    
    Flash_nCS = FLASH_nCS_ACTIVE;
    BSP_SPIFlash_SPISetData(1,FLASH_READ_STATUS);
    RDAT.B[0] = BSP_SPIFlash_SPIGetData();
    Flash_nCS = FLASH_nCS_NoACTIVE;
    
    // When get status fail
    SPIFlash_QuadEnableTmp = (RDAT.B[0] & 0x40);
    
    if(((SPIFlash_QuadEnableTmp == 0) & (QE_Cmd == ENABLE)) ||
       ((SPIFlash_QuadEnableTmp != 0) & (QE_Cmd == DISABLE)))
    {
        State = DRV_Failure;
    }
    else
    {
        BSP_SPIFlash_QE = QEEnableTmp;
    }
    
    return State;
}



