
/**
 ******************************************************************************
 *
 * @file        BSP_Button.c
 * @brief       MG32 button board support c code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2022/07/05
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
 /*==============================================================================
                                 User NOTES
How To use this function:
-----------------------
   + BSP_Button_Init(PortB, (PIN9|{IN10));
   + variable = BSP_Button_GetState(PortB, (PIN9|{IN10));
   
   Button GPIO initialize and get GPIO input state.
   
Driver architecture:
--------------------
   + uint16_t BSP_Button_Init(uint8_t GPIOx, uint16_t BSP_SET_PIN);
   + uint16_t BSP_Button_GetState(uint8_t GPIOx, uint16_t BSP_SET_PIN)
   + uint8_t BSP_Button_Uninit(uint8_t GPIOx, uint8_t PIN_NUM)
   + uint16_t BSP_Button_GetCount(void)                                     // User set by self.
Known Limitations:
------------------
   1- Run in 12MHz. (CSC control).
   2- GPIO PAx, PBx, PCx, PDx, PEx must be DIN mode Pull up enable.
   3- 
   4- 
==============================================================================*/

/*==============================================================================
                                 Require parameter
Require module : CSC / GPIO /

DAC Module : 
    Clock source                : CK_APB 
    Clock frequency             : 12MHz
                                
CSC Module :
    Clock source                : CK_IHRCO (12M Hz)
    Clock frequency             : CK_IHRCO
    CK_APB frequency            : 12MHz
    CK_AHB frequency            : 12MHz
    Clock enable in 'On Mode'   : GPIOA / GPIOB / GPIOC / GPIOD / GPIOE(MG32_1ST | MG32_3RD) /

GPIO pin configuration : 
    Pin / IO mode / AFS
    ---  --------  -----
    PAx / DIN     / GPIO (Default)
    PBx / DIN     / GPIO (Default)
    PCx / DIN     / GPIO (Default)
    PDx / DIN     / GPIO (Default)
    PEx / DIN     / GPIO (Default) for (MG32F02A132 | MG32F02U128 | MG32F02A128) LQFP80
           
    
==============================================================================*/
 
/* Includes ------------------------------------------------------------------*/
#include "MG32_GPIO_DRV.h"
#include "BSP_Button.h"
/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define BSP_BUTTON_NUM 3U                   // Available buttons define(by user self).
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/
 
/**
 *******************************************************************************
 * @brief           BSP_Button_Init
 * @details         Board support button initialize
 * @param[in]       
 * @arg\b           
 * @return
 * @note
 * @par             Example
 * @code
                    if(BSP_Button_GPIO_Init(PortA, (PIN0|PIN1|PIN3))!= DRV_Success)
                    {
                        // BSP Button GPIO initialize failure
                        while(0);                               // to do ...
                    }
 * @endcode
 *******************************************************************************
 */
uint16_t BSP_Button_Init(uint8_t GPIOx, uint16_t BSP_SET_PIN)
{
    uint8_t PIN_CNT = 0x00;
    uint16_t BSP_UnInitPin = 0x0001;
    
    while(BSP_SET_PIN != 0x0000)
    {
        if((BSP_SET_PIN&BSP_UnInitPin) != NONE)
        {
            BSP_Button_SetGPIO(GPIOx, PIN_CNT, GPIO_DIN);
        }
        BSP_SET_PIN &= ~BSP_UnInitPin;
        PIN_CNT = PIN_CNT+1;
        BSP_UnInitPin = (uint16_t)(BSP_UnInitPin<<1);
        
    }    
    
    return DRV_Success;
}
 
/**
 *******************************************************************************
 * @brief           BSP_Button_GetState
 * @details         Board support get button state function
 * @param[in]       
 * @arg\b           
 * @return
 * @note
 * @par             Example
 * @code
                    uint16_t Button_val;
                    
                    Button_val = BSP_Button_GPIO_Init(PortA, (PIN0|PIN1|PIN3));
 * @endcode
 *******************************************************************************
 */
uint16_t BSP_Button_GetState(uint8_t GPIOx, uint16_t BSP_SET_PIN)
{
    uint16_t BSP_State_Read = 0x0000U;
    uint16_t BSP_UnReadPin = 0x0001U;
    
    switch(GPIOx)
    {
        case PortA:
                    BSP_State_Read = GPIOA->IN.H[0];
                    break;
        case PortB:
                    BSP_State_Read = GPIOB->IN.H[0];
                    break;
        case PortC:
                    BSP_State_Read = GPIOC->IN.H[0];
                    break;
        case PortD:
                    BSP_State_Read = GPIOD->IN.H[0];
                    break;
        case PortE:
                    #if defined(MG32_1ST)||defined(MG32_3RD)
                        BSP_State_Read = GPIOE->IN.H[0];
                    #endif
                    break;
    }
    
    while(BSP_SET_PIN!=NONE)
    {
        if((BSP_SET_PIN&BSP_UnReadPin)==NONE) 
            BSP_State_Read &= ~BSP_UnReadPin;
        
        BSP_SET_PIN   &= ~BSP_UnReadPin;
        BSP_UnReadPin = (uint16_t)(BSP_UnReadPin<<1);
    }
    
    return (uint16_t)BSP_State_Read;
}

/**
 *******************************************************************************
 * @brief           BSP_Button_SetGPIO
 * @details         Board support buttons GPIO initialize.
 * @param[in]       
 * @arg\b           
 * @return
 * @note
 * @par             Example
 * @code
                    
                    if(BSP_Button_Init(PortA, 0, GPIO_DIN) != DRV_Success)
                    {
                        // Initialize Failure
                        while(0);               // to do ...
                    }

 * @endcode
 *******************************************************************************
 */
uint8_t BSP_Button_SetGPIO(uint8_t GPIOx, uint8_t PIN_NUM, uint8_t IO_Mode)
{
    switch(GPIOx)
    {
        case PortA:
                            PINA(PIN_NUM)->CR.MBIT.PU = 1;
                            PINA(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PINA(PIN_NUM)->CR.MBIT.IOM |= IO_Mode;
                            break;
        case PortB:
                            PINB(PIN_NUM)->CR.MBIT.PU = 1;
                            PINB(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PINB(PIN_NUM)->CR.MBIT.IOM |= IO_Mode;
                            break;
        case PortC:
                            PINC(PIN_NUM)->CR.MBIT.PU = 1;
                            PINC(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PINC(PIN_NUM)->CR.MBIT.IOM |= IO_Mode;
                            break;
        case PortD:
                            PIND(PIN_NUM)->CR.MBIT.PU = 1;
                            PIND(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PIND(PIN_NUM)->CR.MBIT.IOM |= IO_Mode;
                            break;
        case PortE:
                        #if defined(MG32_1ST)||defined(MG32_3RD)
                            PINE(PIN_NUM)->CR.MBIT.PU = 1;
                            PINE(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PINE(PIN_NUM)->CR.MBIT.IOM |= IO_Mode;
                        #endif
                        break;
    }
    
    return DRV_Success;
}

/**
 *******************************************************************************
 * @brief           BSP_Button_Uninit
 * @details         Board support buttons GPIO initialize to default.
 * @param[in]       
 * @arg\b           
 * @return
 * @note
 * @par             Example
 * @code
                    
                    if(BSP_Button_Uninit(PortA, 0, GPIO_DIN) != DRV_Success)
                    {
                        // Initialize Failure
                        while(0);                   // to do ...
                    }

 * @endcode
 *******************************************************************************
 */
uint8_t BSP_Button_Uninit(uint8_t GPIOx, uint8_t PIN_NUM)
{
    switch(GPIOx)
    {
        case PortA:
                            PINA(PIN_NUM)->CR.MBIT.PU = 1;
                            PINA(PIN_NUM)->CR.MBIT.IOM &= ~GPIO_MSK;
                            PINA(PIN_NUM)->CR.MBIT.IOM |= GPIO_AIN;
                            break;
        case PortB:
                            PINB(PIN_NUM)->CR.MBIT.PU = 1;
                            PINB(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PINB(PIN_NUM)->CR.MBIT.IOM |= GPIO_AIN;
                            break;
        case PortC:
                            PINC(PIN_NUM)->CR.MBIT.PU = 1;
                            PINC(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PINC(PIN_NUM)->CR.MBIT.IOM |= GPIO_AIN;
                            break;
        case PortD:
                            PIND(PIN_NUM)->CR.MBIT.PU = 1;
                            PIND(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PIND(PIN_NUM)->CR.MBIT.IOM |= GPIO_AIN;
                            break;
        case PortE:
                        #if defined(MG32_1ST)||defined(MG32_3RD)
                            PINE(PIN_NUM)->CR.MBIT.PU = 1;
                            PINE(PIN_NUM)->CR.B[0] &= ~GPIO_MSK;
                            PINE(PIN_NUM)->CR.MBIT.IOM |= GPIO_AIN;
                        #endif
                            break;
                        
    }
    
    
    return DRV_Success;
}

/**
 *******************************************************************************
 * @brief           BSP_Button_GetCount
 * @details         Get number of available buttons
 * @param[in]       
 * @arg\b           
 * @return
 * @note
 * @par             Example
 * @code
                    uint8_t Buttons;
                    
                    Buttons = BSP_Button_GetCount();

 * @endcode
 *******************************************************************************
 */
uint16_t BSP_Button_GetCount(void)
{
    return BSP_BUTTON_NUM;
}

