/**
  ******************************************************************************
 *
 * @file        Sample_RST_DetectResetEvent.c
 *
 * @brief       RST module reset sample code.
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2023/03/30
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2019 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 

/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Sample_RST_DetectEvent (void);
void Sample_URT0_Init(void);
void printf_mg( const char *format,...);
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief	   Sample URT0 initial
 * @details     1. Set CSC initial.
 *     \n       2. Set GPIO initial.
 *     \n       3. Set RX/TX Baudrate.
 *     \n       4. Set data character.
 *     \n       5. Set URT0 mode.
 *     \n       6. Set Data line.
 *     \n       7. Set Data buffer control.
 *     \n       8. Enable URT0 interrupt.
 *     \n       9. Enable URT0 
 *     \n      10. Receive data into interrupt function.
 *     \n      11. Send data of receive.
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void Sample_URT0_Init(void)
{
    URT_BRG_TypeDef  URT_BRG;
    URT_Data_TypeDef DataDef;
    PIN_InitTypeDef PINX_InitStruct;
 
    
    
    //==Set CSC init
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 11.0592M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->Port B/URT0 = Enable
    //Configure Peripheral On Mode Clock->URT0->Select URT0_PR Source = CK_APB(11.0592)
    
    //==Set GPIO init
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin8/9
    //GPIO port initial is 0xFFFF
    //Pin8 mode is PPO/Pin9 mode is ODO
    //Pin8/9 pull-up resister Enable
    //Pin8/9 function URT0_TX/RX
    
    PINX_InitStruct.PINX_Mode				= PINX_Mode_PushPull_O;         // Setting PB8 URT0_TX
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Disable;
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;
    PINX_InitStruct.PINX_Alternate_Function = 3;
    GPIO_PinMode_Config(PINB(8),&PINX_InitStruct);
    
    
    //=====Set Clock=====//
    //---Set BaudRate---//
    URT_BRG.URT_InternalClockSource = URT_BDClock_PROC;
    URT_BRG.URT_BaudRateMode = URT_BDMode_Separated;
    URT_BRG.URT_PrescalerCounterReload = 0;	                //Set PSR
    URT_BRG.URT_BaudRateCounterReload = 155;	            //Set RLR
    URT_BaudRateGenerator_Config(URT0, &URT_BRG);   		//BR115200 = f(CK_URTx)/(PSR+1)/(RLR+1)/(OS_NUM+1)
    URT_BaudRateGenerator_Cmd(URT0, ENABLE);	            //Enable BaudRateGenerator
    //---TX/RX Clock---//
    URT_TXClockSource_Select(URT0, URT_TXClock_Internal);	//URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(URT0, URT_RXClock_Internal);	//URT_RX use BaudRateGenerator
    URT_TXOverSamplingSampleNumber_Select(URT0, 7);	        //Set TX OS_NUM
    URT_RXOverSamplingSampleNumber_Select(URT0, 7);	        //Set RX OS_NUM
    URT_RXOverSamplingMode_Select(URT0, URT_RXSMP_3TIME);
    URT_TX_Cmd(URT0, ENABLE);	                            //Enable TX
    URT_RX_Cmd(URT0, ENABLE);	                            //Enable RX
    
    //=====Set Mode=====//
    //---Set Data character config---//
    DataDef.URT_TX_DataLength  = URT_DataLength_8;
    DataDef.URT_RX_DataLength  = URT_DataLength_8;
    DataDef.URT_TX_DataOrder   = URT_DataTyped_LSB;
    DataDef.URT_RX_DataOrder   = URT_DataTyped_LSB;
    DataDef.URT_TX_Parity      = URT_Parity_No;
    DataDef.URT_RX_Parity      = URT_Parity_No;
    DataDef.URT_TX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_RX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_TX_DataInverse = DISABLE;
    DataDef.URT_RX_DataInverse = DISABLE;
    URT_DataCharacter_Config(URT0, &DataDef);
    //---Set Mode Select---//
    URT_Mode_Select(URT0, URT_URT_mode);
    //---Set DataLine Select---//
    URT_DataLine_Select(URT0, URT_DataLine_2);
    
    //=====Set Error Control=====//
    // to do...
    
    //=====Set Bus Status Detect Control=====//
    // to do...
    
    //=====Set Data Control=====//
    URT_RXShadowBufferThreshold_Select(URT0, URT_RXTH_1BYTE);
    URT_IdlehandleMode_Select(URT0, URT_IDLEMode_No);
    URT_TXGuardTime_Select(URT0, 0);
    
    //=====Enable URT Interrupt=====//
    URT_IT_Config(URT0, URT_IT_RX, ENABLE);
    URT_ITEA_Cmd(URT0, ENABLE);
    NVIC_EnableIRQ(URT0_IRQn);

    //=====Enable URT=====//
    URT_Cmd(URT0, ENABLE);
    URT_ClearTXData(URT0);
		
	//==See MG32_URT0_IRQ.c when interrupt in
}


/**
 *******************************************************************************
 * @brief       RST_Example
 * @details     RST module reset sample code
 *  @arg\b      1. Show reset event
 * @return
 * @note        This sample code used file "Sample_Retarget.c" function.
 *              The file "Sample_Retarget.c" must be brought in.
 * @par         Example
 * @code
    Sample_RST_DetectResetEvent();
 * @endcode
 *******************************************************************************
 */
void Sample_RST_DetectResetEvent (void)
{
    
    
    
    Sample_URT0_Init();                                         // Initial URT0
    
    printf_mg("\nShow reset event, ");                          //
    
    /*=== 1. Check reset flag ===*/
    if(RST_GetSingleFlagStatus(RST_PORF) == DRV_Happened)       // When power on reset happened
    {
        // Do something
        printf_mg(" PORF,");                                    // Print " PORF"
        RST_ClearFlag(RST_PORF);                                // Clear RST_PORF flag
    }
    
    if(RST_GetSingleFlagStatus(RST_SWF) == DRV_Happened)        // When software reset happened
    {
        // Do something
        printf_mg(" SWF,");                                     // Print " SWF"
        RST_ClearFlag(RST_SWF);                                 // Clear RST_SWF flag
    }
    
    if(RST_GetSingleFlagStatus(RST_EXF) == DRV_Happened)        // When external reset happened
    {
        // Do something
        printf_mg(" EXF,");                                     // Print " EXF"
        RST_ClearFlag(RST_EXF);                                 // Clear RST_EXF flag
    }
    
    if(RST_GetSingleFlagStatus(RST_CPUF) == DRV_Happened)       // When CPU SYSRESETREQ bit system reset happened
    {
        // Do something
        printf_mg(" CPUF,");                                    // Print " CPUF"
        RST_ClearFlag(RST_CPUF);                                // Clear RST_CPUF flag
    }    
    
    if(RST_GetSingleFlagStatus(RST_BOD0F) == DRV_Happened)      // When BOD0 reset happened
    {
        // Do something
        printf_mg(" BOD0F,");
        RST_ClearFlag(RST_BOD0F);                               // Clear RST_BOD0F flag
    }    
    
    if(RST_GetSingleFlagStatus(RST_BOD1F) == DRV_Happened)      // When BOD1 reset happened
    {
        // Do something
        printf_mg(" BOD1F,");                                   // Print " BOD1F"
        RST_ClearFlag(RST_BOD1F);                               // Clear RST_BOD1F flag
    }      
    
#if (defined(MG32_3RD) || defined(MG32_4TH))                    // Support MG32F02A032/A064/A072/A128/A132/U064/U128  
    if(RST_GetSingleFlagStatus(RST_BOD2F) == DRV_Happened)      // When BOD2 reset happened
    {
        // Do something
        printf_mg(" BOD2F,");                                   // Print " BOD2F"
        RST_ClearFlag(RST_BOD2F);                               // Clear RST_BOD2F flag
    }
#endif
    
    if(RST_GetSingleFlagStatus(RST_LPMF) == DRV_Happened)       // When low power mode reset happened
    {
        // Do something
        printf_mg(" LPMF,");                                    // Print " LPMF"
        RST_ClearFlag(RST_LPMF);                                // Clear RST_LPMF flag
    } 
    
    if(RST_GetSingleFlagStatus(RST_CSCF) == DRV_Happened)       // When CSC missing clock detect reset happened
    {
        // Do something
        printf_mg(" CSCF,");                                    // Print " CSCF"
        RST_ClearFlag(RST_CSCF);                                // Clear RST_CSCF flag
    }     
    
    if(RST_GetSingleFlagStatus(RST_MEMF) == DRV_Happened)       // When memory read/write protect or illegal address error reset happened
    {
        // Do something
        printf_mg(" MEMF,");                                    // Print " MEMF"
        RST_ClearFlag(RST_MEMF);                                // Clear RST_MEMF flag
    }     
    
    if(RST_GetSingleFlagStatus(RST_IWDTF) == DRV_Happened)      // When IWDT reset happened
    {
        // Do something
        printf_mg(" IWDTF,");                                   // Print " IWDTF"
        RST_ClearFlag(RST_IWDTF);                               // Clear RST_IWDTF flag
    }     
    
    if(RST_GetSingleFlagStatus(RST_WWDTF) == DRV_Happened)      // When WWDT reset happened
    {
        // Do something
        printf_mg(" WWDTF,");                                   // Print " WWDTF"
        RST_ClearFlag(RST_WWDTF);                               // Clear RST_WWDF flag
    }     
    
    if(RST_GetSingleFlagStatus(RST_ADCF) == DRV_Happened)       // When ADC reset happened
    {
        // Do something
        printf_mg(" ADCF,");                                    // Print " ADCF"
        RST_ClearFlag(RST_ADCF);                                // Clear RST_ADCF flag
    }     
 
#if (defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD))   // Support MG32F02A032/A064/A072/A128/A132/U064/U128    
    if(RST_GetSingleFlagStatus(RST_CMP0F) == DRV_Happened)          // When comparator CMP0 threshold reset happened
    {
        // Do something
        printf_mg(" CMP0F,");                                       // Print " CMP0F"
        RST_ClearFlag(RST_CMP0F);                                   // Clear RST_CMP0F flag
    } 
#endif 
    
#if (defined(MG32_1ST) || defined(MG32_2ND) || defined(MG32_3RD))   // Support MG32F02A032/A064/A072/A128/A132/U064/U128    
    if(RST_GetSingleFlagStatus(RST_CMP1F) == DRV_Happened)          // When comparator CMP1 threshold reset happened
    {
        // Do something
        printf_mg(" CMP1F,");                                       // Print " CMP1F"
        RST_ClearFlag(RST_CMP1F);                                   // Clear RST_CMP1F flag
    } 
#endif    
    
#if (defined(MG32_1ST))                                         // Support MG32F02A132/A072 
    if(RST_GetSingleFlagStatus(RST_CMP2F) == DRV_Happened)      // When comparator CMP2 threshold reset happened
    {
        // Do something
        printf_mg(" CMP2F,");                                   // Print " CMP2F"
        RST_ClearFlag(RST_CMP2F);                               // Clear RST_CMP2F flag
    } 
#endif
    
#if (defined(MG32_1ST))                                         // Support MG32F02A132/A072 
    if(RST_GetSingleFlagStatus(RST_CMP3F) == DRV_Happened)      // When comparator CMP3 threshold reset happened
    {
        // Do something
        printf_mg(" CMP3F,");                                   // Print " CMP3F"
        RST_ClearFlag(RST_CMP3F);                               // Clear RST_CMP3F flag
    } 
#endif
    
    if(RST_GetSingleFlagStatus(RST_WRF) == DRV_Happened)        // When warm reset happened
    {
        // Do something
        printf_mg(" Warm Reset");                               // Print " Warm Reset"
        RST_ClearFlag(RST_WRF);                                 // Clear RST_WRF
    } 
    
    if(RST_GetSingleFlagStatus(RST_CRF) == DRV_Happened)        // When cold reset happened   
    {
        // Do something
        printf_mg(", Cold Reset");                              // Print " Cold Reset"
        RST_ClearFlag(RST_CRF);                                 // Clear RST_CRF
    } 
}



