/**
 ******************************************************************************
 *
 * @file        BSP_Button.c
 * @brief        
 *
 * @par         Project
 *              MG32F02N128
 * @version     V1.00
 * @date        2024/08/28
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_ChipInit.h"
#include "MG32_GPIO_DRV.h"
#include "BSP_Button.h"
#include "Global.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define BSP_BT_KEY0_PINX     PIND(6)
#define BSP_BT_KEY1_PINX     PIND(8)
#define BSP_BT_KEY2_PINX     PIND(10)
#define BSP_BT_KEY3_PINX     PIND(11)

#define BSP_BTN_TOTAL        4

#define BTN_DebounceTime     10

/* Private typedef -----------------------------------------------------------*/
typedef struct
{
    uint8_t   Button_ChangeFlag[BSP_BTN_TOTAL];                
    uint8_t   ButtonStatus[BSP_BTN_TOTAL];      
    uint8_t   ButtonDebounceCnt[BSP_BTN_TOTAL];
    uint8_t   __padding[4];    
}BTNCTR_TypeDef;

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function ----------------------------------------------------------*/
const static uint32_t BSP_ButtonPortDefine[BSP_BTN_TOTAL] = 
{
    GPIOD_Base,
    GPIOD_Base,
    GPIOD_Base,
    GPIOD_Base,
};

const static uint8_t BSP_ButtonPinDefine[BSP_BTN_TOTAL] = 
{
     6,
     8,
    10,
    11,
}; 

static BTNCTR_TypeDef   BT_KEY;

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/
/**
 *******************************************************************************
 * @brief       Button parameter to default.
 * @details     
 * @exception   
 * @note        
 *******************************************************************************
 */
void BSP_Button_DeInit(void)
{
    uint8_t BT_DeInitTmp;

    for(BT_DeInitTmp = 0; BT_DeInitTmp < BSP_BTN_TOTAL; BT_DeInitTmp++)
    {
        BT_KEY.ButtonStatus[BT_DeInitTmp] = 0;
        BT_KEY.ButtonDebounceCnt[BT_DeInitTmp] = 0;
        BT_KEY.Button_ChangeFlag[BT_DeInitTmp] = 0;
    }
}

/**
 *******************************************************************************
 * @brief       Button initial.
 * @details     
 * @param[in]   
 * @exception   
 * @note        
 *******************************************************************************
 */
void BSP_Button_Init(void)
{
    PIN_InitTypeDef  BT_Pin;

    /*Pin configure*/
    BT_Pin.PINX_Mode               = PINX_Mode_Digital_I;
    BT_Pin.PINX_PUResistant        = PINX_PUResistant_Enable;
    BT_Pin.PINX_Speed              = PINX_Speed_High;
    BT_Pin.PINX_Inverse            = PINX_Inverse_Disable;
    BT_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    BT_Pin.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    BT_Pin.PINX_Alternate_Function = 0;
    GPIO_PinMode_Config(BSP_BT_KEY0_PINX, &BT_Pin);
    GPIO_PinMode_Config(BSP_BT_KEY1_PINX, &BT_Pin);
    GPIO_PinMode_Config(BSP_BT_KEY2_PINX, &BT_Pin);
    GPIO_PinMode_Config(BSP_BT_KEY3_PINX, &BT_Pin);

    /*Parameter initial*/
    BSP_Button_DeInit();
}

/**
 *******************************************************************************
 * @brief       Button handler.  
 * @details     How often to call this function to decide key sensitivity
 * @exception   
 * @note        
 *******************************************************************************
 */
void BSP_Button_Handler(void)
{
    uint8_t  BT_IRQHandlerTmp;
    uint8_t  BT_IRQHandlerStatus;

    for(BT_IRQHandlerTmp = 0; BT_IRQHandlerTmp < BSP_BTN_TOTAL; BT_IRQHandlerTmp++)
    {
        BT_IRQHandlerStatus = PX( BSP_ButtonPortDefine[BT_IRQHandlerTmp],
                                  BSP_ButtonPinDefine[BT_IRQHandlerTmp]);

        if( BT_IRQHandlerStatus != BT_KEY.ButtonStatus[BT_IRQHandlerTmp])
        {
            BT_KEY.ButtonDebounceCnt[BT_IRQHandlerTmp] ++;
        }
        else
        {
            BT_KEY.ButtonDebounceCnt[BT_IRQHandlerTmp] = 0;
        }
        /*Debounce handle*/
        if( BT_KEY.ButtonDebounceCnt[BT_IRQHandlerTmp] > BTN_DebounceTime)
        {
            BT_KEY.ButtonStatus[BT_IRQHandlerTmp] = BT_IRQHandlerStatus;

            if( BT_KEY.ButtonStatus[BT_IRQHandlerTmp] == 0)
            {
                BT_KEY.Button_ChangeFlag[BT_IRQHandlerTmp] = 1;

                BSP_Button_MakeCallback();
            }
        }
    }
}

/**
 *******************************************************************************
 * @brief       Get this key whether to press or not.
 * @details     
 * @param[in]   Key_Number: (0 ~ 3)
 * @exception   
 * @note        
 *******************************************************************************
 */
uint8_t BSP_Button_GetKeyFlag(uint8_t Key_Number)
{
    return(BT_KEY.Button_ChangeFlag[Key_Number]);
}

/**
 *******************************************************************************
 * @brief       Clear this key software flag.
 * @details     
 * @param[in]   Key_Number: (0 ~ 3)
 * @exception   
 * @note        
 *******************************************************************************
 */
void BSP_Button_ClearKeyFlag(uint8_t Key_Number)
{
    BT_KEY.Button_ChangeFlag[Key_Number] = 0;
}

/**
 *******************************************************************************
 * @brief       Key callback function when key be pressed.
 * @details     
 * @param[in]   
 * @exception   
 * @note        
 *******************************************************************************
 */
__WEAK void BSP_Button_MakeCallback(void)
{
    //=========================================================
    //Note : This function should not be modified, when the callback
    //       is needed, the BSP_Button_MakeCallback can
    //       be implemented in the user file.      
}

