


/**
 ******************************************************************************
 *
 * @file        BSP_2ColorDotMatrixLED.c
 * @brief       This is two color dot matrix LED C file.
                (Use 74HC595 to control).
 * @par         Project
 *              MG32
 * @version     V1.03
 * @date        2023/01/31
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
/*==============================================================================
                                 User Notes
How To use this function:
-----------------------
   1. The code use 74HC595 (Serial to parameter shift register) to control 
      2 color dot matrix LED.
   2. Use BSP_2ColorDotMatrixLED_Init() to initial.
      (1) Default is OFF.
   3. Periodic call BSP_2ColorDotMatrixLED_main() function.
      (1) The freq * 8 is 2 color dot matrix LED(8x8) refresh rate.  
   4. 2 color dot matrix display different picture by calling BSP_2ColorDotMatrixLED_RefreshFrame() function. 
      (1) Input parameter is new display picture data pointer.
      (2)   COL 1      2      3      4      5      6      7      8
          ROW 1 Byte0  Byte1  Byte2  Byte3  Byte4  Byte5  Byte6  Byte7
              2 Byte8  Byte9  Byte10 Byte11 Byte12 Byte13 Byte14 Byte15        
              3 Byte16 Byte17 Byte18 Byte19 Byte20 Byte21 Byte22 Byte23
              4 Byte24 Byte25 Byte26 Byte27 Byte28 Byte29 Byte30 Byte31
              5 Byte32 Byte33 Byte34 Byte35 Byte36 Byte37 Byte38 Byte39
              6 Byte40 Byte41 Byte42 Byte43 Byte44 Byte45 Byte46 Byte47
              7 Byte48 Byte49 Byte50 Byte51 Byte52 Byte53 Byte54 Byte55
              8 Byte56 Byte57 Byte58 Byte59 Byte60 Byte61 Byte62 Byte63
              
Driver architecture:
--------------------
   + MG32_GPIO_DRV
   
Known Limitations:
------------------

Require parameter
------------------
    Require module : CSC / GPIO
    
    GPIO pin configuration : 
        Pin / IO mode / AFS
        ---  --------  -----
        PB12/ PPO     / GPIO
        PB13/ PPO     / GPIO
        PB14/ PPO     / GPIO
        PB15/ PPO     / GPIO
    
Example codes:
------------------

==============================================================================*/ 
 
/* Includes ------------------------------------------------------------------*/ 
#include "BSP_17_2ColorDotMatrixLED.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
// GPIO : Serail interface to parallel bus
#define S2P595_SDI_PIN          PX_Pin_12
#define S2P595_LCH_PIN          PX_Pin_13
#define S2P595_SCLK_PIN         PX_Pin_14
#define S2P595_OEn_PIN          PX_Pin_15
#define S2P595_PIN_Mode         PINX_Mode_PushPull_O
#define S2P595_PIN_AFS          0
#define S2P595_IOM              IOMB
#define S2P595_PORT             GPIOB

// Signal control
#define S2P595_SDI_HIGH()       PB12 = 1
#define S2P595_SDI_LOW()        PB12 = 0

#define S2P595_LCH_HIGH()       PB13 = 1
#define S2P595_LCH_LOW()        PB13 = 0

#define S2P595_SCLK_HIGH()      PB14 = 1
#define S2P595_SCLK_LOW()       PB14 = 0

#define S2P595_nOE_HIGH()       PB15 = 1
#define S2P595_nOE_LOW()        PB15 = 0

#define S2P595_nOE_ENABLE()     PB15 = 0
#define S2P595_nOE_DISABLE()    PB15 = 1

#define DOTLED_RED_DATA         0
#define DOTLED_GREEN_DATA       1

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/ 
static uint8_t DotLED_DramaAct;                             // Drama index - the same as ROW counter
static uint8_t DotLED_RefreshFrameFlag;                     // DotLED_RefreshFrameFlag=1, Busy event
                                                            // DotLED_RefreshFrameFlag=0, Free

static uint8_t DotLED_COMTable[8] = {                       // COM Table
    0xFE,       // 1111 1110
    0xFD,       // 1111 1101
    0xFB,       // 1111 1011
    0xF7,       // 1111 0111
    0xEF,       // 1110 1111    
    0xDF,       // 1101 1111
    0xBF,       // 1011 1111
    0x7F};      // 0111 1111

// DotLED_ShowData data source from DotLED_ShowDataTmp
static uint8_t DotLED_ShowData[16];                         // [odd number] : Green data
                                                            // [even number] : Red data
    
static uint8_t DotLED_ShowDataTmp[16];                      // [odd number] : Green data
                                                            // [even number] : Red data
/* Private function prototypes -----------------------------------------------*/
static void BSP_2ColorDotMatrixLED_Series2Parallel(uint8_t R_LED, uint8_t G_LED, uint8_t LED_Com);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief	    2 color dot matrix LED initial
 * @details     Initial 74HC595 control pin & Drama initial
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
void BSP_2ColorDotMatrixLED_Init(void) 
{ 
    PIN_InitTypeDef  DotLED_Pin;
    uint8_t          DotLED_Tmp;

    // ------------------------------------------------------------------------
    // GPIO initial. (74HC595 control pin - Serial bus input)
    // ------------------------------------------------------------------------
    GPIO_SetPortBit(S2P595_PORT, (uint16_t)((S2P595_SDI_PIN | S2P595_LCH_PIN | S2P595_SCLK_PIN)));
    
    DotLED_Pin.PINX_Pin                 = (S2P595_SDI_PIN | S2P595_LCH_PIN | S2P595_SCLK_PIN | S2P595_OEn_PIN);
    DotLED_Pin.PINX_Mode                = S2P595_PIN_Mode;
    DotLED_Pin.PINX_PUResistant         = PINX_PUResistant_Disable;
    DotLED_Pin.PINX_Speed               = PINX_Speed_Low;
    DotLED_Pin.PINX_Inverse             = PINX_Inverse_Disable;
    DotLED_Pin.PINX_OUTDrive            = PINX_OUTDrive_Level0;
    DotLED_Pin.PINX_FilterDivider       = PINX_FilterDivider_Bypass;
    DotLED_Pin.PINX_Alternate_Function  = S2P595_PIN_AFS;   
    
    GPIO_PortMode_Config(S2P595_IOM ,&DotLED_Pin);
    
    // ------------------------------------------------------------------------
    // Parameter initial
    // ------------------------------------------------------------------------
    DotLED_DramaAct                     = 0;
    DotLED_RefreshFrameFlag             = 0;

    // ------------------------------------------------------------------------
    for(DotLED_Tmp = 0; DotLED_Tmp < 16; DotLED_Tmp++)
    {
        DotLED_ShowData[ DotLED_Tmp]    = 0xFF;     // No display (0xFF)
        DotLED_ShowDataTmp[ DotLED_Tmp] = 0xFF;     // No display (0xFF)
    }
}
/**
 *******************************************************************************
 * @brief	    Update 2 color dot matrix LED show.
 * @return      None 
 * @exception   No 
 * @note        1. Update ShowBuffer of DotMatrixLED before call the function.
                2. ShowBuffer[8][8] mapping real dot LED location.
                3. ShowBuffer data = 
                    (0).DOTLED_DISABLE: disable in the LED dot.
                    (1).DOTLED_RED    : show Red in the LED dot.
                    (2).DOTLED_GREEN  : show Green in the LED dot.
                    (3).DOTLED_ORANGE : show ORANGE (Red + Green) in the dot.
 *******************************************************************************
 */
void BSP_2ColorDotMatrixLED_RefreshFrame(uint8_t *pDotLED_Buf)
{
    uint8_t DotLED_RefreshTmp;
    uint8_t DotLED_RefreshRed;
    uint8_t DotLED_RefreshGreen;
    uint8_t DotLED_RefreshEN;
    uint8_t DotLED_RefreshRow;
    
    // ------------------------------------------------------------------------
    DotLED_RefreshFrameFlag = 1;                            // =1, Busy event 
    DotLED_RefreshRow       = 0;                            // =0, 1, 2, ..., 7
    
    DotLED_RefreshRed       = 0xFF;                         // Red LED=0xFF, 8 LEDs, No display (Black)
    DotLED_RefreshGreen     = 0xFF;                         // Green LED=0xFF, 8 LEDs, No display (Black)
    
    DotLED_RefreshEN        = 0x80;                         // Value=0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01
    
    // ------------------------------------------------------------------------
    // 1. DotLED_RefreshTMP=0,1,2,3...63
    // 2. pDotLED_Buf is 8x8 array data. It has 2 bits indicating the color in one byte.
    //      2.1 No colors
    //      2.2 Pure Red
    //      2.3 Pure Green
    //      2.4 ORANGE (Red + Green)
    // 3. Separate color data into DotLED_RefreshRed, DotLED_RefreshGreen. 
    // 4. DotLED_ShowDataTmp[0],[2],[4]...[14] deals RED data.
    // 5. DotLED_ShowDataTmp[1],[3],[5]...[15] deals Green data.
    // ------------------------------------------------------------------------
    for(DotLED_RefreshTmp = 0; DotLED_RefreshTmp < (8*8); DotLED_RefreshTmp++)
    {
        // --------------------------------------------------------------------
        if(pDotLED_Buf[DotLED_RefreshTmp] & DOTLED_RED)     // DOTLED_RED=0x1
        {
            DotLED_RefreshRed &= (~DotLED_RefreshEN);       // Red LED : show Red in the LED dot.
        }
        
        // --------------------------------------------------------------------
        if(pDotLED_Buf[DotLED_RefreshTmp] & DOTLED_GREEN)   // DOTLED_GREEN=0x2    
        {
            DotLED_RefreshGreen &= (~DotLED_RefreshEN);     // Green LED : show Green in the LED dot.
        }
        
        // --------------------------------------------------------------------
        // Judge to whether change next row or not.
        // --------------------------------------------------------------------
        if(DotLED_RefreshEN == 0x01)
        {
//            DotLED_RefreshEN = 0x80;                        // Reset pointer
            
            DotLED_ShowDataTmp[ (DotLED_RefreshRow) * 2 + DOTLED_RED_DATA ]   = DotLED_RefreshRed;  // [0], [2], [4], [6], ... [14]
            DotLED_ShowDataTmp[ (DotLED_RefreshRow) * 2 + DOTLED_GREEN_DATA ] = DotLED_RefreshGreen;// [1], [3], [5], [7], ... [15]
            
            DotLED_RefreshRow   = DotLED_RefreshRow + 1;    // =0, 1, 2, ..., 7
            
            DotLED_RefreshRed   = 0xFF;                     // Red LED=0xFF, 8 LEDs, No display (Black)
            DotLED_RefreshGreen = 0xFF;                     // Green LED=0xFF, 8 LEDs, No display (Black)
            
            DotLED_RefreshEN    = 0x80;
        }
        else
        {
            DotLED_RefreshEN = DotLED_RefreshEN >> 1;       // 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01
        }
    }
    
    DotLED_RefreshFrameFlag = 0;                            // =0, It dicates free.     
}
/**
 *******************************************************************************
 * @brief	    2 color dot matrix main flow. 
 * @details     According visiting function frequency and RefreshFlag parameter 
                to control 2 color dot matrix refresh frequency.
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
void BSP_2ColorDotMatrixLED_main(void)
{
    uint8_t BSP_2ColorDotMatrixLED_mainTmp;
    
    // ------------------------------------------------------------------------
    if(DotLED_DramaAct == 0 && DotLED_RefreshFrameFlag == 0)
    {
        // --------------------------------------------------------------------
        // Update new picture. DotLED_ShowDataTmp -> DotLED_ShowData
        // --------------------------------------------------------------------
        for(BSP_2ColorDotMatrixLED_mainTmp = 0; BSP_2ColorDotMatrixLED_mainTmp < 16; BSP_2ColorDotMatrixLED_mainTmp++)
        {
            DotLED_ShowData[BSP_2ColorDotMatrixLED_mainTmp] = DotLED_ShowDataTmp[BSP_2ColorDotMatrixLED_mainTmp];
        }
    }
    
    BSP_2ColorDotMatrixLED_Series2Parallel(DotLED_ShowData[(DotLED_DramaAct * 2) + DOTLED_RED_DATA],
                                            DotLED_ShowData[(DotLED_DramaAct * 2) + DOTLED_GREEN_DATA], 
                                            DotLED_COMTable[DotLED_DramaAct]);
    
    // ------------------------------------------------------------------------
    // Control display row
    // ------------------------------------------------------------------------
    DotLED_DramaAct = DotLED_DramaAct + 1;                  // DotLED_DramaAct=0~7
    
    if(DotLED_DramaAct == 8)
    {
        DotLED_DramaAct = 0;
    }
}
/**
 *******************************************************************************
 * @brief	    Update 2 color dot matrix LED status.
 * @details     Use serial to parallel shift register(74HC595) to control
                2 color dot matrix LED.
 * @param[in]   R_LED  : Control Red LED in N column. 
 * @param[in]   G_LED  : Control Green LED in N column.
 * @param[in]   LED_Com: Control LED display column.
 * @return      None
 *******************************************************************************
 */
static void BSP_2ColorDotMatrixLED_Series2Parallel(uint8_t R_LED, uint8_t G_LED, uint8_t LED_Com)
{
    uint8_t R_LED_Temp = R_LED;
    uint8_t G_LED_Temp = G_LED;
    uint8_t LED_COM_Temp = LED_Com;
    uint8_t lCount8;

    // ------------------------------------------------------------------------
    // Output Enable :
    //      Active-low Output Enable. A low on this input allows the data from 
    //      the latches to be presented at the outputs. 
    //      A high on this input forces the outputs (QA−QH ) into the 
    //      high−impedance state.
    // ------------------------------------------------------------------------
    S2P595_nOE_LOW();                                       // nOE Enable
    //S2P595_LCH_HIGH();
    
    // ------------------------------------------------------------------------
    // Control LED column.
    // ------------------------------------------------------------------------
    lCount8 = 8;
    do{
        // --------------------------------------------------------------------
        // (COM Selection) Data bit : High or Low state
        // --------------------------------------------------------------------
        // INPUTS (SDI pin) :
        //  Serial Data Input. The data on this pin is shifted into the
        //  8−bit serial shift register.
        // --------------------------------------------------------------------
        if(LED_COM_Temp & 0x01)
            S2P595_SDI_HIGH();
        else
            S2P595_SDI_LOW();
        __ISB();

        // --------------------------------------------------------------------
        // Clock (Low to High)
        // --------------------------------------------------------------------
        // Shift Clock :
        //  Shift Register Clock Input. A low− to−high transition on this input 
        //  causes the data at the Serial Input pin to be shifted into the 8−bit 
        //  shift register.
        // --------------------------------------------------------------------
        S2P595_SCLK_HIGH();
        //S2P595_LCH_HIGH();
        __ISB();
        S2P595_SCLK_LOW();
        //S2P595_LCH_LOW();
        
        // --------------------------------------------------------------------
        LED_COM_Temp = LED_COM_Temp >> 1;
        
    }while(-- lCount8 != 0);
    
    // ------------------------------------------------------------------------
    // Control Green LED in n column.
    // ------------------------------------------------------------------------
    lCount8 = 8;
    do{
        // --------------------------------------------------------------------
        // (Green LED) Data bit : High or Low state
        if(G_LED_Temp & 0x01)
            S2P595_SDI_HIGH();
        else
            S2P595_SDI_LOW();
        __ISB();

        // --------------------------------------------------------------------
        // Clock (Low to High)
        S2P595_SCLK_HIGH();
        //S2P595_LCH_HIGH();
        __ISB();
        S2P595_SCLK_LOW();
        //S2P595_LCH_LOW();

        // --------------------------------------------------------------------
        G_LED_Temp = G_LED_Temp >> 1;
        
    }while(-- lCount8 != 0);
    
    // ------------------------------------------------------------------------
    // Control Red LED in n column.
    // ------------------------------------------------------------------------
    lCount8 = 8;
    do{
        // --------------------------------------------------------------------
        // (Red LED) Data bit : High or Low state
        if(R_LED_Temp & 0x01)
            S2P595_SDI_HIGH();
        else
            S2P595_SDI_LOW();
        __ISB();

        // --------------------------------------------------------------------
        // Clock (Low to High)
        S2P595_SCLK_HIGH();
        //S2P595_LCH_HIGH();
        __ISB();
        S2P595_SCLK_LOW();
        //S2P595_LCH_LOW();

        // --------------------------------------------------------------------
        R_LED_Temp = R_LED_Temp >> 1;

    }while(-- lCount8 != 0);

    // ------------------------------------------------------------------------
    // Latch clock :
    //      Storage Latch Clock Input. A low-to-high transition on this input latches
    //      the shift register data.
    // ------------------------------------------------------------------------
    S2P595_LCH_LOW();
    __ISB();
    
    S2P595_LCH_HIGH();
    __ISB();
    
    
    
}






