

/**
 ******************************************************************************
 *
 * @file        BSP_12_BLE(HC42).C
 *
 * @brief       This is the C code file.
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2023/05/10
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "__BSP_Wizard.h"
#include "MG32.h"

#if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC42
#include "MG32_GPIO_DRV.h"
#include "MG32_TM_DRV.h"
#include "MG32_URT_DRV.h"
#include "MG32_SPI_DRV.h"
#include "BSP_12_BLE_HC42.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO
#define BLE_HC06TX_PINX         PINE(0)
#define BLE_HC06RX_PINX         PINE(1)

#define BLE_HC06TX_AFS          11
#define BLE_HC06RX_AFS          11


// URT Module
#define bBLE_URTn               URT4

//TM                                                     
#define BLE_TM                  TM01            // BLE Timebase Timer.

//TX / RX Buffer size.
#define BufferSize              200             // DataBuffer size

// For StateFlag
#define State_ConnectFail       0x01            // State, connect fail flag
#define State_VersionFail       0x02            // State, got version fail flag
#define State_NameFail          0x04            // State, set name fail flag
#define State_BaudrateFail      0x08            // State, set baud rate fail flag
#define State_RFPMFail          0x10            // State, Set RF power fail flag
#define State_RFAINTFail        0x20            // State, Set broadcast interval fail flag

// HC06 Module Status Define
#define BLE_HC42_DISABLE        0
#define BLE_HC42_GETVERSION     1
#define BLE_HC42_SETBAUDRATE    2
#define BLE_HC42_RFPM           3
#define BLE_HC42_AINT           4
#define BLE_HC42_DEVICENAME     5
#define BLE_HC42_CONNECT        6

#define BLE_HC42_DELAYMASK      0x0000FFFF
#define BLE_HC42_STATEMASK      0xFFFF0000
#define BLE_HC42_STATESHIFT     16

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* External variables --------------------------------------------------------*/
/* External functions --------------------------------------------------------*/

uint8_t BLE_RXF;

//Static variable 
volatile static uint8_t     StateFlag;                              // State flag
volatile static uint8_t     BLE_RX_BUF[BufferSize];                 // BLE RX buffer 
volatile static uint8_t     BLE_TX_BUF[BufferSize];                 // BLE TX buffer

volatile static uint8_t     BLE_RX_BUF_iIndex;                      // BLE_RX_BUF input index.
volatile static uint8_t     BLE_RX_BUF_oIndex;                      // BLE_RX_BUF output index.
volatile static uint8_t     BLE_RX_BUF_IndexCnt;                    // BLE_RX_BUF remain data byte.


volatile static uint8_t     BLE_TX_BUF_iIndex;                      // BLE_TX_BUF input index.
volatile static uint8_t     BLE_TX_BUF_oIndex;                      // BLE_TX_BUF output index.
volatile static uint8_t     BLE_TX_BUF_IndexCnt;                    // BLE_TX_BUF remain data byte.
volatile static uint8_t     BLE_TX_BUF_TXF;                         // The flag represents that software is transmitting data to BLE flag.

volatile static uint32_t    BLEStatus;

static uint8_t AT[]         = {"AT"};                               // 1. Return "OK"
static uint8_t AT_VERSION[] = {"AT+VERSION"};                       // 2. Return "OK+VERSION=HC-42 V2.0.180601"
static uint8_t AT_NAME[]    = {"AT+NAME=ZL_HC-42"};                 // 3. Return "OK+NAME=ZL_HC-42"
                                                                    //    Write  "AT+NAME" 
                                                                    //    Return "HC-42 BLE Module"
static uint8_t AT_RFPM[]    = {"AT+RFPM=0"};                        // 4. Return "OK+RFPM=0"
                                                                    //    Write  "AT+RFPM=" 
                                                                    //    Return "OK+RFPM=0"
static uint8_t AT_UART[]    = {"AT+UART=9600"};                     // 5. Return "OK+UART=9600"
                                                                    //    Write  "AT+UART" 
                                                                    //    Return "OK+UART=9600"
static uint8_t AT_AINT[]    = {"AT+AINT=200"};                      // 6. Return "OK+AINT=200"
                                                                    //    Write  "AT+AINT=1000" 
                                                                    //    Return "OK+AINT=1000"
//static uint8_t AT_PM[]      = {"AT+PM"};                            // 7. Return "OK+PM=0"
//                                                                    //    Write  "AT+PM=1"
//                                                                    //    Return "OK+PM=1"
//static uint8_t AT_SLEEP[]   = {"AT+SLEEP"};                         // 8. Return "OK+SLEEP"
//                                                                    //    When LPIN (PIN19, input pin) is at high level, it can enter 
//                                                                    //    low power consumption operation through AT+SLEEP command, and 
//                                                                    //    when LPIN is at low level, it will unconditionally exit from 
//                                                                    //    low power consumption operation.
//static uint8_t AT_PD[]      = {"AT+PD"};                            // 9. Return "OK+PD" 
//                                                                    //    By pulling down "LPIN" or "RESEET" trigger wake-up process. 
//static uint8_t AT_LEDnM[]   = {"AT+LED1M"};                         // 10. Return "OK+LED1M=2"
//                                                                    //     Write  "AT+LED2M=1"
//                                                                    //     Return "OK+LED2M=1"
//static uint8_t AT_DEFAULT[] = {"AT+DEFAULT"};                       // 11. Return "OK+DEFAULT"
//static uint8_t AT_ROLE[]    = {"AT+ROLE"};                          // 12. Return "OK+ROLE=S/P"
//                                                                    //     Write  "AT+ROLE=M"
//                                                                    //     Return "OK+ROLE=M/C"
//static uint8_t AT_RESET[]   = {"AT+RESET"};                         // 13. Return "OK+RESET"
//// iBeacon                                                          
//static uint8_t AT_IBEN[]    = {"AT+IBEN"};                          // 14. Return "OK+IBEN=0" default 0
//                                                                    //     Write  "AT+IBEN=1"
//                                                                    //     Return "OK+IBEN=1"
//static uint8_t AT_IBUUID[]  = {"AT+IBUUID"};                        // 15. Return "OK+IBUUID=FDA50693-A4E2-4FB1-AFCF-C6EB07647825"
//                                                                    //     Write  "AT+IBUUID=xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx"
//                                                                    //     Return "OK+IBUUID=FDA50693-A4E2-4FB1-AFCF-C6EB07647825"
//static uint8_t AT_IBRSSI[]  = {"AT+IBRSSI"};                        // 16. Return "OK+IBRSSI=C3"
//                                                                    //     Write  "AT+IBRSSI=C8"
//                                                                    //     Return "OK+IBRSSI=C8"
//static uint8_t AT_IBMAJ[]   ={"AT+IBMAJ"};                          // 17. Return "OK+IBMAJ=27C6"
//                                                                    //     Write  "AT+IBMAJ=1234"
//                                                                    //     Return "OK+IBMAJ=1234"
//static uint8_t AT_IBMIN[]   ={"AT+IBMIN"};                          // 18. Return "OK+IBMIN=8B06"
//                                                                    //     Write  "AT+IBMIN=1234"
//                                                                    //     Return "OK+IBMIN=1234"

/* Private function prototypes -----------------------------------------------*/
static void BLE_UART_Init(void);
static void BLE_TM_Init(void);
static void BSP_BLE_TimeBaseIRQHandler(void);
static void BSP_BLE_ParameterInit(void);
static void BSP_BLE_TriggerSendData(void);

void TM0x_IRQHandler(void);

/* External variables --------------------------------------------------------*/
/* External functions --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief	 BLE IRQHandler
 * @details     
 * @return      
 * @note       
 *******************************************************************************
 */
void BSP_BLE_IRQHandler(void)
{
    // When URT received data
    if((bBLE_URTn->STA.W & URT_IT_RX) != 0)
    {
        if(BLE_RX_BUF_IndexCnt<(BufferSize+1))
        {
            BLE_RX_BUF[BLE_RX_BUF_iIndex] = bBLE_URTn->RDAT.B[0];   // URTx RX data to DataBuffer
            BLE_RX_BUF_IndexCnt++;
            if((BLE_RX_BUF_iIndex)<(BufferSize-1))
            {
                BLE_RX_BUF_iIndex++;
            }
            else
            {
                BLE_RX_BUF_iIndex = 0;
            }
        }
        else
        {
            bBLE_URTn->STA.W = URT_IT_RX;                           // Clear URTx RX event flag
            BSP_BLE_RxOverRunCallback();
        }             
    }
    // When TXF or TCF happened
    if((bBLE_URTn->STA.W & ((bBLE_URTn->INT.W) & (URT_IT_TX | URT_IT_TC))) != 0)
    {
        if(BLE_TX_BUF_IndexCnt!=0)
        {
            bBLE_URTn->TDAT.B[0] = BLE_TX_BUF[BLE_TX_BUF_oIndex];
            
            if(BLE_TX_BUF_oIndex==(BufferSize-1))
            {
                BLE_TX_BUF_oIndex = 0;
            }
            else
            {
                BLE_TX_BUF_oIndex++;
            }
            BLE_TX_BUF_IndexCnt--;
        }
        else
        {
            bBLE_URTn->STA.W = (URT_IT_TX | URT_IT_TC);                     // Clear URTx TXF and TCF event flag
            BLE_TX_BUF_TXF = 0;
        }
    }    
}

/**
 *******************************************************************************
 * @brief     BLE timebase interrupt handle.
 * @details
 * @return      
 * @note        
 *******************************************************************************
 */
void TM0x_IRQHandler(void)
{
    BSP_BLE_TimeBaseIRQHandler();
}

/**
 *******************************************************************************
 * @brief    BLE inital.
 * @details     
 * @return      
 * @note       
 *******************************************************************************
 */
void BSP_BLE_Init(void)
{
    /*Parameter Initial*/
    BSP_BLE_ParameterInit();
    
    /*Module Inital*/
    BLE_UART_Init();
    BLE_TM_Init();
}

/**
 *******************************************************************************
 * @brief	 Get BLE connect status. 
 * @details     
 * @return      
 * @note       
 *******************************************************************************
 */
uint8_t GetConnectedStatus(void)
{
    if(((BLEStatus&BLE_HC42_STATEMASK)>>BLE_HC42_STATESHIFT) == BLE_HC42_CONNECT )
    {
        return(1);
    }
    
    return(0);
    
}

/**
 *******************************************************************************
 * @brief     Clear BLE TX buffer.
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_ClearBleTxBuffer (void)
{
    BLE_TX_BUF_iIndex = 0;
    BLE_TX_BUF_oIndex = 0;
    BLE_TX_BUF_IndexCnt = 0;
}
/**
 *******************************************************************************
 * @brief	    BLE transmit data
 * @details     
 * @param[in]   SrcIndex:
 * @param[in]   TX_len:
 *  @arg\b          1 ~255
 * @return      Actually write buffer bytes.
 * @exception   No
 * @note 
 * @par         Example
 * @code
    BLE_TransmitData(&DataBuffer[0], 64);
 * @endcode
 *******************************************************************************
 */
uint8_t BSP_BLE_WriteTxBuffer(uint8_t* SrcIndex, uint8_t TX_len)
{
    uint32_t BSP_BLE_WTxBufTmp;
    uint8_t  BSP_BLE_WTxBufLen;
    
    if((BufferSize - BLE_TX_BUF_IndexCnt)<TX_len)
    {
        BSP_BLE_WTxBufLen = (BufferSize - BLE_TX_BUF_IndexCnt);
    }
    else
    {
        BSP_BLE_WTxBufLen = TX_len;
    }
    
    for(BSP_BLE_WTxBufTmp=0;BSP_BLE_WTxBufTmp<BSP_BLE_WTxBufLen;BSP_BLE_WTxBufTmp++)
    {
        BLE_TX_BUF[BLE_TX_BUF_iIndex] = *SrcIndex;
        SrcIndex++;
        if(BLE_TX_BUF_iIndex==(BufferSize-1))
        {
            BLE_TX_BUF_iIndex = 0;
        }
        else
        {
            BLE_TX_BUF_iIndex++;
        }
        BLE_TX_BUF_IndexCnt++;
    }
    return(BSP_BLE_WTxBufLen);
}
/**
 *******************************************************************************
 * @brief     Clear BLE RX buffer.
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_ClearBleRxBuffer (void)
{
    BLE_RX_BUF_iIndex   = 0;
    BLE_RX_BUF_oIndex   = 0;
    BLE_RX_BUF_IndexCnt = 0;
    BLE_RXF             = 0;
}

/**
 *******************************************************************************
 * @brief     Move BLE RX buffer data to DestIndex 
 * @details
 * @param[in] DestIndex  : Destination address.
 * @param[in] DestMaxSize: Destination buffer max size.
 * @return    Received data byte number. 1 ~ 200 bytes.
 * @note
 *******************************************************************************
 */

uint8_t BSP_BLE_ReadRxBuffer (uint8_t* DestIndex, uint8_t DestMaxSize)
{
    
    uint8_t   BSP_BLE_RDRBuf_Len;
    uint32_t  BSP_BLE_RDRBuf_Tmp;
    
    if(BLE_RX_BUF_IndexCnt>DestMaxSize)
    {
        BSP_BLE_RDRBuf_Len = DestMaxSize;
    }
    else
    {
        BSP_BLE_RDRBuf_Len = BLE_RX_BUF_IndexCnt;
    }
    for(BSP_BLE_RDRBuf_Tmp=0;BSP_BLE_RDRBuf_Tmp<BSP_BLE_RDRBuf_Len;BSP_BLE_RDRBuf_Tmp++)
    {
        DestIndex[BSP_BLE_RDRBuf_Tmp] = BLE_RX_BUF[BLE_RX_BUF_oIndex];
        if(BLE_RX_BUF_oIndex<(BufferSize-1))
        {
            BLE_RX_BUF_oIndex++;
        }
        else
        {
            BLE_RX_BUF_oIndex = 0;
        }
    }
    BLE_RX_BUF_IndexCnt = BLE_RX_BUF_IndexCnt - BSP_BLE_RDRBuf_Len;
    return(BSP_BLE_RDRBuf_Len);
    
}

/**
 *******************************************************************************
 * @brief	 BLE parameter to default.
 * @details     
 * @return      
 * @note       
 *******************************************************************************
 */
static void BSP_BLE_ParameterInit(void)
{
    /*Buffer Related Parameter Initial*/
    BSP_BLE_ClearBleRxBuffer();
    BSP_BLE_ClearBleTxBuffer();

    /*Parameter Initial*/
    BLEStatus                        = BLE_HC42_DISABLE;
    StateFlag                        = 0;

}

/**
 *******************************************************************************
 * @brief	 Set URT module.
 * @details     
 * @return      
 * @note       
 *******************************************************************************
 */
static void BLE_UART_Init(void)
{
    PIN_InitTypeDef  BLE_HC06_PINX;
    URT_BRG_TypeDef  BLE_URT_BRG;
    URT_Data_TypeDef BLE_URT_DataDef;
    
    /* Configure URT IO*/
    //UART_TX
    BLE_HC06_PINX.PINX_Mode               = PINX_Mode_PushPull_O;           // Pin select pusu pull mode
    BLE_HC06_PINX.PINX_PUResistant        = PINX_PUResistant_Enable;        // Enable pull up resistor
    BLE_HC06_PINX.PINX_Speed              = PINX_Speed_Low;                 // Pin output low speed mode
    BLE_HC06_PINX.PINX_OUTDrive           = PINX_OUTDrive_Level0;           // Pin output driver full strength.
    BLE_HC06_PINX.PINX_FilterDivider      = PINX_FilterDivider_Bypass;      // Pin input deglitch filter clock divider bypass
    BLE_HC06_PINX.PINX_Inverse            = PINX_Inverse_Disable;           // Pin input data not inverse
    BLE_HC06_PINX.PINX_Alternate_Function = BLE_HC06TX_AFS;               
    GPIO_PinMode_Config(BLE_HC06TX_PINX,&BLE_HC06_PINX); 
    
    //UART_RX         
    BLE_HC06_PINX.PINX_Mode               = PINX_Mode_Digital_I;            // Setting digital  mode
    BLE_HC06_PINX.PINX_Alternate_Function = BLE_HC06RX_AFS;              
    GPIO_PinMode_Config(BLE_HC06RX_PINX,&BLE_HC06_PINX); 
    
    /*URT Module Inital*/
    //=====Set Clock=====//
    //Set BaudRate
    BLE_URT_BRG.URT_InternalClockSource = URT_BDClock_PROC;
    BLE_URT_BRG.URT_BaudRateMode = URT_BDMode_Separated;
    BLE_URT_BRG.URT_PrescalerCounterReload = 3;	                            //Set PSR
    BLE_URT_BRG.URT_BaudRateCounterReload = 155;	                        //Set RLR
    URT_BaudRateGenerator_Config(bBLE_URTn, &BLE_URT_BRG);   		        //BR9600 = f(CK_URTx)/(PSR+1)/(RLR+1)/(OS_NUM+1)
    URT_BaudRateGenerator_Cmd(bBLE_URTn, ENABLE);	                        //Enable BaudRateGenerator
    //TX/RX Clock                                                           
    URT_TXClockSource_Select(bBLE_URTn, URT_TXClock_Internal);	            //URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(bBLE_URTn, URT_RXClock_Internal);	            //URT_RX use BaudRateGenerator
    URT_TXOverSamplingSampleNumber_Select(bBLE_URTn, 7);	                //Set TX OS_NUM
    URT_RXOverSamplingSampleNumber_Select(bBLE_URTn, 7);	                //Set RX OS_NUM
    URT_RXOverSamplingMode_Select(bBLE_URTn, URT_RXSMP_3TIME);               
    URT_TX_Cmd(bBLE_URTn, ENABLE);	                                        //Enable TX
    URT_RX_Cmd(bBLE_URTn, ENABLE);	                                        //Enable RX
    
    //=====Set Mode=====//
    //Set Data Character
    BLE_URT_DataDef.URT_TX_DataLength  = URT_DataLength_8;
    BLE_URT_DataDef.URT_RX_DataLength  = URT_DataLength_8;
    BLE_URT_DataDef.URT_TX_DataOrder   = URT_DataTyped_LSB;
    BLE_URT_DataDef.URT_RX_DataOrder   = URT_DataTyped_LSB;
    BLE_URT_DataDef.URT_TX_Parity      = URT_Parity_No;
    BLE_URT_DataDef.URT_RX_Parity      = URT_Parity_No;
    BLE_URT_DataDef.URT_TX_StopBits    = URT_StopBits_1_0;
    BLE_URT_DataDef.URT_RX_StopBits    = URT_StopBits_1_0;
    BLE_URT_DataDef.URT_TX_DataInverse = DISABLE;
    BLE_URT_DataDef.URT_RX_DataInverse = DISABLE;
    URT_DataCharacter_Config(bBLE_URTn, &BLE_URT_DataDef);
    //Set Mode
    URT_Mode_Select(bBLE_URTn, URT_URT_mode);
    //Set Data Line
    URT_DataLine_Select(bBLE_URTn, URT_DataLine_2);
    
    //=====Enable URT Interrupt=====//
    URT_IT_Config(bBLE_URTn, (URT_IT_TC | URT_IT_TX | URT_IT_RX) , ENABLE);  // Enable URT TX_IE, RX_IE and TC_IE
    URT_ITEA_Cmd(bBLE_URTn, ENABLE);                                         // Enable URT0 interrupt all enable
    
    //=====Enable URT=====//
    URT_Cmd(bBLE_URTn, ENABLE);
    
}

/**
 *******************************************************************************
 * @brief    Set BLE timebase timer.
 * @details
 * @return      
 * @note        
 *******************************************************************************
 */
static void BLE_TM_Init(void)
{
    TM_TimeBaseInitTypeDef  BLE_TimeBase;
    
    /*Timer initial*/
    //Timer Default initial
    NVIC_DisableIRQ( TM1x_IRQn);
    TM_DeInit(BLE_TM);
    
    //TimeBase structure initial
    TM_TimeBaseStruct_Init(&BLE_TimeBase);
    
    //Set Step motor rotating speed.
    //Speed = TM clock source freq / ((TM_Prescaler + 1) * (TM_Period + 1));
    BLE_TimeBase.TM_MainClockDirection = TM_UpCount;
    BLE_TimeBase.TM_Prescaler          = 240 - 1;                             
    BLE_TimeBase.TM_Period             = 200 - 1;                                 
    BLE_TimeBase.TM_CounterMode        = Cascade;                           // Timer overflow is 1ms in 48MHz.
    TM_TimeBase_Init(BLE_TM, &BLE_TimeBase);
    
    //Interrupt configure
    TM_ClearFlag( BLE_TM, TMx_TOF);
    TM_IT_Config(BLE_TM, TMx_TIE_IE,ENABLE);
    TM_ITEA_Cmd( BLE_TM, ENABLE);
    NVIC_EnableIRQ( TM0x_IRQn);

    //Enable Timer
    TM_Timer_Cmd(BLE_TM,ENABLE);
}

/**
 *******************************************************************************
 * @brief     BLE timebase IRQhandle.
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
static void BSP_BLE_TimeBaseIRQHandler(void)
{
    uint32_t BSP_BLE_TBIRQ_Status;
    uint32_t BSP_BLE_TBIRQ_Delay;
    
    TM_ClearFlag( BLE_TM, (TMx_TOF2|TMx_TOF));

    BSP_BLE_TBIRQ_Status = ((BLEStatus & BLE_HC42_STATEMASK)>>BLE_HC42_STATESHIFT);
    BSP_BLE_TBIRQ_Delay  = (BLEStatus & BLE_HC42_DELAYMASK);

    /*BLE_HC42_DISABLE ~ BLE_HC42_DEVICENAME state is MCU  BLE flow in power on.
      If HC42 already connect to Host will causes the flow fail(HC-42 no response).*/
    switch(BSP_BLE_TBIRQ_Status)
    {
        case BLE_HC42_DISABLE:    
                                  if(BSP_BLE_TBIRQ_Delay==0)
                                  {
                                      /* Test HC-42 connect*/
                                      BSP_BLE_ClearBleRxBuffer();
                                      BSP_BLE_ClearBleTxBuffer();
                                      BSP_BLE_WriteTxBuffer(&AT[0],(sizeof(AT)-1));                   
                                  }
                                  if(BSP_BLE_TBIRQ_Delay<100)                                         // Delay 100ms
                                  {
                                      BLEStatus = BLEStatus + 1;
                                  }
                                  else if((BLE_RX_BUF[0] == 'O')&&(BLE_RX_BUF[1] == 'K'))             // Check HC42 response.
                                  {
                                      /* Get HC-42 version */
                                      BLEStatus = (BLE_HC42_GETVERSION<<BLE_HC42_STATESHIFT);
                                      BSP_BLE_ClearBleRxBuffer();
                                      BSP_BLE_ClearBleTxBuffer();
                                      BSP_BLE_WriteTxBuffer(&AT_VERSION[0], (sizeof(AT_VERSION)-1));  
                                  }
                                  else
                                  {
                                      BLEStatus = BLE_HC42_DISABLE;
                                  }                           
                                  break;
        case BLE_HC42_GETVERSION:
                                  if(BSP_BLE_TBIRQ_Delay<200)                                         // Delay 200ms
                                  {
                                      BLEStatus = BLEStatus + 1;
                                      return;
                                  }
                                  else if((BLE_RX_BUF[0] != 'O')&&(BLE_RX_BUF[1] != 'K')&&(BLE_RX_BUF[1] != '+')&&\
                                          (BLE_RX_BUF[2] != 'V')&&(BLE_RX_BUF[3] != 'E')&&(BLE_RX_BUF[1] != 'R')&&\
                                          (BLE_RX_BUF[1] != 'S')&&(BLE_RX_BUF[1] != 'I')&&(BLE_RX_BUF[1] != 'O')&&\
                                          (BLE_RX_BUF[4] != 'N')&&(BLE_RX_BUF[5] != '=')&&(BLE_RX_BUF[1] != 'H')&&\
                                          (BLE_RX_BUF[6] != 'C')&&(BLE_RX_BUF[7] != '-')&&(BLE_RX_BUF[1] != '4')&&\
                                          (BLE_RX_BUF[8] != '2')&&(BLE_RX_BUF[9] != ' '))             // Check HC42 response 
                                                                                                      // (The follow up version number refers to device version.)
                                  {
                                      StateFlag |= State_VersionFail;                                 
                                  }
                                  /* Set the HC-42 baud rate 9600 */
                                  BLEStatus = (BLE_HC42_SETBAUDRATE<<BLE_HC42_STATESHIFT);
                                  BSP_BLE_ClearBleRxBuffer();
                                  BSP_BLE_ClearBleTxBuffer();
                                  BSP_BLE_WriteTxBuffer(&AT_UART[0],(sizeof(AT_UART)-1));             
                                  break;
        case BLE_HC42_SETBAUDRATE:
                                  if(BSP_BLE_TBIRQ_Delay<100)                                         // Delay 100ms
                                  {
                                      BLEStatus = BLEStatus + 1;
                                      return;
                                  }
                                  else if((BLE_RX_BUF[0]!='O')&&(BLE_RX_BUF[1]!='K')&&(BLE_RX_BUF[1]!='+')&&\
                                          (BLE_RX_BUF[0]!='U')&&(BLE_RX_BUF[0]!='A')&&(BLE_RX_BUF[0]!='R')&&(BLE_RX_BUF[0]!='T')&&(BLE_RX_BUF[0]!='=')&&\
                                          (BLE_RX_BUF[2]!='9')&&(BLE_RX_BUF[3]!='6')&&(BLE_RX_BUF[4]!='0')&&(BLE_RX_BUF[5]!='0'))
                                  {                                                                   // Check HC42 response 
                                      StateFlag |= State_BaudrateFail;                                
                                  }
                                  /* Modify HC-42 RF power*/
                                  BLEStatus = (BLE_HC42_RFPM<<BLE_HC42_STATESHIFT);
                                  BSP_BLE_ClearBleRxBuffer();
                                  BSP_BLE_ClearBleTxBuffer();
                                  BSP_BLE_WriteTxBuffer(&AT_RFPM[0],(sizeof(AT_RFPM)-1));             
                                  break;
        case BLE_HC42_RFPM:
                                  if(BSP_BLE_TBIRQ_Delay<100)                                         // Delay 100ms
                                  {
                                      BLEStatus = BLEStatus + 1;
                                      return;
                                  }
                                  else if((BLE_RX_BUF[0]!='O')&&(BLE_RX_BUF[1]!='K')&&(BLE_RX_BUF[1]!='+')&&\
                                          (BLE_RX_BUF[2]!='R')&&(BLE_RX_BUF[3]!='F')&&(BLE_RX_BUF[4]!='P')&&(BLE_RX_BUF[1]!='M')&&(BLE_RX_BUF[1]!='=')&&\
                                          (BLE_RX_BUF[8]!='0'))
                                  {                                                                   // Check HC42 response
                                      StateFlag |= State_RFPMFail;                                    
                                  }
                                  /* Set HC-42 broadcast interval.*/
                                  BLEStatus = (BLE_HC42_AINT<<BLE_HC42_STATESHIFT);
                                  BSP_BLE_ClearBleRxBuffer();
                                  BSP_BLE_ClearBleTxBuffer();
                                  BSP_BLE_WriteTxBuffer(&AT_AINT[0],(sizeof(AT_AINT)-1)); 
                                  break;
        case BLE_HC42_AINT:
                                  if(BSP_BLE_TBIRQ_Delay<100)                                         // Delay 100ms
                                  {
                                      BLEStatus = BLEStatus + 1;
                                      return;
                                  }
                                  else if((BLE_RX_BUF[0]!='O')&&(BLE_RX_BUF[1]!='K')&&(BLE_RX_BUF[1]!='+')&&\
                                          (BLE_RX_BUF[2]!='A')&&(BLE_RX_BUF[3]!='I')&&(BLE_RX_BUF[4]!='N')&&(BLE_RX_BUF[1]!='T')&&(BLE_RX_BUF[1]!='=')&&\
                                          (BLE_RX_BUF[8]!='2')&&(BLE_RX_BUF[3]!='0')&&(BLE_RX_BUF[3]!='0'))
                                  {                                                                   // Check HC42 response
                                      StateFlag |= State_RFAINTFail;                                  
                                  }
                                  /* Setting HC-42 BLE device name */
                                  BLEStatus = (BLE_HC42_DEVICENAME<<BLE_HC42_STATESHIFT);
                                  BSP_BLE_ClearBleRxBuffer();
                                  BSP_BLE_ClearBleTxBuffer();
                                  BSP_BLE_WriteTxBuffer(&AT_NAME[0],(sizeof(AT_NAME)-1));             
                                  break;
        case BLE_HC42_DEVICENAME:
                                  if(BSP_BLE_TBIRQ_Delay<100)                                         // Delay 100ms
                                  {
                                      BLEStatus = BLEStatus + 1;
                                      return;
                                  }
                                  else if((BLE_RX_BUF[0]!='O')&&(BLE_RX_BUF[1]!='K')&&(BLE_RX_BUF[1]!='+')&&\
                                          (BLE_RX_BUF[2]!='N')&&(BLE_RX_BUF[3]!='A')&&(BLE_RX_BUF[4]!='M')&&(BLE_RX_BUF[1]!='E')&&(BLE_RX_BUF[1]!='=')&&\
                                          (BLE_RX_BUF[5]!='Z')&&(BLE_RX_BUF[6]!='L')&&(BLE_RX_BUF[7]!='_')&&\
                                          (BLE_RX_BUF[8]!='H')&&(BLE_RX_BUF[6]!='C')&&(BLE_RX_BUF[6]!='-')&&(BLE_RX_BUF[6]!='4')&&(BLE_RX_BUF[6]!='2'))
                                  {                                                                   // Check HC42 response
                                      StateFlag |= State_NameFail;                                    
                                  }
                                  BSP_BLE_ClearBleRxBuffer();
                                  BLEStatus = (BLE_HC42_CONNECT<<BLE_HC42_STATESHIFT);
                                  break;
        case BLE_HC42_CONNECT:
                                  if(BLE_RX_BUF_IndexCnt!=0)
                                  {
                                      BLE_RXF = 1;                                                    /* Trigger BLE RX Flag */
                                      BSP_BLE_ReadCallback();
                                  }
                                  break;
        default:
                                  break;
    }
    /*Detect if there is data needs to transmit.*/
    BSP_BLE_TriggerSendData();
 
}
/**
 *******************************************************************************
 * @brief       Send BLE_TX_BUF data
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
static void BSP_BLE_TriggerSendData(void) 
{
    uint8_t BSP_BLE_TSDTmp;
    
    if(BLE_TX_BUF_TXF==1 || BLE_TX_BUF_IndexCnt==0)
    {
        return;
    }
    
    BLE_TX_BUF_TXF = 1;
    BLE_TX_BUF_IndexCnt--;
    BSP_BLE_TSDTmp = BLE_TX_BUF[BLE_TX_BUF_oIndex];
    if(BLE_TX_BUF_oIndex<(BufferSize-1))
    {
        BLE_TX_BUF_oIndex++;
    }
    else
    {
        BLE_TX_BUF_oIndex = 0;
    }
    bBLE_URTn->TDAT.B[0] = BSP_BLE_TSDTmp;
}    

/**
 *******************************************************************************
 * @brief    BLE RX buffer overrun callback function.
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
__WEAK void BSP_BLE_RxOverRunCallback(void)
{
    //=========================================================
    //NOTE : This function should not be modified , when the 
    //       callback is needed, the MID_URT_ErrorCallback can
    //       be implemented in the user file. 
    
    __NOP();
}

/**
 *******************************************************************************
 * @brief    
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
__WEAK void BSP_BLE_ReadCallback(void)
{
    //=========================================================
    //NOTE : This function should not be modified , when the 
    //       callback is needed, the MID_URT_ErrorCallback can
    //       be implemented in the user file. 
    
    __NOP();
}

#endif


/**
 *******************************************************************************
 * @brief   Prevent the file is empty compilcation warning. 
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_HC42PreventComilcationWarning(void);

__WEAK void BSP_BLE_HC42PreventComilcationWarning(void)
{
    __NOP();
}




