/**
  ******************************************************************************
 *
 * @file        Sample_SPI_Slave_StandardSPI_DMA_M2P.c
 *
 * @brief       Standard-SPI slave mode DMA SRAM-to-SPI sample code.
 *
 * @par         Project
 *              MG32
 * @version     V2.15
 * @date        2022/09/21
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2021 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_GPIO_DRV.h"
#include "MG32_SPI_DRV.h"
#include "MG32_DMA_DRV.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define MYBINARYIMAGE2_LENGTH   11
#define SPI_NSS                 PB0     // SPI_NSS

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
#if (defined(MG32F02A132) || defined(MG32_3RD))
static uint8_t SlvSrc[MYBINARYIMAGE2_LENGTH] __attribute__((section(".ARM.__at_0x20003820"))) = 
{0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B};
#else
static uint8_t SlvSrc[MYBINARYIMAGE2_LENGTH] = 
{0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B};
#endif                            

/* Private function prototypes -----------------------------------------------*/
void Sample_SPI_Slave_StandardSPI_DMA_M2P (SPI_Struct* SPIx);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief  	    SPI Slave use DMA receive
 * @details  
 * @param[in]   SPIx
 * 	@arg\b			SPI0
 * @return      None
 * @note        
 * @par         Example
 * @code
    SPI_Slave_Use_DMA_Transmit();
 * @endcode
 *******************************************************************************
 */
void Sample_SPI_Slave_StandardSPI_DMA_M2P (SPI_Struct* SPIx)
{
    uint8_t             CNT;
    DMA_BaseInitTypeDef DMATestPattern;
    PIN_InitTypeDef     PINX_InitStruct;


    
    // SPI default initialization 
    SPI_DeInit(SPIx);
    // CLK configure
    SPI_PreDivider_Select(SPIx, SPI_PDIV_1);
    SPI_Prescaler_Select(SPIx, SPI_PSC_1);
    SPI_Divider_Select(SPIx, SPI_DIV_2);
    SPI_Clock_Select(SPIx, SPI_CK_SPIx_PR);
    // Standard SPI, slave with NSS, first bit LSB, CPHA =0, CPOL=0
    SPI_DataLine_Select(SPIx, SPI_Standard);
    SPI_ModeAndNss_Select(SPIx, SPI_SlaveWithNss);         
    SPI_FirstBit_Select(SPIx, SPI_LSB);
    // SPI transfer data size 8-bits
    SPI_DataSize_Select(SPIx, SPI_8bits);
    // Data bidirectional output
    SPI_BidirectionalOutput_Cmd(SPIx, ENABLE);
    // SPI receive threshold 1 byte
    SPI_SlaveModeReceivedThreshold_Select(SPIx, SPI_1Byte);
    
    // IO initialPIN_InitTypeDef PINX_InitStruct;
    SPI_NSS = 1;
    PINX_InitStruct.PINX_Mode				= PINX_Mode_Digital_I;          // Pin select digital input mode
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;      // Enable pull up resistor
    PINX_InitStruct.PINX_Speed              = PINX_Speed_High;              // Output high speed mode enable 
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;         // Pin output driver full strength.
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;    // Pin input deglitch filter clock divider bypass
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;         // Pin input data not inverse
    PINX_InitStruct.PINX_Alternate_Function = 2;                            // Pin AFS = 2
    
    GPIO_PinMode_Config(PINB(0),&PINX_InitStruct);                          // NSS setup at PB0
    GPIO_PinMode_Config(PINB(2),&PINX_InitStruct);                          // CLK setup at PB2
    GPIO_PinMode_Config(PINB(3),&PINX_InitStruct);                          // MOSI setup at PB3
    
    PINX_InitStruct.PINX_Mode				= PINX_Mode_PushPull_O;         // Setting pusu pull mode
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Disable;     // Enable pull up resistor
    GPIO_PinMode_Config(PINB(1),&PINX_InitStruct);                          // MISO setup at PB1
    
    // Enable SPI function
    SPI_Cmd(SPIx, ENABLE);

    // Enable DMA
    DMA_Cmd(ENABLE);
    // Enable Channel-0
    DMA_Channel_Cmd(DMAChannel0, ENABLE);
    // prepare test pattern (???????,? Flash to SRAM)
    for (CNT=0; CNT<MYBINARYIMAGE2_LENGTH; CNT++)
    {
        SlvSrc[CNT] = CNT;
    }
    
    // initial parameter of structure
    DMA_BaseInitStructure_Init(&DMATestPattern);

    // modify parameter
    {   
        // external trigger pin and mode config
        DMATestPattern.ExtTrgPin = DMA_ExtTRG0;
        DMATestPattern.ExtTrgMode = DMA_DisableExtTrg;
        // DMA channel select
        DMATestPattern.DMAChx = DMAChannel0;
        // channel x loop mode config
        DMATestPattern.DMALoopCmd = DISABLE;
        // channel x source/destination auto increase address
        DMATestPattern.SrcSINCSel = ENABLE;        
        DMATestPattern.DestDINCSel = DISABLE;
        // DMA Burst size config
        DMATestPattern.BurstDataSize = DMA_BurstSize_4Byte;
        // DMA transfer data count initial number (? Bytes ???)
        DMATestPattern.DMATransferNUM = MYBINARYIMAGE2_LENGTH;
        // DMA source peripheral config
        DMATestPattern.SrcSymSel = DMA_MEM_Read;
        // DMA destination peripheral config
        DMATestPattern.DestSymSel = DMA_SPI0_TX;
        // source config
        DMATestPattern.DMASourceAddr = (uint8_t *)&SlvSrc;
    }
    // M2P(CH0) simple parameter
    DMA_Base_Init(&DMATestPattern);

    for(CNT=3; CNT>0; CNT--)
    {
        // start request
        DMA_StartRequest(DMAChannel0);
        // Enable SPI TX DMA
        SPI_TXDMA_Cmd(SPIx, ENABLE);    
        // Waitting DMA trsnafer half
        while(DMA_GetSingleFlagStatus(DMA, DMA_FLAG_CH0_THF) == DRV_UnHappened);
        // Clear SPI0 IDLF flag
        SPI_ClearFlag(SPI0, SPI_IDLF);
        // Waitting DMA trsnafer complete
        while(DMA_GetSingleFlagStatus(DMA, DMA_FLAG_CH0_TCF) == DRV_UnHappened);
        // Clear DMA event flag
        DMA_ClearFlag(DMA, (DMA_FLAG_CH0_TCF | DMA_FLAG_CH0_THF));
        // Waitting SPI trsnafer complete
        while(SPI_GetSingleFlagStatus(SPIx, SPI_TCF) == DRV_UnHappened);
        // Waitting transfer  complete.
        while(SPI_GetSingleFlagStatus(SPI0, SPI_IDLF) == DRV_UnHappened);
        // Clear SPI event flag
        SPI_ClearFlag(SPIx, SPI_ALLF);
    }
    // Disable SPI
    SPI_Cmd(SPIx, DISABLE);       
}



