


/**
 ******************************************************************************
 *
 * @file        MG32_ARGB_Drama.c
 * @brief       
 
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2024/10/04
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32__Common_DRV.h"
#include "MG32_ARGB.h"
#include "MG32_ARGB_Drama.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/




/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
ARGBDramaCTR_TypeDef   ARGB_Drama;                                             /*!< Control ARGB parameter.*/


uint8_t                ARGB_DramaBuf[SMP_ARGB_PIECE][SMP_ARGB_DATASIZE];   /*!<.The buffer is use for ARGB calculate.*/



/* Private function prototypes -----------------------------------------------*/
static void MG32_ARGB_PercentageTable( uint8_t* ARGB_PEBuf, uint32_t molecular ,uint32_t Denominator, uint8_t LoopTime);
                                                                  
static void MG32_ARGB_SignalColorDrama(ARGBDramaCTR_TypeDef* sARGB);
static void MG32_ARGB_BreathDrama(ARGBDramaCTR_TypeDef* sARGB);                                                                 
static void MG32_ARGB_RainBowDrama(ARGBDramaCTR_TypeDef* sARGB);
static void MG32_ARGB_MapphingDrama(ARGBDramaCTR_TypeDef* sARGB);
static void MG32_ARGB_WDropDrama( ARGBDramaCTR_TypeDef* sARGB );
static void MG32_ARGB_MovingDrama( ARGBDramaCTR_TypeDef* sARGB );
static void MG32_ARGB_TwoMovingDrama( ARGBDramaCTR_TypeDef* sARGB );
//void ARGB_UpdateDrama( ARGBDramaCTR_TypeDef* sARGB );
                                                                  
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/



                                                                            
/**
 *******************************************************************************
 * @brief	    ARGB drama parameter inital.
 * @details     
 * @return      
 * @exception   No
 * @note 
 *******************************************************************************
 */
void MG32_ARGBDrama_Init(void)
{    
    ARGB_Drama.DramaBuf     = &ARGB_DramaBuf[0][0]; 
    
    ARGB_Drama.DramaMode    = ARGB_DRAMA_STATIC;
    
    MG32_ARGBDrama_ClearBuf(&ARGB_Drama);
}

/**
 *******************************************************************************
 * @brief	   Calculate ARGB next data according to ARGB mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .
 * @note       
 *******************************************************************************
 */
void MG32_ARGBDrama_UpdateData( ARGBDramaCTR_TypeDef* sARGB )
{   
    if( sARGB->UpdateTRG & ARGB_UPDATE_DATA)
    {
        return;
    }
    else
    {
        switch( sARGB->DramaMode)
        {
            case ARGB_DRAMA_STATIC:
                                       MG32_ARGB_SignalColorDrama(sARGB);
                                       break;
            case ARGB_DRAMA_BREATH:    
                                       MG32_ARGB_BreathDrama( sARGB);
                                       break;
            case ARGB_DRAMA_RAINBOW:   
                                       MG32_ARGB_RainBowDrama( sARGB);
                                       break;
            case ARGB_DRAMA_MORPHING:  
                                       MG32_ARGB_MapphingDrama( sARGB);
                                       break;
            case ARGB_DRAMA_WDROP:     
                                       MG32_ARGB_WDropDrama(sARGB);
                                       break;
            case ARGB_DRAMA_MOVING:    
                                       MG32_ARGB_MovingDrama(sARGB);
                                       break;
            case ARGB_DRAMA_2MOVING:   
                                       MG32_ARGB_TwoMovingDrama(sARGB);
                                       break;
            case ARGB_DRAMA_DISABLE:
            default: 
                                       break;
        }
    } 
} 

/**
 *******************************************************************************
 * @brief	   Update the sARGB line mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .
 * @return      
 * @note       
 *******************************************************************************
 */
void MG32_ARGBDrama_UpdateMode( ARGBDramaCTR_TypeDef* sARGB )
{
    sARGB->DramaAct       = 0;  
    sARGB->DramaScene     = 0;    
    sARGB->DramaTime      = 0;    
    
    switch( sARGB->DramaMode)
    {
        case ARGB_DRAMA_STATIC:
                                    sARGB->DramaActMax    = 1;
                                    sARGB->DramaSceneMax  = 0; 
                                    sARGB->DramaTimeMax   = 0;
                                    break;
        case ARGB_DRAMA_BREATH:     
                                    sARGB->DramaActMax    = (SMP_ARGB_DEFAULT_COLOR - 1) * 2;
                                    sARGB->DramaSceneMax  = 100; 
                                    sARGB->DramaTimeMax   = 0;
                                    break;
        case ARGB_DRAMA_RAINBOW:    
                                    sARGB->DramaActMax    = 6;
                                    sARGB->DramaSceneMax  = SMP_ARGB_COLLOR_MAX;
                                    sARGB->DramaTimeMax   = 4;
                                    break;
        case ARGB_DRAMA_MORPHING:   
                                    sARGB->DramaActMax    = SMP_ARGB_DEFAULT_COLOR;
                                    sARGB->DramaSceneMax  = 100; 
                                    sARGB->DramaTimeMax   = 0;
                                    break;
        case ARGB_DRAMA_WDROP:      
                                    sARGB->DramaActMax    = 18;
                                    sARGB->DramaSceneMax  = 4;                          
                                    sARGB->DramaTimeMax   = 3;
                                    break;
        case ARGB_DRAMA_MOVING:     
                                    sARGB->DramaActMax    = ( SMP_ARGB_DEFAULT_COLOR * 2);
                                    sARGB->DramaSceneMax  = SMP_ARGB_PIECE;                          
                                    sARGB->DramaTimeMax   = 3;
                                    break;
        case ARGB_DRAMA_2MOVING:          
                                    sARGB->DramaActMax    = SMP_ARGB_DEFAULT_COLOR * 2;
                                    sARGB->DramaSceneMax  = (SMP_ARGB_PIECE - 1);            
                                    sARGB->DramaTimeMax   = 3;
                                    break;
        case ARGB_DRAMA_DISABLE:
        default:
                                    break;
    }
    
    sARGB->UpdateTRG = 0;
    
    MG32_ARGBDrama_UpdateModeCallback();
    

} 
/**
 *******************************************************************************
 * @brief	   To trigger change the ARGBs mode.  
 * @details     
 * @param[in]  ARGB_Mode : New ARGB mode.  
 * @note       
 *******************************************************************************
 */
void MG32_ARGBDrama_ChangeMode( uint8_t  ARGB_Mode)
{
    ARGB_Drama.DramaMode  = ARGB_Mode;

    ARGB_Drama.UpdateTRG  = ARGB_UPDATE_MODE;
}


/**
 *******************************************************************************
 * @brief	    Clear 
 * @details     
 * @return      
 * @exception   No
 * @note 
 *******************************************************************************
 */
void MG32_ARGBDrama_ClearBuf( ARGBDramaCTR_TypeDef* sARGB )
{
    uint32_t ARGBDrama_ClrBufTmp;
    
    for(ARGBDrama_ClrBufTmp = 0;ARGBDrama_ClrBufTmp <(SMP_ARGB_PIECE*3);ARGBDrama_ClrBufTmp++)
    {
        sARGB->DramaBuf[ARGBDrama_ClrBufTmp] = 0x00;
    }
}

/**
 *******************************************************************************
 * @brief	   Calculate color.
 * @details     
 * @param[in]  ARGB_PEBuf  : Refer to Color.
 * @param[in]  molecular   : Calculate molecular.
 * @param[in]  Denominator : Calculate Denominator.
 * @param[in]  LoopTime    : Calculate time.  
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_PercentageTable( uint8_t* ARGB_PEBuf, uint32_t molecular ,uint32_t Denominator, uint8_t LoopTime)
{
    uint8_t  SMP_ARGB_PETmp;
    uint8_t  SMP_ARGB_PETmp2;
    uint32_t SMP_ARGB_PEDataTmp;
    
    for( SMP_ARGB_PETmp2 = 0; SMP_ARGB_PETmp2 < LoopTime ; SMP_ARGB_PETmp2++)
    {
        for(SMP_ARGB_PETmp = 0; SMP_ARGB_PETmp < 3; SMP_ARGB_PETmp++)
        {
            SMP_ARGB_PEDataTmp         = molecular * ARGB_PEBuf[SMP_ARGB_PETmp];
            ARGB_PEBuf[SMP_ARGB_PETmp] = (uint8_t)(SMP_ARGB_PEDataTmp / Denominator);
        }
    }
}

/**
 *******************************************************************************
 * @brief	   Calculate ARGB next data in ARGB_DRAMA_STATIC mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .     
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_SignalColorDrama(ARGBDramaCTR_TypeDef* sARGB)
{
    uint32_t SMP_ARGB_SIGDTmp;
    uint32_t SMP_ARGB_SIGDAdr;
    
    if( sARGB->DramaActMax !=0)
    {
        sARGB->DramaActMax = 0;
        
        for( SMP_ARGB_SIGDTmp = 0; SMP_ARGB_SIGDTmp < SMP_ARGB_PIECE ; SMP_ARGB_SIGDTmp++)
        {
            SMP_ARGB_SIGDAdr = (SMP_ARGB_SIGDTmp * 3);
            
            sARGB->DramaBuf[ SMP_ARGB_SIGDAdr + SMP_ARGB_G] = SMP_ARGBDefault_TABLE[0][ SMP_ARGB_G];
            sARGB->DramaBuf[ SMP_ARGB_SIGDAdr + SMP_ARGB_R] = SMP_ARGBDefault_TABLE[0][ SMP_ARGB_R];
            sARGB->DramaBuf[ SMP_ARGB_SIGDAdr + SMP_ARGB_B] = SMP_ARGBDefault_TABLE[0][ SMP_ARGB_B];
        }
        
        sARGB->UpdateTRG |= ARGB_UPDATE_DATA;
    }
}

/**
 *******************************************************************************
 * @brief	   Calculate ARGB next data in ARGB_DRAMA_BREATH mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .   
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_BreathDrama(ARGBDramaCTR_TypeDef* sARGB)
{
    uint32_t SMP_ARGB_BRDTmp;

    
    sARGB->DramaTmp[SMP_ARGB_G] = SMP_ARGBDefault_TABLE[(sARGB->DramaAct >> 1)][SMP_ARGB_G];
    sARGB->DramaTmp[SMP_ARGB_R] = SMP_ARGBDefault_TABLE[(sARGB->DramaAct >> 1)][SMP_ARGB_R];
    sARGB->DramaTmp[SMP_ARGB_B] = SMP_ARGBDefault_TABLE[(sARGB->DramaAct >> 1)][SMP_ARGB_B];
    
    if( sARGB->DramaAct & 0x01)
    {
        MG32_ARGB_PercentageTable( &sARGB->DramaTmp[0] , ( sARGB->DramaSceneMax -  sARGB->DramaScene ) , sARGB->DramaSceneMax ,2);
    }
    else
    {
        MG32_ARGB_PercentageTable( &sARGB->DramaTmp[0] , sARGB->DramaScene, sARGB->DramaSceneMax , 2);
    }
    
    
    if( sARGB->DramaScene == sARGB->DramaSceneMax)
    {
        if( sARGB->DramaAct > sARGB->DramaActMax)
        {
            sARGB->DramaAct = 0;
        }
        else
        {
            sARGB->DramaAct = sARGB->DramaAct + 1;
        }
        sARGB->DramaScene = 0;        
    }
    else
    {
        sARGB->DramaScene = sARGB->DramaScene + 1;
    }
    
    for( SMP_ARGB_BRDTmp = 0; SMP_ARGB_BRDTmp < SMP_ARGB_PIECE ; SMP_ARGB_BRDTmp++)
    {
        sARGB->DramaBuf[ (SMP_ARGB_BRDTmp * 3) + SMP_ARGB_G] = sARGB->DramaTmp[ SMP_ARGB_G];
        sARGB->DramaBuf[ (SMP_ARGB_BRDTmp * 3) + SMP_ARGB_R] = sARGB->DramaTmp[ SMP_ARGB_R];
        sARGB->DramaBuf[ (SMP_ARGB_BRDTmp * 3) + SMP_ARGB_B] = sARGB->DramaTmp[ SMP_ARGB_B];
    }
    
    sARGB->UpdateTRG |= ARGB_UPDATE_DATA;
}

/**
 *******************************************************************************
 * @brief	   Calculate ARGB next data in ARGB_DRAMA_RAINBOW mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .     
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_RainBowDrama(ARGBDramaCTR_TypeDef* sARGB)
{
    uint32_t SMP_ARGB_RBDTmp;
    
    for( SMP_ARGB_RBDTmp = 0; SMP_ARGB_RBDTmp < (SMP_ARGB_PIECE - 1); SMP_ARGB_RBDTmp++)
    {
        sARGB->DramaBuf[ (SMP_ARGB_RBDTmp * 3) + SMP_ARGB_G] = sARGB->DramaBuf[ ((SMP_ARGB_RBDTmp + 1)* 3) + SMP_ARGB_G];
        sARGB->DramaBuf[ (SMP_ARGB_RBDTmp * 3) + SMP_ARGB_R] = sARGB->DramaBuf[ ((SMP_ARGB_RBDTmp + 1)* 3) + SMP_ARGB_R];
        sARGB->DramaBuf[ (SMP_ARGB_RBDTmp * 3) + SMP_ARGB_B] = sARGB->DramaBuf[ ((SMP_ARGB_RBDTmp + 1)* 3) + SMP_ARGB_B];
    }
    
    SMP_ARGB_RBDTmp = 0;
    
    switch(sARGB->DramaAct)
    {
        case 2:
        case 5:
                SMP_ARGB_RBDTmp = 1;
                break;
        case 1:
        case 4:
                SMP_ARGB_RBDTmp = 2;
                break;
        case 0:
        case 3:
                break;
        default:
                return;
    }
    
    if( (sARGB->DramaAct & 0x01)==0)
    {
        sARGB->DramaScene = sARGB->DramaScene + sARGB->DramaTimeMax;
        sARGB->DramaBuf[ ((SMP_ARGB_PIECE - 1) * 3) + SMP_ARGB_RBDTmp] = (uint8_t)sARGB->DramaScene;
        
        if( sARGB->DramaScene > sARGB->DramaSceneMax)
        {
            sARGB->DramaAct ++;
        }
    }
    else
    {
        sARGB->DramaScene = sARGB->DramaScene - sARGB->DramaTimeMax;
        sARGB->DramaBuf[ ((SMP_ARGB_PIECE - 1) * 3) + SMP_ARGB_RBDTmp] = (uint8_t)sARGB->DramaScene;
        if( sARGB->DramaScene == 0)
        {
            sARGB->DramaAct ++;
            
            if( sARGB->DramaAct == sARGB->DramaActMax)
            {
                sARGB->DramaAct = 0;
            }
        }
    }
    
    sARGB->UpdateTRG |= ARGB_UPDATE_DATA;
}

/**
 *******************************************************************************
 * @brief	   Calculate ARGB next data in ARGB_DRAMA_MORPHING mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .     
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_MapphingDrama(ARGBDramaCTR_TypeDef* sARGB)
{
    uint32_t SMP_ARGB_MPDTmp;
    uint8_t  SMP_ARGB_COLOR1[3];
    uint8_t  SMP_ARGB_COLOR2[3];
    
    //========================================================================================
    sARGB->DramaTmp[SMP_ARGB_G] = SMP_ARGBDefault_TABLE[ sARGB->DramaAct  ][SMP_ARGB_G];
    sARGB->DramaTmp[SMP_ARGB_R] = SMP_ARGBDefault_TABLE[ sARGB->DramaAct  ][SMP_ARGB_R];
    sARGB->DramaTmp[SMP_ARGB_B] = SMP_ARGBDefault_TABLE[ sARGB->DramaAct  ][SMP_ARGB_B];
    
    MG32_ARGB_PercentageTable( &sARGB->DramaTmp[0] , ( sARGB->DramaSceneMax -  sARGB->DramaScene ) , sARGB->DramaSceneMax ,1);
    
    SMP_ARGB_COLOR1[SMP_ARGB_G] = sARGB->DramaTmp[SMP_ARGB_G];
    SMP_ARGB_COLOR1[SMP_ARGB_R] = sARGB->DramaTmp[SMP_ARGB_R];
    SMP_ARGB_COLOR1[SMP_ARGB_B] = sARGB->DramaTmp[SMP_ARGB_B];
    
    //========================================================================================
    if( sARGB->DramaAct == 5)
    {
        sARGB->DramaTmp[SMP_ARGB_G] = SMP_ARGBDefault_TABLE[ 0 ][SMP_ARGB_G];
        sARGB->DramaTmp[SMP_ARGB_R] = SMP_ARGBDefault_TABLE[ 0 ][SMP_ARGB_R];
        sARGB->DramaTmp[SMP_ARGB_B] = SMP_ARGBDefault_TABLE[ 0 ][SMP_ARGB_B];
    }
    else
    {
        sARGB->DramaTmp[SMP_ARGB_G] = SMP_ARGBDefault_TABLE[ (sARGB->DramaAct + 1) ][SMP_ARGB_G];
        sARGB->DramaTmp[SMP_ARGB_R] = SMP_ARGBDefault_TABLE[ (sARGB->DramaAct + 1) ][SMP_ARGB_R];
        sARGB->DramaTmp[SMP_ARGB_B] = SMP_ARGBDefault_TABLE[ (sARGB->DramaAct + 1) ][SMP_ARGB_B];
    }
    
    MG32_ARGB_PercentageTable( &sARGB->DramaTmp[0] , sARGB->DramaScene, sARGB->DramaSceneMax , 1);
    
    SMP_ARGB_COLOR2[SMP_ARGB_G] = sARGB->DramaTmp[SMP_ARGB_G];
    SMP_ARGB_COLOR2[SMP_ARGB_R] = sARGB->DramaTmp[SMP_ARGB_R];
    SMP_ARGB_COLOR2[SMP_ARGB_B] = sARGB->DramaTmp[SMP_ARGB_B];
    
    
    //========================================================================================
    
    sARGB->DramaTmp[ SMP_ARGB_G] = SMP_ARGB_COLOR1[ SMP_ARGB_G ] + SMP_ARGB_COLOR2[ SMP_ARGB_G];
    sARGB->DramaTmp[ SMP_ARGB_R] = SMP_ARGB_COLOR1[ SMP_ARGB_R ] + SMP_ARGB_COLOR2[ SMP_ARGB_R];
    sARGB->DramaTmp[ SMP_ARGB_B] = SMP_ARGB_COLOR1[ SMP_ARGB_B ] + SMP_ARGB_COLOR2[ SMP_ARGB_B];
    
    
    if( sARGB->DramaScene < sARGB->DramaSceneMax)
    {
        sARGB->DramaScene = sARGB->DramaScene + 1;
    }
    else
    {
        sARGB->DramaScene = 0;
        
        if( sARGB->DramaAct < (sARGB->DramaActMax - 1))
        {
            sARGB->DramaAct = sARGB->DramaAct + 1;
        }
        else
        {
            sARGB->DramaAct = 0;
        }
        
    }
    
    for( SMP_ARGB_MPDTmp = 0; SMP_ARGB_MPDTmp < SMP_ARGB_PIECE ; SMP_ARGB_MPDTmp++)
    {
        sARGB->DramaBuf[ (SMP_ARGB_MPDTmp * 3) + SMP_ARGB_G] = sARGB->DramaTmp[ SMP_ARGB_G];
        sARGB->DramaBuf[ (SMP_ARGB_MPDTmp * 3) + SMP_ARGB_R] = sARGB->DramaTmp[ SMP_ARGB_R];
        sARGB->DramaBuf[ (SMP_ARGB_MPDTmp * 3) + SMP_ARGB_B] = sARGB->DramaTmp[ SMP_ARGB_B];
    }
    
    sARGB->UpdateTRG |= ARGB_UPDATE_DATA;
   
}
/**
 *******************************************************************************
 * @brief	   Calculate ARGB next data in ARGB_DRAMA_WDROP mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .     
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_WDropDrama( ARGBDramaCTR_TypeDef* sARGB )
{
    uint32_t SMP_ARGB_WVDTmp;

    if( sARGB->DramaTime < ( sARGB->DramaTimeMax - 1))
    {
        sARGB->DramaTime = sARGB->DramaTime + 1;
    }
    else
    {
        sARGB->DramaTime = 0;
        
        for( SMP_ARGB_WVDTmp = 0; SMP_ARGB_WVDTmp < (SMP_ARGB_PIECE - 1); SMP_ARGB_WVDTmp++)
        {
            sARGB->DramaBuf[ (SMP_ARGB_WVDTmp * 3) + SMP_ARGB_G] = sARGB->DramaBuf[ ((SMP_ARGB_WVDTmp + 1)* 3) + SMP_ARGB_G];
            sARGB->DramaBuf[ (SMP_ARGB_WVDTmp * 3) + SMP_ARGB_R] = sARGB->DramaBuf[ ((SMP_ARGB_WVDTmp + 1)* 3) + SMP_ARGB_R];
            sARGB->DramaBuf[ (SMP_ARGB_WVDTmp * 3) + SMP_ARGB_B] = sARGB->DramaBuf[ ((SMP_ARGB_WVDTmp + 1)* 3) + SMP_ARGB_B];
        }
        
        
        if( sARGB->DramaScene < sARGB->DramaSceneMax)
        {
            sARGB->DramaScene = sARGB->DramaScene +  1;
            
            sARGB->DramaBuf[ (( SMP_ARGB_PIECE - 1)* 3) + SMP_ARGB_G ] = 0;
            sARGB->DramaBuf[ (( SMP_ARGB_PIECE - 1)* 3) + SMP_ARGB_R ] = 0;    
            sARGB->DramaBuf[ (( SMP_ARGB_PIECE - 1)* 3) + SMP_ARGB_B ] = 0;
        }        
        else
        {
            sARGB->DramaTmp[SMP_ARGB_G] = SMP_ARGBDefault_TABLE[ 0 ][SMP_ARGB_G];
            sARGB->DramaTmp[SMP_ARGB_R] = SMP_ARGBDefault_TABLE[ 0 ][SMP_ARGB_R];
            sARGB->DramaTmp[SMP_ARGB_B] = SMP_ARGBDefault_TABLE[ 0 ][SMP_ARGB_B];
            
            
            MG32_ARGB_PercentageTable( &sARGB->DramaTmp[0] , ( sARGB->DramaActMax -  sARGB->DramaAct ) , sARGB->DramaActMax ,4);
            
            
            sARGB->DramaBuf[ (( SMP_ARGB_PIECE - 1)* 3) + SMP_ARGB_G ] = sARGB->DramaTmp[ SMP_ARGB_G ];
            sARGB->DramaBuf[ (( SMP_ARGB_PIECE - 1)* 3) + SMP_ARGB_R ] = sARGB->DramaTmp[ SMP_ARGB_R ];    
            sARGB->DramaBuf[ (( SMP_ARGB_PIECE - 1)* 3) + SMP_ARGB_B ] = sARGB->DramaTmp[ SMP_ARGB_B ];
            
            
            if( sARGB->DramaAct < sARGB->DramaActMax)
            {
                sARGB->DramaAct = sARGB->DramaAct + 1;
            }
            else
            {
                sARGB->DramaAct = 0;
                sARGB->DramaScene  = 0;
            }
            
            
        }
        sARGB->UpdateTRG |= ARGB_UPDATE_DATA;
    }
}

 /**
 *******************************************************************************
 * @brief	   Calculate ARGB next data in ARGB_DRAMA_MOVING mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .      
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_MovingDrama( ARGBDramaCTR_TypeDef* sARGB )
{
    uint8_t    SMP_ARGB_MOVDColor;
    uint32_t   SMP_ARGB_MOVDUpdate;
    
    if( sARGB->DramaTime < sARGB->DramaTimeMax)
    {
        sARGB->DramaTime = sARGB->DramaTime + 1;
    }
    else
    {
        sARGB->DramaTime = 0;
        
        if( sARGB->DramaAct & 0x01)
        {
            SMP_ARGB_MOVDUpdate = (sARGB->DramaScene* 3);
            
            sARGB->DramaBuf[ SMP_ARGB_MOVDUpdate + SMP_ARGB_G ] = 0;
            sARGB->DramaBuf[ SMP_ARGB_MOVDUpdate + SMP_ARGB_R ] = 0;    
            sARGB->DramaBuf[ SMP_ARGB_MOVDUpdate + SMP_ARGB_B ] = 0;
        }
        else
        {
            SMP_ARGB_MOVDColor = sARGB->DramaAct >> 1;
            SMP_ARGB_MOVDUpdate = ((( sARGB->DramaSceneMax - 1) - sARGB->DramaScene )* 3);
            
            sARGB->DramaBuf[ SMP_ARGB_MOVDUpdate + SMP_ARGB_G ] = SMP_ARGBDefault_TABLE[ SMP_ARGB_MOVDColor ][ SMP_ARGB_G ];
            sARGB->DramaBuf[ SMP_ARGB_MOVDUpdate + SMP_ARGB_R ] = SMP_ARGBDefault_TABLE[ SMP_ARGB_MOVDColor ][ SMP_ARGB_R ];    
            sARGB->DramaBuf[ SMP_ARGB_MOVDUpdate + SMP_ARGB_B ] = SMP_ARGBDefault_TABLE[ SMP_ARGB_MOVDColor ][ SMP_ARGB_B ];
            
        }
        
        if( sARGB->DramaScene < (sARGB->DramaSceneMax - 1))
        {
            sARGB->DramaScene = sARGB->DramaScene + 1;
        }
        else
        {
            sARGB->DramaScene = 0;
            
            if( sARGB->DramaAct < (sARGB->DramaActMax - 1))
            {
                sARGB->DramaAct = sARGB->DramaAct + 1;
            }
            else
            {
                sARGB->DramaAct = 0;
            }
            
        }
        
        sARGB->UpdateTRG |= ARGB_UPDATE_DATA;
    }
}

/**
 *******************************************************************************
 * @brief	   Calculate ARGB next data in ARGB_DRAMA_2MOVING mode.
 * @details     
 * @param[in]  sARGB : Control ARGB data line .     
 * @note       
 *******************************************************************************
 */
static void MG32_ARGB_TwoMovingDrama( ARGBDramaCTR_TypeDef* sARGB )
{
    uint8_t SMP_ARGB_TMDTmp;
    uint8_t SMP_ARGB_TMDMode;
    
    if( sARGB->DramaTime < sARGB->DramaTimeMax)
    {
        sARGB->DramaTime = sARGB->DramaTime + 1;
    }
    else
    {
        sARGB->DramaTime = 0;
        
        SMP_ARGB_TMDTmp   = sARGB->DramaAct & 0x7F;
        SMP_ARGB_TMDMode  = sARGB->DramaAct & 0x80;
        
        if( SMP_ARGB_TMDMode == 0)
        {
            if( sARGB->DramaScene != 0)
            {
                sARGB->DramaBuf[ ((sARGB->DramaScene - 1) * 3) + SMP_ARGB_G ]    = 0;
                sARGB->DramaBuf[ ((sARGB->DramaScene - 1) * 3) + SMP_ARGB_R ]    = 0;    
                sARGB->DramaBuf[ ((sARGB->DramaScene - 1) * 3) + SMP_ARGB_B ]    = 0;
            }
            if( sARGB->DramaSceneMax != (SMP_ARGB_PIECE-1))
            {
                sARGB->DramaBuf[ ((sARGB->DramaSceneMax + 1) * 3) + SMP_ARGB_G ] = 0;
                sARGB->DramaBuf[ ((sARGB->DramaSceneMax + 1) * 3) + SMP_ARGB_R ] = 0;    
                sARGB->DramaBuf[ ((sARGB->DramaSceneMax + 1) * 3) + SMP_ARGB_B ] = 0;
            }
        }
        else
        {
            sARGB->DramaBuf[ ((sARGB->DramaScene + 1) * 3) + SMP_ARGB_G ]       = 0;
            sARGB->DramaBuf[ ((sARGB->DramaScene + 1) * 3) + SMP_ARGB_R ]       = 0;    
            sARGB->DramaBuf[ ((sARGB->DramaScene + 1) * 3) + SMP_ARGB_B ]       = 0;
            
            sARGB->DramaBuf[ ((sARGB->DramaSceneMax - 1) * 3) + SMP_ARGB_G ]    = 0;
            sARGB->DramaBuf[ ((sARGB->DramaSceneMax - 1) * 3) + SMP_ARGB_R ]    = 0;    
            sARGB->DramaBuf[ ((sARGB->DramaSceneMax - 1) * 3) + SMP_ARGB_B ]    = 0;
        }
        
        sARGB->DramaBuf[ (sARGB->DramaScene * 3) + SMP_ARGB_G ]    = SMP_ARGBDefault_TABLE[ SMP_ARGB_TMDTmp ][ SMP_ARGB_G ];
        sARGB->DramaBuf[ (sARGB->DramaScene * 3) + SMP_ARGB_R ]    = SMP_ARGBDefault_TABLE[ SMP_ARGB_TMDTmp ][ SMP_ARGB_R ];    
        sARGB->DramaBuf[ (sARGB->DramaScene * 3) + SMP_ARGB_B ]    = SMP_ARGBDefault_TABLE[ SMP_ARGB_TMDTmp ][ SMP_ARGB_B ];
        
        sARGB->DramaBuf[ (sARGB->DramaSceneMax * 3) + SMP_ARGB_G ] = SMP_ARGBDefault_TABLE[ SMP_ARGB_TMDTmp ][ SMP_ARGB_G ];
        sARGB->DramaBuf[ (sARGB->DramaSceneMax * 3) + SMP_ARGB_R ] = SMP_ARGBDefault_TABLE[ SMP_ARGB_TMDTmp ][ SMP_ARGB_R ];    
        sARGB->DramaBuf[ (sARGB->DramaSceneMax * 3) + SMP_ARGB_B ] = SMP_ARGBDefault_TABLE[ SMP_ARGB_TMDTmp ][ SMP_ARGB_B ];
        
        if( SMP_ARGB_TMDMode == 0)
        {
            if( (sARGB->DramaScene + 1) < sARGB->DramaSceneMax)
            {
                sARGB->DramaScene    = sARGB->DramaScene + 1;
                sARGB->DramaSceneMax = sARGB->DramaSceneMax - 1;
            }                
            else
            {
                sARGB->DramaAct = sARGB->DramaAct | 0x80;
                
                if( ( (sARGB->DramaAct & 0x7F) == ( SMP_ARGB_DEFAULT_COLOR - 1)))
                {
                    sARGB->DramaAct = sARGB->DramaAct & (~0x7F);
                }
                else
                {
                    sARGB->DramaAct = sARGB->DramaAct + 1;
                }
            }
        }
        else
        {
            if(sARGB->DramaScene == 0) 
            {
                sARGB->DramaAct = sARGB->DramaAct & (~0x80);
            }
            else
            {
                sARGB->DramaScene    = sARGB->DramaScene    - 1;
                sARGB->DramaSceneMax = sARGB->DramaSceneMax + 1;
            }
        }
        
        sARGB->UpdateTRG |= ARGB_UPDATE_DATA;
    }
}

/**
 *******************************************************************************
 * @brief	   ARGB_Drama complete updating control  
 * @details     
 * @return      
 * @note       No
 *******************************************************************************
 */
__WEAK void MG32_ARGBDrama_UpdateModeCallback(void)
{
    //=========================================================
    //Note : This function should not be modified, when th callback
    //       is needed, the MID_URT_AbortReceiveCpltCallback can
    //       be implemented in the user file.
    
}



