
/*
 sinewave for BLDC with HALL sensor
 
 MCU : MG82F6D17
 MCU core Frequency : PLL to 24M Hz
 PWM frequency : 48M Hz with 10 bit PWM resolution and central align
 
 Power : DC 24V (Main power) -> regulator : 15V (Gate drive) -> regulator : 5V 
 Gate drive : IR2101 (direct drive)
 
 Motor U (hold-Yellow) connect 'A'
 Motor V (hold-Green) connect 'B'
 Motor W (hold-Blue) connect 'C'
 
 Motor HA (thick-Yellow) connect 'A det'
 Motor HB (thick-Green) connect 'B det'
 Motor HC (thick-Blue) connect 'C det'
 
 
*/




#include "REG_MG82F6D17.H"
#include "typedef.h"
#include "config.h"
#include "motor_parameter.h"
#include <math.h>


extern bit CommutateSearch;         // the time flag of Hall index 0 to 5
extern bit Timer2_flag;             // timer 2 ISR flag
extern LTYPE commutateTx6;          // record time of Hall index 0 to 5
extern bit SynchronizationFlag;     // commutate signal for synchronization
extern U8 StuckCNT;                 // avoid motor stuck

LTYPE CaledPLLT;                    // for calculate timer2 RCAP2H:L
U16 CurrRPM;                        // the current speed (RPM)
U16 Duty;                           // Duty cycle
U16 TargetRPM;                      // Motor speed (from ADC sample)
U8 State;                           // Motor state (IDLE/HallError/RUN/STUCK)
U8 commCNT = 0;

#define HoldT   4

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
int main (void)
{ 
    U8 StartupFailCNT;
    
    // ----------------------------------------------------------------
    // initial H/W
    Port_Init ();	        // Port1 (PCA), P0 (debug bus), P2.0~2 (KIB:commutate) ....
    disable_PCA ();	        // Disable PCA
    
    Timer0_Init ();
    Timer1_Init ();
    Timer2_init ();         // replace new PWM duty  
    PLL24MHz ();            // PLL to 24MHz
    ADC_init ();
    
    State = IDLE;
    
    // ----------------------------------------------------------------
    // initial KB-pattern
    SetupKeypadPattern ();
    EA = 1;
    while (State == HallError);             // hold here    

    while (1) 
    {
        
        // ----------------------------------------------------------------
        // initial sinewave PWM duty & parameter
        Duty = INIT_DUTY;
        initial_SineWavePWM (Duty);
        
        while (TargetRPM == 0) {
            ADC_SpeedSample ();
        }
        
        // ----------------------------------------------------------------
        // sine wave - start up
        StartupFailCNT = 0;
        while (1) {
            PCA_Init ();                            // PCA initial for sinewave
            MotorRotate ();
                
            if (State == STARTFAIL) {
                disable_PCA ();
                StartupFailCNT ++;
                
                // release Motor (in idle mode)
                StuckCNT = 0;
                while (StuckCNT < 250);
                StuckCNT = 0;
                while (StuckCNT < 250);
                
                Duty = Duty + 60;
                initial_SineWavePWM (Duty);

                if (StartupFailCNT < 5) 
                    continue;       // re-try
                else 
                    while (1);      // startup fail -> STOP (need hardware reset)
            }
            break;                  // entry RUN mode
        }
        
        // ----------------------------------------------------------------
        // output sinewave 
        CommutateSearch = 1; 
        StuckCNT = 0;
        State = Normal_RUN;
        
        while (1) 
        {
            //---------------------------------------------------------------------
            // reset period (CommutateSearch from KBISR.c) - 6 steps
            if (CommutateSearch) {
                CommutateSearch = 0;                        // clear flag
                CurrRPM = CalRPMTBase / CaledPLLT.LW[1].W;  // current speed (RPM)
                
                // initial period
                CaledPLLT.LW[1].W = 0 - CaledPLLT.LW[1].W;
                TR2 = 0;
                RCAP2H = CaledPLLT.LW[1].B[0];
                RCAP2L = CaledPLLT.LW[1].B[1];
                TR2 = 1;  
                
                // set trigger flag
                Timer2_flag = 1;
                if (commCNT < HoldT)
                    commCNT ++;
            }
        
            //---------------------------------------------------------------------
            // Synchronization depend on Hall sensor pattern 
            if (SynchronizationFlag) {        
                SynchronizationUVW ();
                StuckCNT = 0;
                SynchronizationFlag = 0;
            }
            
            //---------------------------------------------------------------------
            // prepare duty data after it triggers T2ISR to replace 
            if (Timer2_flag) {
                Timer2_flag = 0;
                PWMOutput ();
            } 
            
            //---------------------------------------------------------------------
            // calcuate sine wave data 
            if (commCNT == HoldT)               // stable 
                PrepareSineWaveDuty ();
                    
            //---------------------------------------------------------------------
            // stuck routine 
            if (StuckCNT > 253) {
                State = STUCK;
                disable_PCA ();
                EIE1 &= ~EKB; 
                while (1);
            }
            
            //---------------------------------------------------------------------
            // sample Variable Resistor value
            ADC_SpeedSample ();                 // sample ADC value -> TargetRPM
            if (TargetRPM == 0) break;          // exit if Target speed = 0
        }
        disable_PCA ();                         // disable PCA PWM output
        
    }
    // ----------------------------------------------------------------
}

