#include "mg32f157.h"
#include "usbd_user.h"
#include "usbd_cdc.h"
#include "usbd_user_cdc.h"

volatile uint32_t bulk_in_pending_flag = 0;
volatile uint32_t bulk_in_zlp_flag = 0;


/**
 * @brief  Initializes the USB Device.
 * @return None
 */
void USBD_User_Init(void)
{
  NVIC_InitTypeDef NVIC_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA |RCC_APB2Periph_AFIO, ENABLE);

  /* Enable USB peripheral clock */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_USB, ENABLE);

  /* Reset USB peripheral */
  RCC_APB1PeriphResetCmd(RCC_APB1Periph_USB, ENABLE);
  RCC_APB1PeriphResetCmd(RCC_APB1Periph_USB, DISABLE);

  /* Configure and enable USBCLK */
  RCC_USBCLKConfig(RCC_USBCLKSource_PLLCLK_Div2);

  USB->INTRUSBE = 0x00;
  USB->INTRTXE = 0x00;
  USB->INTRRXE = 0x00;

  /* IN Endpoint 1 FIFO ADDR=0, SIZE=16 */
  USB->INDEX = 0x01;
  USB->TXFIFOADD = (0) >> 3;
  USB->TXFIFOSZ = 1;

  /* IN Endpoint 2 FIFO ADDR=0+16, SIZE=64 */
  USB->INDEX = 0x02;
  USB->TXFIFOADD = (0 + 16) >> 3;
  USB->TXFIFOSZ = 3;

  /* OUT Endpoint 2 FIFO ADDR=16+64, SIZE=64 */
  USB->INDEX = 0x02;
  USB->RXFIFOADD = (16 + 64) >> 3;
  USB->RXFIFOSZ = 3;

  GPIO_InitStructure.GPIO_Pin   = GPIO_Pin_11 | GPIO_Pin_12;
  GPIO_InitStructure.GPIO_Mode  = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);

  /* Configure and enable USB interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = USB_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 1;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

/**
 * @brief  Deinitializes the USB Device.
 * @return None
 */
void USBD_User_DeInit(void)
{
  /* Disable USB interrupt channel */
  NVIC_DisableIRQ(USB_IRQn);

  /* Disable USB peripheral clock */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_USB, DISABLE);
}

/**
 * @brief  Connects the device to the USB host.
 * @return None
 */
void USBD_User_Connect(void)
{
  USB->POWER = USB_POWER_SUSEN | USB_POWER_SOFTCONN;
  USB->INTRUSBE = USB_INTRUSBE_RSTIE | USB_INTRUSBE_RSUIE | USB_INTRUSBE_SUSIE;
}

/**
 * @brief  Disconnects the device from the USB host.
 * @return None
 */
void USBD_User_Disconnect(void)
{
  USB->POWER &= ~USB_POWER_SOFTCONN;
}

/**
 * @brief  USB Reset Event Service Routine.
 * @return None
 */
void USBD_User_Reset(void)
{
//  USB->POWER |= USB_POWER_SUSEN;
  USB->INTRTXE = USB_INTRTXE_EP0E;
  USB->INTRRXE = 0x00;

  bulk_in_pending_flag = 0;
  bulk_in_zlp_flag = 0;
}

/**
 * @brief  USB Resume Event Service Routine.
 * @return None
 */
void USBD_User_Resume(void)
{
  GPIO_ResetBits(GPIOB, GPIO_Pin_11);   // Turn on LED2
}

/**
 * @brief  USB Suspend Event Service Routine.
 * @return None
 */
void USBD_User_Suspend(void)
{
  GPIO_SetBits(GPIOB, GPIO_Pin_11);     // Turn off LED2
}

/**
 * @brief  USB SOF Event Service Routine.
 * @return None
 */
void USBD_User_SOF(void)
{
}


/**
 * @brief  Configures device.
 * @param  cfgidx: the configuration index.
 * @return true - Success, false - Error
 */
bool USBD_User_SetConfig(uint8_t cfgidx)
{
  if (cfgidx == 1)
  {
    // Configure IN Endpoint 1 (Interrupt)
    USB->INDEX = 0x01;
    USB->TXCSRH = 0x00;
    USB->TXMAXP = (16 >> 3);
    USB->TXCSRL = USB_TXCSRL_CLRDATATOG;
    USB->TXCSRL = USB_TXCSRL_FLUSHFIFO;
    USB->TXCSRL = USB_TXCSRL_FLUSHFIFO;
    USB->INTRTXE |= (0x01 << 1);

    // Configure IN Endpoint 2 (Bulk)
    USB->INDEX = 0x02;
    USB->TXCSRH = 0x00;
    USB->TXMAXP = (64 >> 3);
    USB->TXCSRL = USB_TXCSRL_CLRDATATOG;
    USB->TXCSRL = USB_TXCSRL_FLUSHFIFO;
    USB->TXCSRL = USB_TXCSRL_FLUSHFIFO;
    USB->INTRTXE |= (0x01 << 2);

    // Configure OUT Endpoint 2 (Bulk)
    USB->INDEX = 0x02;
    USB->RXCSRH = 0x00;
    USB->RXMAXP = (64 >> 3);
    USB->RXCSRL = USB_RXCSRL_CLRDATATOG;
    USB->RXCSRL = USB_RXCSRL_FLUSHFIFO;
    USB->RXCSRL = USB_RXCSRL_FLUSHFIFO;
    USB->INTRRXE |= (0x01 << 2);

    return true;
  }

  return false;
}

/**
 * @brief  Clear current configuration.
 * @param  cfgidx: the configuration index.
 * @note   If cfgidx is 0, this function should clear all configuration.
 * @return None
 */
void USBD_User_ClearConfig(uint8_t cfgidx)
{
  USB->INTRTXE &= ~(0x01 << 1);
  USB->INTRTXE &= ~(0x01 << 2);
  USB->INTRRXE &= ~(0x01 << 2);
}


/**
 * @brief  Handle the setup device requests (Except the recipient is device).
 * @return The next control stage.
 */
UsbdControlStage USBD_User_EndPoint0_Setup(void)
{
  UsbdControlStage next_stage = USBD_CONTROL_STAGE_STALL;

  if ((UsbdCoreInfo.SetupPacket.bmRequestType & (USB_REQUEST_TYPE_Msk | USB_REQUEST_RECIPIENT_Msk)) == 
      (USB_REQUEST_TYPE_CLASS | USB_REQUEST_RECIPIENT_INTERFACE))
  {
    if ((UsbdCoreInfo.SetupPacket.wIndexL == USBD_CDC_ACM_CIF_NUM) ||
        (UsbdCoreInfo.SetupPacket.wIndexL == USBD_CDC_ACM_DIF_NUM)) {
      next_stage = USBD_EndPoint0_Setup_CDC_Req();
    }
  }

  return next_stage;
}

/**
 * @brief  Handle the out device requests.
 * @return The next control stage.
 */
UsbdControlStage USBD_User_EndPoint0_Out(void)
{
  UsbdControlStage next_stage = USBD_CONTROL_STAGE_STALL;

  if ((UsbdCoreInfo.SetupPacket.bmRequestType & (USB_REQUEST_TYPE_Msk | USB_REQUEST_RECIPIENT_Msk)) == 
      (USB_REQUEST_TYPE_CLASS | USB_REQUEST_RECIPIENT_INTERFACE))
  {
    if ((UsbdCoreInfo.SetupPacket.wIndexL == USBD_CDC_ACM_CIF_NUM) ||
        (UsbdCoreInfo.SetupPacket.wIndexL == USBD_CDC_ACM_DIF_NUM)) {
      next_stage = USBD_EndPoint0_Out_CDC_Req();
    }
  }

  return next_stage;
}



/**
 * @brief  IN Endpoint 1 Service Routine.
 * @return None
 */
void USBD_User_EP1_IN(void)
{
}

/**
 * @brief  IN Endpoint 2 Service Routine.
 * @return None
 */
void USBD_User_EP2_IN(void)
{
  if (bulk_in_pending_flag)
  {
    if (bulk_in_zlp_flag)
    {
      USBD_HW_Transmit(0x82, 0, 0);
      bulk_in_zlp_flag = 0;
    }
    else
    {
      /* Ready to get next BULK out */
      USBD_HW_ReadyToReceive(0x02);
      bulk_in_pending_flag = 0;
    }
  }
}

/**
 * @brief  IN Endpoint 3 Service Routine.
 * @return None
 */
void USBD_User_EP3_IN(void)
{
}

/**
 * @brief  IN Endpoint 4 Service Routine.
 * @return None
 */
void USBD_User_EP4_IN(void)
{
}

/**
 * @brief  IN Endpoint 5 Service Routine.
 * @return None
 */
void USBD_User_EP5_IN(void)
{
}

/**
 * @brief  OUT Endpoint 1 Service Routine.
 * @return None
 */
void USBD_User_EP1_OUT(void)
{
}

/**
 * @brief  OUT Endpoint 2 Service Routine.
 * @return None
 */
uint8_t  EchoBuffer[64];
void USBD_User_EP2_OUT(void)
{
  USB->INDEX = 0x02;
  if (USB->RXCSRL & USB_RXCSRL_RXPKTRDY)
  {
    uint16_t len;
    len = USBD_HW_GetRxDataCount(0x02);
    USBD_HW_ReadEP(0x02, EchoBuffer, len);

    USBD_HW_Transmit(0x82, EchoBuffer, len);
    bulk_in_pending_flag = 1;

    if (len % 64 == 0)
      bulk_in_zlp_flag = 1;
    else
      bulk_in_zlp_flag = 0;
  }
}

/**
 * @brief  OUT Endpoint 3 Service Routine.
 * @return None
 */
void USBD_User_EP3_OUT(void)
{
}

/**
 * @brief  OUT Endpoint 4 Service Routine.
 * @return None
 */
void USBD_User_EP4_OUT(void)
{
}

/**
 * @brief  OUT Endpoint 5 Service Routine.
 * @return None
 */
void USBD_User_EP5_OUT(void)
{
}
