#include "mg32f157.h"
#include "usbd_user.h"
#include "usbd_hid.h"
#include "usbd_user_hid.h"

/**
 * @brief  Initializes the USB Device.
 * @return None
 */
void USBD_User_Init(void)
{
  NVIC_InitTypeDef NVIC_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA |RCC_APB2Periph_AFIO, ENABLE);

  /* Enable USB peripheral clock */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_USB, ENABLE);

  /* Reset USB peripheral */
  RCC_APB1PeriphResetCmd(RCC_APB1Periph_USB, ENABLE);
  RCC_APB1PeriphResetCmd(RCC_APB1Periph_USB, DISABLE);

  /* Configure and enable USBCLK */
  RCC_USBCLKConfig(RCC_USBCLKSource_PLLCLK_Div2);

  USB->INTRUSBE = 0x00;
  USB->INTRTXE = 0x00;
  USB->INTRRXE = 0x00;
  /* IN Endpoint 1 FIFO ADDR=0, SIZE=64 */
  USB->INDEX = 0x01;
  USB->TXFIFOADD = (0)>>3;
  USB->TXFIFOSZ = 3;

  GPIO_InitStructure.GPIO_Pin   = GPIO_Pin_11 | GPIO_Pin_12;
  GPIO_InitStructure.GPIO_Mode  = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);

  /* Configure and enable USB interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = USB_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 1;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

/**
 * @brief  Deinitializes the USB Device.
 * @return None
 */
void USBD_User_DeInit(void)
{
  /* Disable USB interrupt channel */
  NVIC_DisableIRQ(USB_IRQn);

  /* Disable USB peripheral clock */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_USB, DISABLE);
}

/**
 * @brief  Connects the device to the USB host.
 * @return None
 */
void USBD_User_Connect(void)
{
  USB->POWER = USB_POWER_SUSEN | USB_POWER_SOFTCONN;
  USB->INTRUSBE = USB_INTRUSBE_RSTIE | USB_INTRUSBE_RSUIE | USB_INTRUSBE_SUSIE;;
}

/**
 * @brief  Disconnects the device from the USB host.
 * @return None
 */
void USBD_User_Disconnect(void)
{
  USB->POWER &= ~USB_POWER_SOFTCONN;
}

/**
 * @brief  USB Reset Event Service Routine.
 * @return None
 */
void USBD_User_Reset(void)
{
//  USB->POWER |= USB_POWER_SUSEN;
  USB->INTRTXE = USB_INTRTXE_EP0E;
  USB->INTRRXE = 0x00;

  hid_report_xfer_flag = 0;
}

/**
 * @brief  USB Resume Event Service Routine.
 * @return None
 */
void USBD_User_Resume(void)
{
  GPIO_ResetBits(GPIOB, GPIO_Pin_10);  // Turn on LED
}

/**
 * @brief  USB Suspend Event Service Routine.
 * @return None
 */
void USBD_User_Suspend(void)
{
  GPIO_SetBits(GPIOB, GPIO_Pin_10);    // Turn off LED
}

/**
 * @brief  USB SOF Event Service Routine.
 * @return None
 */
void USBD_User_SOF(void)
{
}


/**
 * @brief  Configures device.
 * @param  cfgidx: the configuration index.
 * @return true - Success, false - Error
 */
bool USBD_User_SetConfig(uint8_t cfgidx)
{
  if (cfgidx == 1)
  {
    // Configure IN Endpoint 1 (Interrupt)
    USB->INDEX = 0x01;
    USB->TXCSRH = 0x00;
    USB->TXMAXP = (64 >> 3);
    USB->TXCSRL = USB_TXCSRL_CLRDATATOG;
    USB->TXCSRL = USB_TXCSRL_FLUSHFIFO;
    USB->TXCSRL = USB_TXCSRL_FLUSHFIFO;
    USB->INTRTXE |= (0x01 << 1);

    return true;
  }

  return false;
}

/**
 * @brief  Clear current configuration.
 * @param  cfgidx: the configuration index.
 * @note   If cfgidx is 0, this function should clear all configuration.
 * @return None
 */
void USBD_User_ClearConfig(uint8_t cfgidx)
{
  USB->INTRTXE &= ~(0x01 << 1);
}


/**
 * @brief  Handle the setup device requests (Except the recipient is device).
 * @return The next control stage.
 */
UsbdControlStage USBD_User_EndPoint0_Setup(void)
{
  UsbdControlStage next_stage = USBD_CONTROL_STAGE_STALL;

  if ((UsbdCoreInfo.SetupPacket.bmRequestType & USB_REQUEST_RECIPIENT_Msk) == USB_REQUEST_RECIPIENT_INTERFACE)
  {
    if (UsbdCoreInfo.SetupPacket.wIndexL == USBD_HID_IF_NUM) {
      next_stage = USBD_EndPoint0_Setup_HID_Req();
    }
  }

  return next_stage;
}

/**
 * @brief  Handle the out device requests.
 * @return The next control stage.
 */
UsbdControlStage USBD_User_EndPoint0_Out(void)
{
  UsbdControlStage next_stage = USBD_CONTROL_STAGE_STALL;

  if ((UsbdCoreInfo.SetupPacket.bmRequestType & (USB_REQUEST_TYPE_Msk | USB_REQUEST_RECIPIENT_Msk)) == 
      (USB_REQUEST_TYPE_CLASS | USB_REQUEST_RECIPIENT_INTERFACE))
  {
    if (UsbdCoreInfo.SetupPacket.wIndexL == USBD_HID_IF_NUM) {
      next_stage = USBD_EndPoint0_Out_HID_Req();
    }
  }

  return next_stage;
}



/**
 * @brief  IN Endpoint 1 Service Routine.
 * @return None
 */
void USBD_User_EP1_IN(void)
{
  hid_report_xfer_flag = 0;
}

/**
 * @brief  IN Endpoint 2 Service Routine.
 * @return None
 */
void USBD_User_EP2_IN(void)
{
}

/**
 * @brief  IN Endpoint 3 Service Routine.
 * @return None
 */
void USBD_User_EP3_IN(void)
{
}

/**
 * @brief  IN Endpoint 4 Service Routine.
 * @return None
 */
void USBD_User_EP4_IN(void)
{
}

/**
 * @brief  IN Endpoint 5 Service Routine.
 * @return None
 */
void USBD_User_EP5_IN(void)
{
}

/**
 * @brief  OUT Endpoint 1 Service Routine.
 * @return None
 */
void USBD_User_EP1_OUT(void)
{
}

/**
 * @brief  OUT Endpoint 2 Service Routine.
 * @return None
 */
void USBD_User_EP2_OUT(void)
{
}

/**
 * @brief  OUT Endpoint 3 Service Routine.
 * @return None
 */
void USBD_User_EP3_OUT(void)
{
}

/**
 * @brief  OUT Endpoint 4 Service Routine.
 * @return None
 */
void USBD_User_EP4_OUT(void)
{
}

/**
 * @brief  OUT Endpoint 5 Service Routine.
 * @return None
 */
void USBD_User_EP5_OUT(void)
{
}
