#include "bsp_uart4.h"


/**
  * @name    uart4_init
  * @brief   Init UART4
  * @param   apbclk: UART4 APB Clock frequency
  * @param   baud: Baudrate to configure
  * @retval  None
  */
void uart4_init(uint32_t apbclk, uint32_t baud)
{
  /* Enable UART4 and GPIOC Clock */
  RCC->APB2ENR |= RCC_APB2ENR_IOPCEN;
  RCC->APB1ENR |= RCC_APB1ENR_UART4EN;
  
  /* Reset UART4 */
  RCC->APB1RSTR |= RCC_APB1RSTR_UART4RST;
  RCC->APB1RSTR &= ~RCC_APB1RSTR_UART4RST;
  
  /* PC10(UART4_TX) configure to push pull, PC11(UART4_RX) configure to Input with Pull-up resistance */
  GPIOC->CRH = (GPIOC->CRH & 0xFFFF00FF) | 0x00008B00;
  GPIOC->BSRR = GPIO_BSRR_BS11;
  
  /* Configure UART4 */
  UART4->CR2 = 0x0000;
  UART4->CR1 = USART_CR1_TE | USART_CR1_RE;
  UART4->BRR = (apbclk + (baud >> 1)) / baud;    
  
  /* Enable UART4 */
  UART4->CR1 |= USART_CR1_UE;
}


/**
  * @name    uart4_send
  * @brief   Send data through UART4
  * @param   buffer: Buffer to send
  * @param   length: Buffer length
  * @retval  None
  */
void uart4_send(const uint8_t* buffer, uint32_t length)
{
  while(length)
  {
    while(!(UART4->SR & USART_SR_TC));
    UART4->DR = *buffer;
    buffer++;  length--;
  }
  while(!(UART4->SR & USART_SR_TC));
}







