/**
 * @file    mg32f157_trng.c
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file provides all the TRNG firmware functions.
 */

/* Includes ------------------------------------------------------------------*/
#include "mg32f157_trng.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @defgroup TRNG 
  * @brief TRNG driver modules
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

/** @defgroup TRNG_Private_Defines
  * @{
  */
  
#define TRNG_Set                    ((uint32_t)0x00000004)
#define TRNG_Reset                  ((uint32_t)0x00000000)

#define TRNG_CED_Set                ((uint32_t)0x00000020)
#define TRNG_IE_Set                 ((uint32_t)0x00000008)
/**
  * @}
  */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/** @defgroup TRNG_Private_Functions
  * @{
  */

/**
 * @brief  Deinitializes the TRNG peripheral registers to their default reset values.
 * @param  None
 * @return None
 */
void TRNG_DeInit(void)
{ 
  /* Deinitialize the TRNG registers */
  RNG->CR = TRNG_Reset;
  RNG->SR = TRNG_Reset;
}

/**
 * @brief  Enables or disables the TRNG clock error detection.
 * @param  NewState: new state of the TRNG clock error detection. 
 *         This parameter can be: ENABLE or DISABLE.
 * @note   When enables the clock error detection, TRNG peripheral should be disabled,
 *         and then TRNG peripheral can not generate true random number.
 * @return None.
 */
void TRNG_Clock_CheckCmd(FunctionalState NewState)
{ 
  assert_param(IS_FUNCTIONAL_STATE(NewState));
  
  if (NewState != DISABLE)
  {
    /* Enable the TRNG detection error of clock */
    RNG->CR |= TRNG_CED_Set;
  }
  else
  {
    /* Disable the TRNG detection error of clock */
    RNG->CR &= (~TRNG_CED_Set);
  }
}

/**
 * @brief  Enables or disables the TRNG peripheral generate true random number.
 * @param  NewState: new state of the TRNG.
 *         This parameter can be: ENABLE or DISABLE.
 * @note   When enables the TRNG peripheral, clock error detection should be disabled.
 * @return None
 */
void TRNG_Cmd(FunctionalState NewState)
{ 
  assert_param(IS_FUNCTIONAL_STATE(NewState));
  
  if (NewState != DISABLE)
  {
    /* Enable the TRNG */
    RNG->CR |= TRNG_Set;
  }
  else
  {
    /* Disable the TRNG */
    RNG->CR &= (~TRNG_Set);
  }
}

/**
 * @brief  Returns true random number value.
 * @param  None
 * @return 32-bit TRNG
 */
uint32_t TRNG_GetData(void)
{
  return (RNG->DR);
}

/**
 * @brief  Checks whether the specified TRNG flag is set or not.
 * @param  TRNG_FLAG: specifies the flag to check.
 *         This parameter can be one of the following values:
 *         @arg @ref TRNG_FLAG_SECS: Seed error current status flag
 *         @arg @ref TRNG_FLAG_CECS: Clock error current status flag
 *         @arg @ref TRNG_FLAG_DRDY: TRNG data ready flag
 * @return The new state of TRNG_FLAG (SET or RESET).
 */
FlagStatus TRNG_GetFlagStatus(uint8_t TRNG_FLAG)
{
  FlagStatus bitstatus = RESET;

  /* Check the parameters */
  assert_param(IS_TRNG_GET_FLAG(TRNG_FLAG));

  /* Check the status of the specified TRNG flag */
  if ((RNG->SR & TRNG_FLAG) != (uint8_t)RESET)
  {
    /* TRNG_FLAG is set */
    bitstatus = SET;
  }
  else
  {
    /* TRNG_FLAG is reset */
    bitstatus = RESET;
  }

  /* Return the TRNG_FLAG status */
  return  bitstatus;
}

/**
 * @brief  Clears the TRNG's pending flags.
 * @param  TRNG_FLAG: specifies the flags to clear.
 *         This parameter can be any combination  of the following values:
 *         @arg @ref TRNG_FLAG_SECS: Seed error current status flag
 *         @arg @ref TRNG_FLAG_CECS: Clock error current status flag
 * @note   Data ready bit(DRDY) in SR register only can be read but written,
 *         so you can not clear the TRNG_FLAG_DRDY.
 * @return None
 */
void TRNG_ClearFlag(uint8_t TRNG_FLAG)
{
  assert_param(IS_TRNG_CLEAR_FLAG(TRNG_FLAG));
  
  /* Clear the selected TRNG flags */
  RNG->SR &= (~(uint32_t)TRNG_FLAG);
}

/**
 * @brief  Enable or Disable the TRNG interrupt.
 * @param  NewState: new state of the TRNG interrupt.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void TRNG_ITCmd(FunctionalState NewState)
{

  assert_param(IS_FUNCTIONAL_STATE(NewState));

  if (NewState != DISABLE)
  {
    /* Enable the TRNG interrupt */
    RNG->CR |= TRNG_IE_Set;
  }
  else
  {
    /* Disable the TRNG interrupt */
    RNG->CR &= (~(uint32_t)TRNG_IE_Set);
  }
}

/**
 * @brief  Checks whether the specified TRNG interrupt has occurred or not.
 * @param  TRNG_IT: specifies the interrupt source to check.
 *         This parameter can be one of the following values:
 *         @arg @ref TRNG_IT_SEIS: Seed error interrupt
 *         @arg @ref TRNG_IT_CEIS: Clock error interrupt
 *         @arg @ref TRNG_IT_DRDY: TRNG data ready interrupt
 * @return The new state of TRNG_IT (SET or RESET).
 */
ITStatus TRNG_GetITStatus(uint16_t TRNG_IT)
{
  ITStatus bitstatus = RESET;
  uint32_t itmask = 0, enablestatus = 0;

  /* Check the parameters */
  assert_param(IS_TRNG_GET_IT(TRNG_IT));

  /* Get the TRNG IT index */
  itmask = (TRNG_IT >> 8);
  /* Get the TRNG_IT enable bit status */
  enablestatus = (RNG->CR & (uint8_t)TRNG_IT) ;
  /* Check the status of the specified TRNG interrupt */
  if (((RNG->SR & itmask) != RESET) && enablestatus)
  {
    /* TRNG_IT is set */
    bitstatus = SET;
  }
  else
  {
    /* TRNG_IT is reset */
    bitstatus = RESET;
  }
  
  /* Return the TRNG_IT status */
  return  bitstatus;
}

/**
 * @brief  Clears the TRNG's pending interrupt pending bits.
 * @param  TRNG_IT: specifies the interrupt pending bit to clear.
 *   This parameter can be any combination of the following values:
 *         @arg @ref TRNG_IT_SEIS: Seed error interrupt
 *         @arg @ref TRNG_IT_CEIS: Clock error interrupt
 * @note   Data ready bit(DRDY) in SR register only can be read but writen,
 *         so you can not clear TRNG_IT_DRDY.
 * @return None
 */
void TRNG_ClearITPendingBit(uint16_t TRNG_IT)
{
  uint8_t itmask = 0;

  /* Check the parameters */
  assert_param(IS_TRNG_CLEAR_IT(TRNG_IT));

  /* Get the TRNG IT index */
  itmask = (uint8_t)(TRNG_IT >> 8);
  /* Clear the selected TRNG interrupt pending bits */
  RNG->SR &= (~(uint32_t)itmask);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
