/**
 * @file    mg32f157_dbgmcu.c
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file provides all the DBGMCU firmware functions.
 */

/* Includes ------------------------------------------------------------------*/
#include "mg32f157_dbgmcu.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @defgroup DBGMCU 
  * @brief DBGMCU driver modules
  * @{
  */ 

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

/** @defgroup DBGMCU_Private_Defines
  * @{
  */

#define IDCODE_DEVID_MASK    ((uint32_t)0x00000FFF)

/**
  * @}
  */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/** @defgroup DBGMCU_Private_Functions
  * @{
  */

/**
 * @brief  Returns the device revision identifier.
 * @param  None
 * @return Device revision identifier
 */
uint32_t DBGMCU_GetREVID(void)
{
  return(DBGMCU->IDCODE >> 16);
}

/**
 * @brief  Returns the device identifier.
 * @param  None
 * @return Device identifier
 */
uint32_t DBGMCU_GetDEVID(void)
{
  return(DBGMCU->IDCODE & IDCODE_DEVID_MASK);
}

/**
 * @brief  Configures the specified peripheral and low power mode behavior
 *         when the MCU under Debug mode.
 * @param  DBGMCU_Periph: specifies the peripheral and low power mode.
 *         This parameter can be any combination of the following values:
 *         @arg @ref DBGMCU_SLEEP: Keep debugger connection during SLEEP mode
 *         @arg @ref DBGMCU_STOP: Keep debugger connection during STOP mode
 *         @arg @ref DBGMCU_STANDBY: Keep debugger connection during STANDBY mode
 *         @arg @ref DBGMCU_IWDG_STOP: Debug IWDG stopped when Core is halted
 *         @arg @ref DBGMCU_WWDG_STOP: Debug WWDG stopped when Core is halted
 *         @arg @ref DBGMCU_TIM1_STOP: TIM1 counter stopped when Core is halte
 *         @arg @ref DBGMCU_TIM2_STOP: TIM2 counter stopped when Core is halted
 *         @arg @ref DBGMCU_TIM3_STOP: TIM3 counter stopped when Core is halted
 *         @arg @ref DBGMCU_TIM4_STOP: TIM4 counter stopped when Core is halted
 *         @arg @ref DBGMCU_CAN1_STOP: Debug CAN2 stopped when Core is halted
 *         @arg @ref DBGMCU_I2C1_SMBUS_TIMEOUT: I2C1 SMBUS timeout mode stopped when Core is halted
 *         @arg @ref DBGMCU_I2C2_SMBUS_TIMEOUT: I2C2 SMBUS timeout mode stopped when Core is halted
 *         @arg @ref DBGMCU_TIM5_STOP: TIM5 counter stopped when Core is halted
 *         @arg @ref DBGMCU_TIM6_STOP: TIM6 counter stopped when Core is halted
 *         @arg @ref DBGMCU_TIM7_STOP: TIM7 counter stopped when Core is halted
 *         @arg @ref DBGMCU_TIM8_STOP: TIM8 counter stopped when Core is halted
 *         @arg @ref DBGMCU_CAN2_STOP: Debug CAN2 stopped when Core is halted 
 * @param  NewState: new state of the specified peripheral in Debug mode.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void DBGMCU_Config(uint32_t DBGMCU_Periph, FunctionalState NewState)
{
  /* Check the parameters */
  assert_param(IS_DBGMCU_PERIPH(DBGMCU_Periph));
  assert_param(IS_FUNCTIONAL_STATE(NewState));

  if (NewState != DISABLE)
  {
    DBGMCU->CR |= DBGMCU_Periph;
  }
  else
  {
    DBGMCU->CR &= ~DBGMCU_Periph;
  }
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
