/**
 * @file    mg32f157_can.c
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file provides all the I2C firmware functions.
 */

/* Includes ------------------------------------------------------------------*/
#include "mg32f157_can.h"
#include "mg32f157_rcc.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @defgroup CAN
  * @brief CAN driver modules
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

/** @defgroup CAN_Private_Defines
  * @{
  */

/* ---------------------- CAN registers bit mask ------------------------ */
#define CAN_BTR_MASK                  (0xFFFFUL << 16)
#define CAN_IDType_Mask               (0x01UL << 7)
#define CAN_FrameType_Mask            (0x01UL << 6)
#define CAN_FrameFormat_Mask          (0x01UL << 5)
#define CAN_BitRateSwitch_Mask        (0x01UL << 4)
#define CAN_ErrorState_Mask_Std       (0x01UL << 19)
#define CAN_ErrorState_Mask_Ext       (0x01UL << 1)
#define CAN_DataLength_Mask           (0x0FUL << 0)

/**
  * @}
  */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/

/** @defgroup CAN_Private_Variables
  * @{
  */

static const uint8_t CAN_DLCTOLengthTable[16] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 12, 16, 20, 24, 32, 48, 64};
static const uint8_t CAN_LengthTODLCTable[65] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 9, 9, 9, 10, 10, 10,
                                                 10, 11, 11, 11, 11, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13,
                                                 13, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
                                                 14, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15};
/**
  * @}
  */

/* Private functions ---------------------------------------------------------*/

/** @defgroup CAN_Private_Functions
  * @{
  */

/**
 * @brief  Deinitializes the CAN peripheral registers to their default reset values.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return None
 */
void CAN_DeInit(CAN_TypeDef* CANx)
{
  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  if (CANx == CAN1)
  {
    RCC_APB1PeriphResetCmd(RCC_APB1Periph_CAN1, ENABLE);
    RCC_APB1PeriphResetCmd(RCC_APB1Periph_CAN1, DISABLE);
  }
}

/**
 * @brief  Initializes the CAN peripheral according to the
 *         specified parameters in the CAN_InitStruct
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CAN_InitStruct: pointer to a CAN_InitTypeDef structure that
 *         contains the configuration information for the CAN peripheral.
 * @return None
 */
void CAN_Init(CAN_TypeDef* CANx, CAN_InitTypeDef* CAN_InitStruct)
{
  uint32_t tmpreg;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  /*---------------------------- CANx Mode Configuration ------------------------*/
  if ((CAN_InitStruct->CAN_Mode == CAN_Mode_Normal) || (CAN_InitStruct->CAN_Mode == CAN_Mode_Silent))
  {
    CANx->ISR_SR_CMR_MR |= (((uint32_t)(CAN_InitStruct->CAN_Mode)) << 1);
  }
  else
  {
    CANx->TEST |= (CAN_InitStruct->CAN_Mode & (0x0F));
  }

  /*---------------------------- CANx BTR Configuration ------------------------*/
  tmpreg = CANx->BT1_BT0_RMC_IMR;
  tmpreg &= CAN_BTR_MASK;
  tmpreg |= (((uint32_t)(CAN_InitStruct->CAN_Prescaler / 2 - 1) << 16) |
             ((uint32_t)(CAN_InitStruct->CAN_SJW) << 22) |
             ((uint32_t)(CAN_InitStruct->CAN_SEG1) << 24) |
             ((uint32_t)(CAN_InitStruct->CAN_SEG2) << 28) |
             ((uint32_t)(CAN_InitStruct->CAN_Sample) << 31));
  CANx->BT1_BT0_RMC_IMR = tmpreg;

  /*---------------------------- CANx FDCFG Configuration ------------------------*/
  tmpreg = CANx->APERR_DPERR_FDSR_FDCFG;
  if(CAN_InitStruct->AutoRetransmiteCmd != DISABLE)
  {
    tmpreg &= (~CAN_FDCFG_DAR);
  }
  else
  {
    tmpreg |= CAN_FDCFG_DAR;
  }
  CANx->APERR_DPERR_FDSR_FDCFG = tmpreg;
}

/**
 * @brief  Initializes the CAN peripheral to FD Mode according to the
 *         specified parameters in the CAN_InitStruct
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CANFD_InitTypeDef: pointer to a CANFD_InitTypeDef structure that
 *         contains the configuration information for the CAN peripheral FD Mode.
 * @return None
 */
void CANFD_Init(CAN_TypeDef* CANx, CANFD_InitTypeDef* CANFD_InitStruct)
{
  uint32_t tmpreg;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  /*---------------------------- CANx Mode Configuration ------------------------*/
  if ((CANFD_InitStruct->CAN_Mode == CAN_Mode_Normal) || (CANFD_InitStruct->CAN_Mode == CAN_Mode_Silent))
  {
    CANx->ISR_SR_CMR_MR |= (((uint32_t)(CANFD_InitStruct->CAN_Mode)) << 1);
  }
  else
  {
    CANx->TEST |= (CANFD_InitStruct->CAN_Mode & (0x0F));
  }

  /*---------------------------- CANx BTR Configuration ------------------------*/
  tmpreg = CANx->BT1_BT0_RMC_IMR;
  tmpreg &= CAN_BTR_MASK;
  tmpreg |= (((uint32_t)(CANFD_InitStruct->CAN_Prescaler / 2 - 1) << 16) |
             ((uint32_t)(CANFD_InitStruct->CAN_SJW) << 22) |
             ((uint32_t)(CANFD_InitStruct->CAN_SEG1) << 24) |
             ((uint32_t)(CANFD_InitStruct->CAN_SEG2) << 28) |
             ((uint32_t)(CANFD_InitStruct->CAN_Sample) << 31));
  CANx->BT1_BT0_RMC_IMR = tmpreg;

  /*---------------------------- CANx NBT Configuration ------------------------*/
  tmpreg = (((uint32_t)(CANFD_InitStruct->FD_ArbitrationSJW) << 20) |
            ((uint32_t)(CANFD_InitStruct->FD_ArbitrationSEG2) << 16) |
            ((uint32_t)(CANFD_InitStruct->FD_ArbitrationSEG1) << 10) |
            ((uint32_t)(CANFD_InitStruct->FD_ArbitrationPrescaler) / 2 - 1));
  CANx->NBT = tmpreg;

  /*---------------------------- CANx DBT TDCR Configuration ------------------------*/
  tmpreg = CANx->SSPP_TDCR_DBT;
  tmpreg |= (((uint32_t)(CANFD_InitStruct->FD_DataPrescaler) / 2 - 1) |
             ((uint32_t)(CANFD_InitStruct->FD_DataSJW) << 5) |
             ((uint32_t)(CANFD_InitStruct->FD_DataSEG1) << 8) |
             ((uint32_t)(CANFD_InitStruct->FD_DataSEG2) << 13) |
             ((uint32_t)(CANFD_InitStruct->FD_TransmateDelayCompensationOffset) << 16));

  if(CANFD_InitStruct->FD_TransmateDelayCompensationCmd != DISABLE)
  {
    tmpreg |= CAN_TDCR_TDCEN;
  }

  CANx->SSPP_TDCR_DBT = tmpreg;

  /*---------------------------- CANx DBT TDCR Configuration ------------------------*/
  tmpreg = CANx->APERR_DPERR_FDSR_FDCFG;
  tmpreg |= (((uint32_t)(CANFD_InitStruct->FD_FrameFormat) << 0) |
             ((uint32_t)(CANFD_InitStruct->FD_DataTimingSection) << 1) |
             ((uint32_t)(CANFD_InitStruct->FD_ArbitrationTimingSection) << 2) |
             ((uint32_t)(CANFD_InitStruct->FD_ISOFrameFormatSection) << 3));

  if(CANFD_InitStruct->AutoRetransmiteCmd != DISABLE)
  {
    tmpreg &= (~CAN_FDCFG_DAR);
  }
  else
  {
    tmpreg |= CAN_FDCFG_DAR;
  }

  if(CANFD_InitStruct->FD_RestrictedOperationCmd != DISABLE)
  {
    tmpreg |= CAN_FDCFG_REOM;
  }
  else
  {
    tmpreg &= (~CAN_FDCFG_REOM);
  }
  
  CANx->APERR_DPERR_FDSR_FDCFG = tmpreg;
}

/**
 * @brief  Initializes the CAN peripheral according to the specified.
 *         parameters in the CAN_FilterInitTypeDef.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CAN_FilterInitStruct: pointer to a CAN_FilterInitTypeDef
 *         structure that contains the configuration information.
 * @return None
 */
void CAN_FilterInit(CAN_TypeDef* CANx, CAN_FilterInitTypeDef* CAN_FilterInitStruct)
{
  uint32_t tmpreg = 0;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  if(CAN_FilterInitStruct->Filter_Mode == CAN_Filter_Mode_Single)
  {
    CANx->ISR_SR_CMR_MR |= CAN_MR_AFM;
    if(CAN_FilterInitStruct->Filter_FrameFormat == CAN_Filter_FrameFormat_Standard)
    {
      tmpreg = ((((CAN_FilterInitStruct->Filter1_Code >> 3) & 0xFF) << 0) |
                ((CAN_FilterInitStruct->Filter1_Code & 0x07) << 13) |
                (CAN_FilterInitStruct->Filter1_Type << 12) |
                (0xFU << 8));
      CANx->ACR = tmpreg;

      tmpreg = ((((CAN_FilterInitStruct->Filter1_Mask >> 3) & 0xFF) << 0) |
                ((CAN_FilterInitStruct->Filter1_Mask & 0x07) << 13) |
                (0xFU << 8) |
                (0xFFFFU << 16));
      if(CAN_FilterInitStruct->Filter1_RTRCmd == DISABLE)
      {
        tmpreg |= (0x01 << 12);
      }

      CANx->AMR = tmpreg;
    }
    else
    {
      tmpreg = ((((CAN_FilterInitStruct->Filter1_Code >> 21) & 0xFF) << 0) |
                (((CAN_FilterInitStruct->Filter1_Code >> 13) & 0xFF) << 8) |
                (((CAN_FilterInitStruct->Filter1_Code >> 5) & 0xFF) << 16) |
                ((CAN_FilterInitStruct->Filter1_Code & 0x1F) << 27) |
                (CAN_FilterInitStruct->Filter1_Type << 26));
      CANx->ACR = tmpreg;

      tmpreg = ((((CAN_FilterInitStruct->Filter1_Mask >> 21) & 0xFF) << 0) |
                (((CAN_FilterInitStruct->Filter1_Mask >> 13) & 0xFF) << 8) |
                (((CAN_FilterInitStruct->Filter1_Mask >> 5) & 0xFF) << 16) |
                ((CAN_FilterInitStruct->Filter1_Mask & 0x1F) << 27) |
                (0x03U << 24));

      if(CAN_FilterInitStruct->Filter1_RTRCmd == DISABLE)
      {
        tmpreg |= (0x01 << 26);
      }
      CANx->AMR = tmpreg;
    }
  }
  else
  {
    CANx->ISR_SR_CMR_MR &= (~CAN_MR_AFM);
    if(CAN_FilterInitStruct->Filter_FrameFormat == CAN_Filter_FrameFormat_Standard)
    {
      tmpreg = ((((CAN_FilterInitStruct->Filter1_Code >> 3) & 0xFF) << 0) |
                ((CAN_FilterInitStruct->Filter1_Code & 0x07) << 13) |
                (CAN_FilterInitStruct->Filter1_Type << 12) |
                (0xFU << 8) |
                (((CAN_FilterInitStruct->Filter2_Code >> 3) & 0xFF) << 16) |
                ((CAN_FilterInitStruct->Filter2_Code & 0x07) << 29) |
                (CAN_FilterInitStruct->Filter2_Type << 28) |
                (0xFU << 24));
      CANx->ACR = tmpreg;

      tmpreg = ((CAN_FilterInitStruct->Filter1_Mask >> 3) |
                ((CAN_FilterInitStruct->Filter1_Mask & 0x07) << 13) |
                (0x1FU << 8) |
                ((CAN_FilterInitStruct->Filter2_Mask >> 3) << 16) |
                ((CAN_FilterInitStruct->Filter2_Mask & 0x07) << 29) |
                (0x1FU << 24));
      if(CAN_FilterInitStruct->Filter1_RTRCmd == DISABLE)
      {
        tmpreg |= (0x01 << 12);
      }
      if (CAN_FilterInitStruct->Filter2_RTRCmd == DISABLE)
      {
        tmpreg |= (0x01 << 28);
      }
      CANx->AMR = tmpreg;
    }
    else
    {
      tmpreg = (((CAN_FilterInitStruct->Filter1_Code >> 21) & 0xFF) |
                (((CAN_FilterInitStruct->Filter1_Code >> 13) & 0xFF) << 8) |
                (((CAN_FilterInitStruct->Filter2_Code >> 21) & 0xFF) << 16) |
                (((CAN_FilterInitStruct->Filter2_Code >> 13) & 0xFF) << 24));
      CANx->ACR = tmpreg;

      tmpreg = (((CAN_FilterInitStruct->Filter1_Mask >> 21) & 0xFF) |
                (((CAN_FilterInitStruct->Filter1_Mask >> 13) & 0xFF) << 8) |
                (((CAN_FilterInitStruct->Filter2_Mask >> 21) & 0xFF) << 16) |
                (((CAN_FilterInitStruct->Filter2_Mask >> 13) & 0xFF) << 24));
      CANx->AMR = tmpreg;
    } 
  }
}

/**
 * @brief  Fills each CAN_InitStruct member with its default value.
 * @param  CAN_InitStruct: pointer to a CAN_InitTypeDef structure which
 *         will be initialized
 * @return None
 */
void CAN_StructInit(CAN_InitTypeDef* CAN_InitStruct)
{
  CAN_InitStruct->CAN_Mode = CAN_Mode_Normal;
  CAN_InitStruct->CAN_Sample = CAN_Sample_Once;

  CAN_InitStruct->CAN_Prescaler = 6;
  CAN_InitStruct->CAN_SEG1 = CAN_SEG1_5tq;
  CAN_InitStruct->CAN_SEG2 = CAN_SEG2_2tq;
  CAN_InitStruct->CAN_SJW = 2;
  
  CAN_InitStruct->AutoRetransmiteCmd = ENABLE;
}

/**
 * @brief  Fills each CANFD_InitTypeDef member with its default value.
 * @param  CANFD_InitStruct: pointer to a CANFD_InitTypeDef structure which
 *         will be initialized
 * @return None
 */
 void CANFD_StructInit(CANFD_InitTypeDef* CANFD_InitStruct)
{
  CANFD_InitStruct->CAN_Mode = CAN_Mode_Normal;
  CANFD_InitStruct->CAN_Sample = CAN_Sample_Once;
  CANFD_InitStruct->CAN_Prescaler = 6;
  CANFD_InitStruct->CAN_SJW = CAN_SJW_2tq;
  CANFD_InitStruct->CAN_SEG1 = CAN_SEG1_5tq;
  CANFD_InitStruct->CAN_SEG2 = CAN_SEG2_2tq;

  CANFD_InitStruct->FD_ArbitrationPrescaler = 6;
  CANFD_InitStruct->FD_ArbitrationSJW = CANFD_ArbitrationSJW_2tq;
  CANFD_InitStruct->FD_ArbitrationSEG1 = CANFD_ArbitrationSEG1_5tq;
  CANFD_InitStruct->FD_ArbitrationSEG2 = CANFD_ArbitrationSEG2_2tq;

  CANFD_InitStruct->FD_DataPrescaler = 4;
  CANFD_InitStruct->FD_DataSJW = CANFD_DataSJW_2tq;
  CANFD_InitStruct->FD_DataSEG1 = CANFD_DataSEG1_5tq;
  CANFD_InitStruct->FD_DataSEG2 = CANFD_DataSEG2_2tq;

  CANFD_InitStruct->FD_TransmateDelayCompensationCmd = ENABLE;
  CANFD_InitStruct->FD_TransmateDelayCompensationOffset = 20;
  CANFD_InitStruct->FD_RestrictedOperationCmd = DISABLE;
  CANFD_InitStruct->AutoRetransmiteCmd = ENABLE;
  CANFD_InitStruct->FD_ISOFrameFormatSection = CANFD_ISOFrameFormatSection_ISO11898;
  CANFD_InitStruct->FD_ArbitrationTimingSection = CANFD_ArbitrationTimingSection_NBT;
  CANFD_InitStruct->FD_DataTimingSection = CANFD_DataTimingSection_DBT;
  CANFD_InitStruct->FD_FrameFormat = CANFD_FrameFormat_FD;
}

/**
 * @brief  Fills each CANFD_InitStruct member with its default value.
 * @param  CANFD_InitStruct: pointer to a CANFD_InitStruct structure which
 *         will be initialized
 * @return None
 */
 void CAN_FilterStructInit(CAN_FilterInitTypeDef* CAN_FilterInitStruct)
{
  CAN_FilterInitStruct->Filter_Mode = CAN_Filter_Mode_Single;
  CAN_FilterInitStruct->Filter_FrameFormat = CAN_Filter_FrameFormat_Extended;
  CAN_FilterInitStruct->Filter1_Type = CAN_Filter_Type_Data;
  CAN_FilterInitStruct->Filter1_RTRCmd = DISABLE;
  CAN_FilterInitStruct->Filter1_Code = 0xFFFFFFFF;
  CAN_FilterInitStruct->Filter1_Mask = 0xFFFFFFFF;
}

/**
 * @brief  Set peripheral CAN to reset mode.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  NewState: new state of the CAN Reser mode.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void CAN_Reset(CAN_TypeDef* CANx, FunctionalState NewState)
{
  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  if(NewState != DISABLE)
  {
    CANx->ISR_SR_CMR_MR |= CAN_MR_RM;
  }
  else
  {
    CANx->ISR_SR_CMR_MR &= (~CAN_MR_RM);
  }
}

/**
 * @brief  Use peripheral CAN to send the transmission command.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CAN_Cmd: specifies the CAN transmission cmd.
 *         This parameter can be one of the following flags:
 *         @arg @ref CAN_Cmd_TransmitAbort
 *         @arg @ref CAN_Cmd_TransmitStart
 *         @arg @ref CAN_Cmd_TransmitSingle
 * @return None
 */
void CAN_SendCmd(CAN_TypeDef* CANx, uint32_t CAN_Cmd)
{
  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  CANx->ISR_SR_CMR_MR |= CAN_Cmd;
}

/**
 * @brief  Initiates the transmission of a message in class mode.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  Message: pointer to a structure which contains CAN info.
 * @param  AutoRetransmissionEnable: ENABLE or DISABLE AutoRetransmission.
 * @return None
 */
void CAN_Transmit(CAN_TypeDef* CANx, CAN_Msg* Message, uint32_t AutoRetransmissionEnable)
{
  uint32_t tmpreg, i;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  if(Message->IDType == CAN_IDType_Standard)
  {
    tmpreg = (((uint32_t)(Message->IDType) << 7) |
              ((uint32_t)(Message->FrameType) << 6) |
              ((uint32_t)(Message->DataLength)) |
              ((((uint32_t)(Message->Identifier) >> 3) & 0xFF) << 8) |
              ((((uint32_t)(Message->Identifier) & 0x07 ) << 5) << 16) |
              ((uint32_t)(Message->Data[0]) << 24 ));
    CANx->TXBUF = tmpreg;
    for(i = 1; i < Message->DataLength; i += 4)
    {
      CANx->TXBUF = *(uint32_t *)&Message->Data[i];
    }
  }
  else
  {
    tmpreg = (((uint32_t)(Message->IDType) << 7) |
              ((uint32_t)(Message->FrameType) << 6) |
              ((uint32_t)(Message->DataLength)) |
              (((Message->Identifier >> 21) & 0xFF) << 8) |
              (((Message->Identifier >> 13) & 0xFF) << 16) |
              (((Message->Identifier >> 5) & 0xFF) << 24));
    CANx->TXBUF = tmpreg;

    tmpreg = (((Message->Identifier & 0X1F) << 3) |
              (Message->Data[0] << 8) |
              (Message->Data[1] << 16) |
              (Message->Data[2] << 24));
    CANx->TXBUF = tmpreg;

    for (i = 3; i < Message->DataLength; i += 4)
    {
      CANx->TXBUF = *(uint32_t *)&Message->Data[i];
    }
  }

  if(AutoRetransmissionEnable == ENABLE)
    CANx->ISR_SR_CMR_MR |= CAN_TransmitMode_Retransmission;
  else
    CANx->ISR_SR_CMR_MR |= CAN_TransmitMode_Single;
}

/**
 * @brief  Receives a message in class mode.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  Message: pointer to a structure receive message.
 * @return None
 */
void CAN_Receive(CAN_TypeDef* CANx, CAN_Msg* Message)
{
  uint32_t tmpreg, i;

  tmpreg = CANx->RXBUF;
  if(tmpreg & CAN_IDType_Mask)
    Message->IDType = CAN_IDType_Extended;
  else
    Message->IDType = CAN_IDType_Standard;

  if(tmpreg & CAN_FrameType_Mask)
    Message->FrameType = CAN_FrameType_Remote;
  else
    Message->FrameType = CAN_FrameType_Data;

  Message->DataLength = (tmpreg & CAN_DataLength_Mask);

  if(Message->IDType == CAN_IDType_Standard)
  {
    Message->Identifier = ((tmpreg & (0xFFU << 8)) >> 5) | ((tmpreg & 0x00E00000U) >> 21);

    Message->Data[0] = (tmpreg >> 24) & 0xFF;
    for (i = 1; i < Message->DataLength; i += 4)
    {
      *(uint32_t *)&Message->Data[i] = CANx->RXBUF;
    }
  }
  else
  {
    Message->Identifier = (((tmpreg & (0xFFU << 8)) << 13) |
                             ((tmpreg & (0xFFU << 16)) >> 3) |
                             ((tmpreg & (0xFFU << 24)) >> 19));

    tmpreg = CANx->RXBUF;
    Message->Identifier |= ((tmpreg & (0x1F << 3)) >> 3);

    Message->Data[0] = (tmpreg & (0xFFU << 8)) >> 8;
    Message->Data[1] = (tmpreg & (0xFFU << 16)) >> 16;
    Message->Data[2] = (tmpreg & (0xFFU << 24)) >> 24;
    for (i = 3; i < Message->DataLength; i += 4)
    {
      *(uint32_t *)&Message->Data[i] = CANx->RXBUF;
    }
  }

  CANx->ISR_SR_CMR_MR |= CAN_ISR_RI;
}

/**
 * @brief  Initiates the transmission of a message in FD mode.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  Message: pointer to a structure which contains CAN FD info.
 * @param  AutoRetransmissionEnable: ENABLE or DISABLE AutoRetransmission.
 * @return None
 */
void CANFD_Transmit(CAN_TypeDef* CANx, CANFD_Msg* Message, uint32_t AutoRetransmissionEnable)
{
  uint32_t tmpreg, i, length;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  length = CAN_LengthTODLCTable[Message->DataLength];
  if(Message->IDType == CAN_IDType_Standard)
  {
    tmpreg = (((uint32_t)(Message->IDType) << 7) |
              ((uint32_t)(Message->FrameType) << 6) |
              ((uint32_t)(Message->FrameFormat) << 5) |
              ((uint32_t)(Message->BitRateSwitch) << 4) |
              (length) |
              ((uint32_t)(Message->ErrorStateIndicator) << 19) |
              ((((uint32_t)(Message->Identifier) >> 3) & 0xFF) << 8) |
              ((((uint32_t)(Message->Identifier) & 0x07) << 5) << 16) |
              ((uint32_t)(Message->Data[0]) << 24));
    CANx->TXBUF = tmpreg;
    for(i = 1; i < Message->DataLength; i += 4)
    {
      CANx->TXBUF = *(uint32_t *)&Message->Data[i];
    }
  }

  else
  {
    tmpreg = (((uint32_t)(Message->IDType) << 7) |
              ((uint32_t)(Message->FrameType) << 6) |
              ((uint32_t)(Message->FrameFormat) << 5) |
              ((uint32_t)(Message->BitRateSwitch) << 4) |
              (length) |
              (((Message->Identifier >> 21) & 0xFF) << 8) |
              (((Message->Identifier >> 13) & 0xFF) << 16) |
              (((Message->Identifier >> 5) & 0xFF) << 24));
    CANx->TXBUF = tmpreg;

    tmpreg = (((Message->Identifier & 0X1F) << 3) |
              ((uint32_t)(Message->BitRateSwitch) << 2) |
              ((uint32_t)(Message->Data[0]) << 8) |
              ((uint32_t)(Message->Data[1]) << 16) |
              ((uint32_t)(Message->Data[2]) << 24));
    CANx->TXBUF = tmpreg;

    for (i = 3; i < Message->DataLength; i += 4)
    {
      CANx->TXBUF = *(uint32_t *)&Message->Data[i];
    }
  }

  if(AutoRetransmissionEnable == ENABLE)
    CANx->ISR_SR_CMR_MR |= CAN_TransmitMode_Retransmission;
  else
    CANx->ISR_SR_CMR_MR |= CAN_TransmitMode_Single;
}

/**
 * @brief  Receives a message in FD mode.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  Message: pointer to a structure receive message.
 * @return None
 */
void CANFD_Receive(CAN_TypeDef* CANx, CANFD_Msg* Message)
{
  uint32_t tmpreg, i;

  tmpreg = CANx->RXBUF;
  if(tmpreg & CAN_IDType_Mask)
    Message->IDType = CAN_IDType_Extended;
  else
    Message->IDType = CAN_IDType_Standard;

  if(tmpreg & CAN_FrameType_Mask)
    Message->FrameType = CAN_FrameType_Remote;
  else
    Message->FrameType = CAN_FrameType_Data;

  if(tmpreg & CAN_FrameFormat_Mask)
    Message->FrameFormat = CAN_FrameFormat_FD;
  else
    Message->FrameFormat = CAN_FrameFormat_Class;
  
  if(tmpreg & CAN_BitRateSwitch_Mask)
    Message->FrameFormat = CAN_BitRateSwitch_Switch;
  else
    Message->FrameFormat = CAN_BitRateSwitch_WithoutSwitch;

  Message->DataLength = CAN_DLCTOLengthTable[(tmpreg & CAN_DataLength_Mask)];
  
  if(Message->IDType == CAN_IDType_Standard)
  {
    Message->Identifier = ((tmpreg & (0xFFU << 8)) >> 5) | ((tmpreg & 0x00E00000U) >> 21);
    
    if(tmpreg & CAN_ErrorState_Mask_Std)
      Message->ErrorStateIndicator = CAN_ErrorStateIndicator_PassiveErrorNode;
    else
      Message->ErrorStateIndicator = CAN_ErrorStateIndicator_ActiveErrorNode;

    Message->Data[0] = (tmpreg >> 24) & 0xFF;
    for (i = 1; i < Message->DataLength; i += 4)
    {
      *(uint32_t *)&Message->Data[i] = CANx->RXBUF;
    }
  }
  else
  {
    Message->Identifier = (((tmpreg & (0xFFU << 8)) << 13) |
                           ((tmpreg & (0xFFU << 16)) >> 3) |
                           ((tmpreg & (0xFFU << 24)) >> 19));

    tmpreg = CANx->RXBUF;
    Message->Identifier |= ((tmpreg & (0x1F << 3)) >> 3);

    if(tmpreg & CAN_ErrorState_Mask_Ext)
      Message->ErrorStateIndicator = CAN_ErrorStateIndicator_PassiveErrorNode;
    else
      Message->ErrorStateIndicator = CAN_ErrorStateIndicator_ActiveErrorNode;

    Message->Data[0] = (tmpreg & (0xFFU << 8)) >> 8;
    Message->Data[1] = (tmpreg & (0xFFU << 16)) >> 16;
    Message->Data[2] = (tmpreg & (0xFFU << 24)) >> 24;
    for (i = 3; i < Message->DataLength; i += 4)
    {
      *(uint32_t *)&Message->Data[i] = CANx->RXBUF;
    }
  }

  CANx->ISR_SR_CMR_MR |= CAN_ISR_RI;
}

/**
 * @brief  Enables or disables the specified CANx interrupts.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CAN_IT: specifies the CAN interrupt sources to be enabled or disabled.
 *         This parameter can be any combination of the following values:
 *         @arg @ref CAN_IT_DOI: CAN Data Overrun Interrupt Enable
 *         @arg @ref CAN_IT_BEI: CAN Buss Error Interrupt Enable
 *         @arg @ref CAN_IT_TI: CAN Transmission Interrupt Enable
 *         @arg @ref CAN_IT_RI: CAN Receive Interrupt Enable
 *         @arg @ref CAN_IT_EPI: CAN Error Passive Interrupt Enable
 *         @arg @ref CAN_IT_EWI: CAN Error Warning Interrupt Enable
 *         @arg @ref CAN_IT_ALI: CAN Arbitration Lost Interrupt Enable
 * @param  NewState: new state of the CAN interrupts.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void CAN_ITConfig(CAN_TypeDef* CANx, uint32_t CAN_IT, FunctionalState NewState)
{
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  if(NewState != DISABLE)
  {
    CANx->BT1_BT0_RMC_IMR |= CAN_IT;
  }
  else
  {
    CANx->ISR_SR_CMR_MR &= (~CAN_IT);
  }
}

/**
 * @brief  Checks whether the specified CAN flag is set or not.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CAN_FLAG: specifies the flag to check.
 *         This parameter can be one of the following flags:
 *         @arg @ref CAN_Flag_DOI: CAN Data Overrun Interrupt Flag Bit
 *         @arg @ref CAN_Flag_BEI: CAN Buss Error Interrupt Flag Bit
 *         @arg @ref CAN_Flag_TI: CAN Transmission Interrupt Flag Bit
 *         @arg @ref CAN_Flag_RI: CAN Receive Interrupt Flag Bit
 *         @arg @ref CAN_Flag_EPI: CAN Error Passive Interrupt Flag Bit
 *         @arg @ref CAN_Flag_EWI: CAN Error Warning Interrupt Flag Bit
 *         @arg @ref CAN_Flag_ALI: CAN Arbitration Lost Interrupt Flag Bit
 * @return The new state of CAN_FLAG (SET or RESET).
 */
FlagStatus CAN_GetITStatus(CAN_TypeDef* CANx, uint32_t CAN_Flag)
{
  FlagStatus bitstatus = RESET;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  /* Check the status of the specified SPI/I2S flag */
  if ((CANx->ISR_SR_CMR_MR & CAN_Flag) != (uint32_t)RESET)
  {
    /* CAN Status is set */
    bitstatus = SET;
  }
  else
  {
    /* CAN Status is reset */
    bitstatus = RESET;
  }
  /* Return the CAN status */
  return  bitstatus;
}

/**
 * @brief  Clears the CAN's pending flags.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CAN_FLAG: specifies the flag to clear.
 *         This parameter can be one of the following flags:
 *         @arg @ref CAN_Flag_DOI: CAN Data Overrun Interrupt Flag Bit
 *         @arg @ref CAN_Flag_BEI: CAN Buss Error Interrupt Flag Bit
 *         @arg @ref CAN_Flag_TI: CAN Transmission Interrupt Flag Bit
 *         @arg @ref CAN_Flag_RI: CAN Receive Interrupt Flag Bit
 *         @arg @ref CAN_Flag_EPI: CAN Error Passive Interrupt Flag Bit
 *         @arg @ref CAN_Flag_EWI: CAN Error Warning Interrupt Flag Bit
 *         @arg @ref CAN_Flag_ALI: CAN Arbitration Lost Interrupt Flag Bit
 * @return None
 */
void CAN_ClearITPendingBit(CAN_TypeDef* CANx, uint32_t CAN_Flag)
{
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  CANx->ISR_SR_CMR_MR |= CAN_Flag;
}

/**
 * @brief  Checks whether the specified CAN status flag is set or not.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CAN_Status: specifies the flag to check.
 *         This parameter can be one of the following flags:
 *         @arg @ref CAN_Status_BS：CAN Bus off Status
 *         @arg @ref CAN_Status_ES：CAN Error Status
 *         @arg @ref CAN_Status_TS：CAN Transmit Status
 *         @arg @ref CAN_Status_RS：CAN Receive Status
 *         @arg @ref CAN_Status_TBS：CAN Transmit Buffer Status
 *         @arg @ref CAN_Status_DSO：CAN Data Overrun Status
 *         @arg @ref CAN_Status_RBS：CAN Receive Buffer Status
 * @return The new state of CAN_FLAG (SET or RESET).
 */
FlagStatus CAN_GetFlagStatus(CAN_TypeDef* CANx, uint32_t CAN_Status)
{
  FlagStatus bitstatus = RESET;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  /* Check the status of the specified SPI/I2S flag */
  if ((CANx->ISR_SR_CMR_MR & CAN_Status) != (uint32_t)RESET)
  {
    /* CAN Status is set */
    bitstatus = SET;
  }
  else
  {
    /* CAN Status is reset */
    bitstatus = RESET;
  }
  /* Return the CAN status */
  return  bitstatus;
}

/**
 * @brief  Checks CAN arbitrations lost capture position.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return The arbitrations lost capture position.
 */
uint8_t CAN_GetArbitrationLosePosition(CAN_TypeDef* CANx)
{
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  return (CANx->ALC_TXERR_RXERR_ECC & CAN_ALC);
}

/**
 * @brief  Checks CAN arbitrations error count.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return The arbitrations error count.
 */
uint8_t CANFD_GetArbitrationErrorCount(CAN_TypeDef* CANx)
{
  uint32_t tmpreg;

  assert_param(IS_CAN_ALL_PERIPH(CANx));

  tmpreg = CANx->APERR_DPERR_FDSR_FDCFG;
  return (tmpreg & CAN_APERR_APERR);
}

/**
 * @brief  Checks CAN data error count.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return The data error count.
 */
uint8_t CANFD_GetDataErrorCount(CAN_TypeDef* CANx)
{
  uint32_t tmpreg;

  assert_param(IS_CAN_ALL_PERIPH(CANx));

  tmpreg = CANx->APERR_DPERR_FDSR_FDCFG;
  return (tmpreg & CAN_DPERR_DPXERR);
}

/**
 * @brief  Checks CAN FD running status.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CANFD_Status:  specifies the status to check.
 *         This parameter can be one of the following flags:
 *         @arg @ref CANFD_Status_Init：CAN Init Status
 *         @arg @ref CANFD_Status_Idle：CAN Idle Status
 *         @arg @ref CANFD_Status_Recv：CAN Recv Status
 *         @arg @ref CANFD_Status_Send：CAN Send Statuss
 * @return The new state of CAN_FLAG (SET or RESET).
 */
FlagStatus CANFD_GetStatus(CAN_TypeDef* CANx, uint32_t CANFD_Status)
{
  uint32_t tmpreg;

  assert_param(IS_CAN_ALL_PERIPH(CANx));

  tmpreg = CANx->APERR_DPERR_FDSR_FDCFG;
  if((tmpreg & CAN_FDSTA_STATE) == CANFD_Status)
    return SET;
  else
    return RESET;
}

/**
 * @brief  Checks CAN FD error.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @param  CANFD_Status:  specifies the error to check.
 *         This parameter can be one of the following flags:
 *         @arg @ref CANFD_Error_BITERR：CAN Bit Error.
 *         @arg @ref CANFD_Error_CRCERR：CAN Cyclic Redundancy Check Error.
 *         @arg @ref CANFD_Error_FRMERR：CAN Form Error in Data Phase.
 *         @arg @ref CANFD_Error_STFERR：CAN Bit filling Error.
 * @return The new state of CAN_FLAG (SET or RESET).
 */
FlagStatus CANFD_GetError(CAN_TypeDef* CANx, uint32_t CANFD_Error)
{
  FlagStatus bitstatus = RESET;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  /* Check the status of the specified SPI/I2S flag */
  if ((CANx->APERR_DPERR_FDSR_FDCFG & CANFD_Error) != (uint32_t)RESET)
  {
    /* CAN Status is set */
    bitstatus = SET;
  }
  else
  {
    /* CAN Status is reset */
    bitstatus = RESET;
  }
  /* Return the CAN status */
  return bitstatus; 
}

/**
 * @brief  Checks CAN FD Secondary Sample Position.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return The Secondary Sample Position.
 */
uint8_t CANFD_GetSecondarySamplePosition(CAN_TypeDef* CANx)
{
  uint32_t tmpreg;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  tmpreg = CANx->SSPP_TDCR_DBT;
  tmpreg = (((tmpreg & CAN_SSPP_SSPP) >> CAN_SSPP_SSPP_Pos) & 0x7F);

  return tmpreg;
}

/**
 * @brief  Get CAN number of framed in the rxbuf.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return The number of data.
 */
uint8_t CAN_GetMessageCount(CAN_TypeDef* CANx)
{
  uint32_t tmpreg;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  tmpreg = CANx->BT1_BT0_RMC_IMR;
  tmpreg = (((tmpreg & CAN_RMC) >> CAN_RMC_Pos) & 0x1F);

  return tmpreg; 
}

/**
 * @brief  Get CAN Transmit Error Count.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return The number of transmit error.
 */
uint8_t CAN_GetTransmitErrorCount(CAN_TypeDef* CANx)
{
  uint32_t tmpreg;

  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  tmpreg = CANx->ALC_TXERR_RXERR_ECC;
  tmpreg = (((tmpreg & CAN_TXERR) >> CAN_TXERR_Pos) & 0xFF);

  return tmpreg; 
}

/**
 * @brief  Get CAN Receive Error Count.
 * @param  CANx: where x can be 1 select the CAN peripheral.
 * @return The number of receive error.
 */
uint8_t CAN_GetReceiveErrorCount(CAN_TypeDef* CANx)
{
  uint32_t tmpreg;
  
  /* Check the parameters */
  assert_param(IS_CAN_ALL_PERIPH(CANx));

  tmpreg = CANx->ALC_TXERR_RXERR_ECC;
  tmpreg = (((tmpreg & CAN_RXERR) >> CAN_RXERR_Pos) & 0xFF);

  return tmpreg; 
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
