/**
 * @file    mg32f157_aes.c
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file provides all the AES firmware functions.
 */

/* Includes ------------------------------------------------------------------*/
#include "mg32f157_aes.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @defgroup AES 
  * @brief AES driver modules
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

/** @defgroup AES_Private_Defines
  * @{
  */

#define AES_Set                            ((uint32_t)0x00000001)
#define AES_Reset                          ((uint32_t)0x00000000)

#define AES_IT_MASK                        ((uint32_t)0xFFFBFFFF)

#define AES_ERRC                           ((uint16_t)0x0100)
#define AES_CCFC                           ((uint16_t)0x0080)
#define AES_ERRC_CCFC                      ((uint16_t)0x0180)

#define CHMOD_ECB                          ((uint32_t)0x00000000)
#define CHMOD_CBC                          ((uint32_t)0x00000020)
#define CHMOD_CTR                          ((uint32_t)0x00000040)

#define MODE_Encryption                    ((uint32_t)0x00000000)
#define MODE_Derivation                    ((uint32_t)0x00000008)
#define MODE_Decryption                    ((uint32_t)0x00000010)
#define MODE_Derivation_Decryption         ((uint32_t)0x00000018)

#define DATATYPE_32Bits                    ((uint32_t)0x00000000)
#define DATATYPE_16Bits                    ((uint32_t)0x00000002)
#define DATATYPE_8Bits                     ((uint32_t)0x00000004)
#define DATATYPE_1Bit                      ((uint32_t)0x00000006)

/**
  * @}
  */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/** @defgroup AES_Private_Functions
  * @{
  */
  
/**
 * @brief  Enables or disables the AES.
 * @param  NewState: new state of the AES.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void AES_Cmd(FunctionalState NewState)
{
  assert_param(IS_FUNCTIONAL_STATE(NewState));

  if(NewState != DISABLE)
  {
    AES->CR |= AES_Set;
  }
  else
  {
    AES->CR = AES_Reset;
  }
}

/**
 * @brief  Deinitializes the AES peripheral registers to their default reset values.
 * @param  None
 * @return None
 */
void AES_DeInit(void)
{
  AES->CR = AES_Reset;
  AES->SR = AES_Reset;
  
  AES->DINR = AES_Reset;
  AES->DOUTR = AES_Reset;
  
  AES->KEYR0 = AES_Reset;
  AES->KEYR1 = AES_Reset;
  AES->KEYR2 = AES_Reset;
  AES->KEYR3 = AES_Reset;
  
  AES->IVR0 = AES_Reset;
  AES->IVR1 = AES_Reset;
  AES->IVR2 = AES_Reset;
  AES->IVR3 = AES_Reset;
}

/**
 * @brief  Fills each AES_InitStruct member with its default value.
 * @param  AES_InitStruct: pointer to a AES_InitTypeDef structure
 *         which will be initialized.
 * @return None
 */
void AES_StructInit(AES_InitTypeDef* AES_InitStruct)
{
  /* Reset AES init structure parameters values */
  AES_InitStruct->AES_CHMOD = AES_CHMOD_ECB;
  AES_InitStruct->AES_MODE = AES_MODE_Encryption;
  AES_InitStruct->AES_DATATYPE = AES_DATATYPE_32Bits;
}

/**
 * @brief  Initializes the AES peripheral according to the specified
 *         parameters in the AES_InitStruct.
 * @param  AES_InitStruct: pointer to a AES_InitTypeDef structure that
 *         contains the configuration information for the specified AES peripheral.
 * @return None
 */
void AES_Init(AES_InitTypeDef* AES_InitStruct)
{
  /* Check the parameters */
  assert_param(IS_AES_CHMOD(AES_InitStruct->AES_CHMOD));
  assert_param(IS_AES_MODE(AES_InitStruct->AES_MODE));
  assert_param(IS_AES_DATATYPE(AES_InitStruct->AES_DATATYPE));
  
  /* Config AES with the selected working mode */
  if(AES_InitStruct->AES_CHMOD == AES_CHMOD_ECB)
  {
    AES->CR |= CHMOD_ECB;
  }
  else if(AES_InitStruct->AES_CHMOD == AES_CHMOD_CBC)
  {
    AES->CR |= CHMOD_CBC;
  }
  else
  {
    if(AES_InitStruct->AES_CHMOD == AES_CHMOD_CTR)
      AES->CR |= CHMOD_CTR;
  }

  /* Config AES with the selected block chaining mode */
  switch (AES_InitStruct->AES_MODE)
  {
    case AES_MODE_Encryption:
      AES->CR |= MODE_Encryption;
      break;
    case AES_MODE_Decryption:
      AES->CR |= MODE_Decryption;
      break;
    case AES_MODE_Derivation:
      AES->CR |= MODE_Derivation;
      break;
    case AES_MODE_Derivation_Decryption:
      AES->CR |= MODE_Derivation_Decryption;
      break;
    default:
      break;
  }

  /* Config AES with the selected datatype */
  switch (AES_InitStruct->AES_DATATYPE)
  {
    case AES_DATATYPE_32Bits:
      AES->CR |= DATATYPE_32Bits;
      break;
    case AES_DATATYPE_16Bits:
      AES->CR |= DATATYPE_16Bits;
      break;
    case AES_DATATYPE_8Bits:
      AES->CR |= DATATYPE_8Bits;
      break;
    case AES_DATATYPE_1Bit:
      AES->CR |= DATATYPE_1Bit;
      break;
    default:
      break;
  }
}

/**
 * @brief  Enables or disables the specified AES interrupts.
 * @param  AES_IT: specifies the AES interrupt sources to be enabled or disabled.
 *         This parameter can be any combination of the following values:
 *         @arg @ref AES_IT_ERR: Error interrupt mask
 *         @arg @ref AES_IT_CCF: Calulation completly interrupt mask
 * @param  NewState: new state of the specified AES interrupts.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void AES_IT_Config(uint32_t AES_IT, FunctionalState NewState)
{
  /* Check the parameters */
  assert_param(IS_AES_IT(AES_IT));
  assert_param(IS_FUNCTIONAL_STATE(NewState));

  if(NewState != DISABLE)
  {
    if(AES_IT == AES_IT_ERR)
    {
      AES->CR |= (uint16_t)(AES_IT_ERR >> 16);
    }
    else if(AES_IT == AES_IT_CCF)
    {
      AES->CR |= (uint16_t)(AES_IT_CCF >> 16);
    }
    else
    {
      AES->CR |= (uint16_t)((AES_IT_CCF | AES_IT_ERR) >> 16);
    }
  }
  else
  {
    AES->CR &= AES_IT_MASK;
  }
}

/**
 * @brief  Checks whether the specified AES flag is set or not.
 * @param  AES_FLAG: specifies the flag to check.
 *         This parameter can be one of the following values:
 *         @arg @ref AES_FLAG_WRERR: Write error flag
 *         @arg @ref AES_FLAG_RDERR: Read error flag
 *         @arg @ref AES_FLAG_CCF: Calculation completly flag
 * @return The new state of AES_FLAG (SET or RESET).
 */
FlagStatus AES_GetFlagStatus(uint8_t AES_FLAG)
{
  FlagStatus bitstatus = RESET;

  /* Check the parameters */
  assert_param(IS_AES_GET_FLAG(AES_FLAG));

  /* Check the status of the specified AES flag */
  if ((AES->SR & AES_FLAG) != (uint8_t)RESET)
  {
    /* AES_FLAG is set */
    bitstatus = SET;
  }
  else
  {
    /* AES_FLAG is reset */
    bitstatus = RESET;
  }

  /* Return the AES_FLAG status */
  return  bitstatus;
}

/**
 * @brief  Clears the AES's pending flags.
 * @param  AES_FLAG: specifies the flag to clear.
 *         This parameter can be one of the following values:
 *         @arg @ref AES_FLAG_WRERR: Write error flag
 *         @arg @ref AES_FLAG_RDERR: Read error flag
 *         @arg @ref AES_FLAG_CCF: Calculation completly flag
 * @return None
 */
void AES_ClearFlag(uint8_t AES_FLAG)
{
  /* Check the parameters */
  assert_param(IS_AES_CLEAR_FLAG(AES_FLAG));

  /* Clear the selected AES flags */
  if((AES_FLAG == AES_FLAG_WRERR) || (AES_FLAG == AES_FLAG_RDERR))
  {
    /* Clear write or read error flag */
    AES->CR |= (uint32_t)AES_ERRC;
  }
  else
  {
    /* Clear calculation completly flag */
    AES->CR |= (uint32_t)AES_ERRC_CCFC;
  }
}

/**
 * @brief  Checks whether the specified AES interrupt has occurred or not.
 * @param  AES_IT: specifies the interrupt source to check.
 *         This parameter can be one of the following values:
 *         @arg @ref AES_IT_ERR: Error interrupt
 *         @arg @ref AES_IT_CCF: Calculation completly interrupt
 * @return The new state of AES_IT (SET or RESET).
 */
ITStatus AES_GetITStatus(uint32_t AES_IT)
{
  ITStatus bitstatus = RESET;
  uint16_t itstatus = 0, itenable = 0;
  
  /* Check the parameters */
  assert_param(IS_AES_GET_IT(AES_IT));
  
  /* Get the AES enable status*/
  itenable = AES->CR & ((uint16_t)(AES_IT >> 16));
  /* Get the AES interrupt status */
  itstatus = (AES->SR & (uint16_t)AES_IT);
  
  if (itenable && itstatus)
  {
    /* AES_IT is set */
    bitstatus = SET;
  }
  else
  {
    /* AES_IT is reset */
    bitstatus = RESET;
  }

  /* Return the AES_IT status */
  return  bitstatus;
}

/**
 * @brief  Clears the AES's interrupt pending bits.
 * @param  AES_IT: specifies the interrupt pending bit to clear.
 *         This parameter can be any combination of the following values:
 *         @arg @ref AES_IT_ERR: Error interrupt
 *         @arg @ref AES_IT_CCF: Calculation completly interrupt
 * @return None
 */
void AES_ClearITPendingBit(uint32_t AES_IT)
{
  /* Check the parameters */
  assert_param(IS_AES_IT(AES_IT));
  
  if(AES_IT == AES_IT_ERR)
  {
    AES->CR |= (uint32_t)AES_ERRC;
  }
  else if(AES_IT == AES_IT_CCF)
  {
    AES->CR |= (uint32_t)AES_CCFC;
  }
  else
  {
    AES->CR |= (uint32_t)(AES_CCFC | AES_ERRC);
  }
}

/**
 * @brief  Writes the Key in Key registers.
 * @param  Key: Pointer to Key buffer.
 * @note   Key must be written as little endian.
 *         If Key pointer points at address n,
 *         n[15:0] contains key[96:127],
 *         (n+4)[15:0] contains key[64:95],
 *         (n+8)[15:0] contains key[32:63] and
 *         (n+12)[15:0] contains key[0:31]
 * @return None
 */
void AES_SetKey(uint8_t *Key)
{  
  AES->KEYR3 = __REV(*((uint32_t*)Key));
  Key += 4;
  AES->KEYR2 = __REV(*((uint32_t*)Key));
  Key += 4;
  AES->KEYR1 = __REV(*((uint32_t*)Key));
  Key += 4;
  AES->KEYR0 = __REV(*((uint32_t*)Key));
}

/**
 * @brief  Writes the InitVector/InitCounter in IV registers.
 * @param  InitVector : Pointer to InitVector/InitCounter buffer
 * @note   Init Vector must be written as little endian.
 *         If Init Vector pointer points at address n,
 *         n[15:0] contains Vector[96:127],
 *         (n+4)[15:0] contains Vector[64:95],
 *         (n+8)[15:0] contains Vector[32:63] and
 *         (n+12)[15:0] contains Vector[0:31]
 * @return None
 */
void AES_SetInitVector(uint8_t *InitVector)
{
  AES->IVR3 = __REV(*((uint32_t*)(InitVector)));
  InitVector += 4;
  AES->IVR2 = __REV(*((uint32_t*)(InitVector)));
  InitVector += 4;
  AES->IVR1 = __REV(*((uint32_t*)(InitVector)));
  InitVector += 4;
  AES->IVR0 = __REV(*((uint32_t*)(InitVector)));
}

/**
 * @brief  Writes the origin data into AES DINR register.
 * @param  inputaddr: Pointer to input data buffer.
 * @note   Set the 128bits input data by writing the AES
 *         Input Data Register(DINR) 4 times.
 * @return None
 */
void AES_DataInput(uint32_t *inputaddr)
{
  /* Write the Input block into the DINR register */
  AES->DINR = *((uint32_t*)(inputaddr++));
  AES->DINR = *((uint32_t*)(inputaddr++));
  AES->DINR = *((uint32_t*)(inputaddr++));
  AES->DINR = *((uint32_t*)(inputaddr));
}

/**
 * @brief  Gets the output data from AES DOUTR register.
 * @param  outputaddr: Pointer to output data buffer.
 * @note   Get the 128bits output data by reading the AES
 *         Output Data Register(DOUTR) 4 times.
 * @return None
 */
void AES_DataOutput(uint32_t *outputaddr)
{
  /* Get the Output block in the DOUTR register */
  *((uint32_t*)(outputaddr++)) = AES->DOUTR;
  *((uint32_t*)(outputaddr++)) = AES->DOUTR;
  *((uint32_t*)(outputaddr++)) = AES->DOUTR;
  *((uint32_t*)(outputaddr)) = AES->DOUTR;
}

/**
 * @brief  Enables or disables the specified AES DMA request.
 * @param  AES_DMA: specifies the DMA input request or DAM output request.
 *         This parameter can be: AES_DMA_IN or AES_DMA_OUT.
 * @param  NewState: new state of the specified AES DMAC request.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void AES_DMA_Cmd(uint16_t AES_DMA, FunctionalState NewState)
{
  assert_param(IS_AES_DMA(AES_DMA));
  
  if(NewState != DISABLE)
  {
    /* Enable the specified AES DMA request */
    if(AES_DMA == AES_DMA_IN)
    {
      AES->CR |= (uint32_t)AES_DMA_IN;
    }
    else
    {
      AES->CR |= (uint32_t)AES_DMA_OUT;
    }
  }
  /* Disable the specified AES DMA request */
  else
  {
    if(AES_DMA == AES_DMA_IN)
    {
      AES->CR &= (~(uint32_t)AES_DMA_IN);
    }
    else
    {
      AES->CR &= (~(uint32_t)AES_DMA_OUT);
    }
  }
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
