/**
 * @file    mg32f157_AES.h
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file contains all the functions prototypes for the AES firmware
 *          library.
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MG32F157_AES_H
#define __MG32F157_AES_H

#ifdef __cplusplus
 extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "mg32f157.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @addtogroup AES
  * @{
  */

/* Exported types ------------------------------------------------------------*/

/**
  * @brief  AES chaining mode definition  
  */
typedef enum
{ 
  AES_CHMOD_ECB,
  AES_CHMOD_CBC,
  AES_CHMOD_CTR
}AESChMod_TypeDef;

#define IS_AES_CHMOD(MODE) (((MODE) == AES_CHMOD_ECB) || ((MODE) == AES_CHMOD_CBC) || \
                            ((MODE) == AES_CHMOD_CTR))

/**
  * @brief  AES working mode definition  
  */
typedef enum
{ 
  AES_MODE_Encryption,
  AES_MODE_Decryption, 
  AES_MODE_Derivation,
  AES_MODE_Derivation_Decryption
}AESMode_TypeDef;

#define IS_AES_MODE(MODE) (((MODE) == AES_MODE_Encryption) || ((MODE) == AES_MODE_Decryption) || \
                           ((MODE) == AES_MODE_Derivation) || ((MODE) == AES_MODE_Derivation_Decryption))


/**
  * @brief  AES datatype definition
  */
typedef enum
{ 
  AES_DATATYPE_32Bits,
  AES_DATATYPE_16Bits, 
  AES_DATATYPE_8Bits,
  AES_DATATYPE_1Bit
}AESData_TypeDef;

#define IS_AES_DATATYPE(DATA) (((DATA) == AES_DATATYPE_32Bits) || ((DATA) == AES_DATATYPE_16Bits) || \
                               ((DATA) == AES_DATATYPE_8Bits) || ((DATA) == AES_DATATYPE_1Bit))

/** 
  * @brief  AES Init structure definition  
  */
typedef struct
{
  AESChMod_TypeDef AES_CHMOD;             /*!< Specifies the AES CHMOD to be configured.
                                      This parameter can be any value of @ref AESChMod_TypeDef */

  AESMode_TypeDef AES_MODE;             /*!< Specifies the AES MOD to be configured.
                                      This parameter can be a value of @ref AESMode_TypeDef */

  AESData_TypeDef AES_DATATYPE;             /*!< Specifies the AES pins to be configured.
                                      This parameter can be a value of @ref AESData_TypeDef */
}AES_InitTypeDef;

/* Exported constants --------------------------------------------------------*/

/** @defgroup AES_Exported_Constants
  * @{
  */

/** @defgroup AES_DMA
  * @{
  */
#define AES_DMA_IN                           ((uint16_t)0x0800)
#define AES_DMA_OUT                          ((uint16_t)0x1000)

#define IS_AES_DMA(AES_DMA)  (((AES_DMA) == AES_DMA_IN) || ((AES_DMA) == AES_DMA_OUT))
/**
  * @}
  */

/** @defgroup AES_IT 
  * @{
  */
#define AES_IT_ERR                           ((uint32_t)0x04000006)
#define AES_IT_CCF                           ((uint32_t)0x02000001)

#define IS_AES_IT(AES_IT)  (((uint16_t)(AES_IT >> 16) & 0x0600) != 0x0000)

#define IS_AES_GET_IT(AES_IT)  (((AES_IT) == AES_IT_ERR) || ((AES_IT) == AES_IT_CCF))
/**
  * @}
  */

/** @defgroup AES_FLAG
  * @{
  */
#define AES_FLAG_WRERR                       ((uint8_t)0x04)
#define AES_FLAG_RDERR                       ((uint8_t)0x02)
#define AES_FLAG_CCF                         ((uint8_t)0x01)

#define IS_AES_GET_FLAG(AES_FLAG)  (((AES_FLAG) == AES_FLAG_WRERR) || ((AES_FLAG) == AES_FLAG_RDERR) || ((AES_FLAG) == AES_FLAG_CCF))
#define IS_AES_CLEAR_FLAG(AES_FLAG)  (((AES_FLAG) == AES_FLAG_WRERR) || ((AES_FLAG) == AES_FLAG_RDERR) || ((AES_FLAG) == AES_FLAG_CCF))
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/

void AES_DeInit(void);
void AES_Cmd(FunctionalState NewState);
void AES_IT_Config(uint32_t AES_IT, FunctionalState NewState);
void AES_Init( AES_InitTypeDef* AES_InitStruct);
void AES_SetKey(uint8_t *Key);
void AES_SetInitVector(uint8_t *InitVector);
void AES_DataInput(uint32_t *inputaddr);
void AES_DataOutput(uint32_t *outputaddr);
void AES_DMA_Cmd(uint16_t AES_DMA, FunctionalState NewState);
FlagStatus AES_GetFlagStatus(uint8_t AES_FLAG);
void AES_ClearFlag(uint8_t AES_FLAG);
ITStatus AES_GetITStatus(uint32_t AES_IT);
void AES_ClearITPendingBit(uint32_t AES_IT);

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __MG32F157_AES_H */
