#include "drv_spi_flash_quad.h"

extern uint8_t *pTxData, *pRxData;
extern __IO int16_t ubRxIndex,ubTxIndex;
extern uint8_t aRxBuffer[];
extern uint8_t aTxBuffer[];
extern __IO uint8_t CmdCplt, RxCplt, TxCplt, StatusMatch;
extern __IO uint32_t NbrOfDataToTransfer;

/**
 * @brief  Configures the QSPI.
 * @param  None.
 * @return None.
 */
void SpiFlash_Init(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  QSPI_InitTypeDef QSPI_InitStructure;
  
  /* Peripheral Clock Enable -------------------------------------------------*/
  /* Enable the QSPI clock */
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_QUADSPI, ENABLE);
    
  /* Enable GPIO clocks */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOC | RCC_APB2Periph_GPIOD, ENABLE);
  
  /* QSPI GPIO Configuration -------------------------------------------------*/
  /**
   * QSPI_IO0: PC8
   * QSPI_IO1: PC9
   * QSPI_IO2: PC10
   * QSPI_IO3: PC11
   * QSPI_CLK: PC12
   * QSPI_CS:  PD2
   */
  /* Connect QSPI pins to Alternate function */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_8 | GPIO_Pin_9 | GPIO_Pin_10 | GPIO_Pin_11 | GPIO_Pin_12;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOC, &GPIO_InitStructure);
    
  /* QSPI CS pin configuration */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_2;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOD, &GPIO_InitStructure);
  
  /* QSPI NVIC Configuration -------------------------------------------------*/  
  /* Configure the SPI interrupt priority */
  NVIC_InitStructure.NVIC_IRQChannel = QUADSPI_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 1;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 1;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);  
  
  /* Initialize QSPI ------------------------------------------------------ */
  QSPI_StructInit(&QSPI_InitStructure);
  QSPI_InitStructure.QSPI_SShift    = QSPI_SShift_HalfCycleShift;
  QSPI_InitStructure.QSPI_Prescaler = 255; 
  QSPI_InitStructure.QSPI_CKMode    = QSPI_CKMode_Mode0;
  QSPI_InitStructure.QSPI_CSHTime   = QSPI_CSHTime_1Cycle;
  QSPI_InitStructure.QSPI_FSize     = 25;
  QSPI_InitStructure.QSPI_FSelect   = QSPI_FSelect_1;
  QSPI_InitStructure.QSPI_DFlash    = QSPI_DFlash_Disable;
  QSPI_Init(&QSPI_InitStructure);
  
  /* 4 bytes FIFO for qspi in indirect mode */ 
  QSPI_SetFIFOThreshold(3);  
  /* QSPI enable */
  QSPI_Cmd(ENABLE);
}

/**
 * @brief  Wait for a flag state until timeout.
 * @param  QSPI_FLAG: specifies the SPI flag to check. 
 *         This parameter can be one of the following values:
 *         @arg @ref QSPI_FLAG_BUSY: Busy flag.
 *         @arg @ref QSPI_FLAG_TO: Timeout flag.
 *         @arg @ref QSPI_FLAG_SM: Status match flag.
 *         @arg @ref QSPI_FLAG_FT: FIFO threshold flag.
 *         @arg @ref QSPI_FLAG_TC: Transfer complete flag.
 *         @arg @ref QSPI_FLAG_TE: Transfer error flag.
 * @param  Status:  Value of the flag expected.
 * @param  TimeOut:  Duration of the timeout.
 * @return The state of the wait timeout (SET or RESET).
 */
uint8_t QSPI_WaitFlagStateUntilTimeout(uint32_t QSPI_FLAG, uint8_t Status, uint32_t TimeOut)
{
  while (TimeOut--)
  {
    if (QSPI_GetFlagStatus(QSPI_FLAG)  == Status)
      return SET;
  }
  
  return RESET;
}

/**
 * @brief  Receive an amount of data in non-blocking mode with interrupt.
 * @param  None.
 * @note   This function is used only in Indirect Read Mode.
 * @return None.
 */
void QSPI_Receive_IT(void)
{
  uint32_t address = READ_REG(QUADSPI->AR);
  
  /* Configure QSPI: CCR register with functional as indirect read */
  MODIFY_REG(QUADSPI->CCR, QUADSPI_CCR_FMODE, QSPI_ComConfig_FMode_Indirect_Read);
  /* Start the transfer by re-writing the address in AR register */
  WRITE_REG(QUADSPI->AR, address);
  /* Clear interrupt */
  QSPI_ClearFlag(QSPI_FLAG_TE | QSPI_FLAG_TC);

  /* Enable the QSPI transfer error, FIFO threshold and transfer complete Interrupts */
  QSPI_ITConfig(QSPI_IT_TE | QSPI_IT_FT | QSPI_IT_TC, ENABLE);
}

/**
 * @brief  Send an amount of data in non-blocking mode with interrupt.
 * @param  None.
 * @note   This function is used only in Indirect Write Mode
 * @return None.
 */
void QSPI_Transmit_IT(void)
{
  /* Configure QSPI: CCR register with functional as indirect write */
  MODIFY_REG(QUADSPI->CCR, QUADSPI_CCR_FMODE, QSPI_ComConfig_FMode_Indirect_Write);

  /* Clear interrupt */
  QSPI_ClearFlag(QSPI_FLAG_TE | QSPI_FLAG_TC);
  
  /* Enable the QSPI transfer error, FIFO threshold and transfer complete Interrupts */
  QSPI_ITConfig(QSPI_IT_TE | QSPI_IT_FT | QSPI_IT_TC, ENABLE);
}

/**
 * @brief  Polls the status of the Write In Progress (WIP) flag in the FLASH's
 *         status register and loop until write operation has completed.
 * @param  None.
 * @return None.
 */
void SpiFlash_WaitReady(void)
{
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};
  
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_AutoPollingModeStopCmd(ENABLE);
  QSPI_AutoPollingMode_SetInterval(0x10);
  QSPI_AutoPollingMode_Config(0x00, 0x01, QSPI_PMM_AND);
  
  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Auto_Polling;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_ReadStatusReg1;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_BUSY, RESET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
    QSPI_SetDataLength(0);
    NbrOfDataToTransfer = 1;
  }
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_SM, SET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ClearFlag(QSPI_FLAG_SM);
  }
}

/**
 * @brief  This function send a Write Enable and wait it is effective.
 * @param  None.
 * @return None.
 */
void QSPI_WriteEnable(void)
{
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};
  
  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_NoData;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_WriteEnable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_BUSY, RESET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
    QSPI_SetAddress(0);
    QSPI_SetDataLength(0);
    NbrOfDataToTransfer = 0;
  }
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_TC, SET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ClearFlag(QSPI_FLAG_TC);
  }
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_BUSY, RESET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_AutoPollingModeStopCmd(ENABLE);
    QSPI_AutoPollingMode_SetInterval(0x10);
    QSPI_AutoPollingMode_Config(0x02, 0x02, QSPI_PMM_AND);
  }
    
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_ReadStatusReg1;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Auto_Polling;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_BUSY, RESET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
    QSPI_SetDataLength(0);
    NbrOfDataToTransfer = 0;
  }
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_SM, SET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ClearFlag(QSPI_FLAG_SM);
  }
  
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
}

/**
 * @brief  Read data by polling.
 * @param  buffer: The point to read buffer.
 * @return The point to received data.
 */
uint8_t* QSPI_Receive(uint8_t * buffer)
{
  uint32_t address = READ_REG(QUADSPI->AR);
  
  /* Configure QSPI: CCR register with functional as indirect read */
  MODIFY_REG(QUADSPI->CCR, QUADSPI_CCR_FMODE, QSPI_ComConfig_FMode_Indirect_Read);
  /* Start the transfer by re-writing the address in AR register */
  WRITE_REG(QUADSPI->AR, address);
  
  ubRxIndex = 0;
  
  while (ubRxIndex < NbrOfDataToTransfer)
  {
    /* Wait until FT or TC flag is set to read received data */
    if (QSPI_WaitFlagStateUntilTimeout((QSPI_FLAG_FT | QSPI_FLAG_TC), SET, QSPI_TIMEOUT_DEFAULT))
    {
      while(READ_BIT(QUADSPI->SR, QUADSPI_SR_FLEVEL) != 0)
      {
        if(ubRxIndex < NbrOfDataToTransfer)
        {
          buffer[ubRxIndex++] = QSPI_ReceiveData8();
        }
        else
        {
          QSPI_AbortRequest();
          break;
        }
      }
    }
  }
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_TC, SET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ClearFlag(QSPI_FLAG_TC);
  }
  
  return &buffer[0];
}

/**
 * @brief  Transmit data by polling.
 * @param  buffer: The point to write buffer.
 * @return None.
 */
void QSPI_Transmit(const uint8_t* buffer)
{    
 /* Configure QSPI: CCR register with functional as indirect write */
  MODIFY_REG(QUADSPI->CCR, QUADSPI_CCR_FMODE, QSPI_ComConfig_FMode_Indirect_Write);

  ubTxIndex = 0;
  
  while(ubTxIndex < NbrOfDataToTransfer)
  {
    if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_FT, SET, QSPI_TIMEOUT_DEFAULT))
    {
      QSPI_SendData8(buffer[ubTxIndex++]);
    }
    else
      QSPI_AbortRequest();
    
    if(ubTxIndex > NbrOfDataToTransfer)
      break;
  }
  
  if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_TC, SET, QSPI_TIMEOUT_DEFAULT) == SET)
  {
    QSPI_ClearFlag(QSPI_FLAG_TC);
  }
}

/**
 * @brief  Read Status Register-1 (05h)
 * @param  None.
 * @return Status Register-1
 */
uint8_t SpiFlash_ReadStatusReg1(void)
{
  uint8_t status;
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};

  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_ReadStatusReg1 ;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  QSPI_SetDataLength(0);
  NbrOfDataToTransfer = 1;
  
  ubRxIndex = 0;
  pRxData = &status;
  RxCplt = 0;
  QSPI_Receive_IT();
  
  while (RxCplt == 0);
  
  return status;
}

/**
 * @brief  Read Status Register-2 (35h)
 * @param  None.
 * @return Status Register-2
 */
uint8_t SpiFlash_ReadStatusReg2(void)
{
  uint8_t status;
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};

  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_ReadStatusReg2 ;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  QSPI_SetDataLength(0);
  NbrOfDataToTransfer = 1;
  
  ubRxIndex = 0;
  RxCplt = 0;
  pRxData = &status;
  QSPI_Receive_IT();
  
  while (RxCplt == 0);
      
  return status;
}

/**
 * @brief  Read Status Register-3 (15h)
 * @param  None.
 * @return Status Register-3
 */
uint8_t SpiFlash_ReadStatusReg3(void)
{
  uint8_t status;
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};

  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_ReadStatusReg3 ;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  QSPI_SetDataLength(0);
  NbrOfDataToTransfer = 1;
  
  ubRxIndex = 0;
  RxCplt = 0;
  pRxData = &status;
  QSPI_Receive_IT();
  
  while (RxCplt == 0);
  
  return status;
}

/**
 * @brief  Read JEDEC ID (9Fh).
 * @param  None.
 * @return 24bit JEDEC ID (Eg. the w25q32 id is 0xEF4016)
 */
uint32_t SpiFlash_ReadIdentification(void)
{
  uint32_t jedec_id = 0;
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};

  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_JedecDeviceID ;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  QSPI_SetDataLength(2);
  NbrOfDataToTransfer = 3;
  
  ubRxIndex = 0;
  pRxData = (uint8_t *)&jedec_id;
  RxCplt = 0;
  QSPI_Receive_IT();
  
  while (RxCplt == 0);
  
  return __REV(jedec_id) >> 8;
}

/**
 * @brief  Read Manufacturer / Device ID through Quad I/O
 * @param  None.
 * @return [15:8] is Manufacturer ID, [7:0] is Device ID
 *         (Eg. the w25q32 id is 0xEF15)
 */
uint16_t SpiFlash_QuadReadMidDid(void)
{
  uint16_t mid_id;
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};
  
  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_32bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_4Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_4Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_ManufactDeviceIDQSPI; // Read Manufacturer / Device ID Quad I/O (94h)
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 4;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  QSPI_SetAddress(0xFA);
  QSPI_SetDataLength(1);
  NbrOfDataToTransfer = 2;
  
  ubRxIndex = 0;
  
  pRxData = (uint8_t *)&mid_id;
  RxCplt = 0;
  QSPI_Receive_IT();
  
  while (RxCplt == 0);
  
  return __REV16(mid_id);
}

/**
 * @brief  Erases the entire FLASH.
 * @param  None.
 * @return None.
 */
void SpiFlash_ChipErase(void)
{
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};
  
  QSPI_WriteEnable();
  
  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_NoData;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_NoAddress;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_ChipErase;  // Chip Erase (C7h)
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);

  QSPI_SetDataLength(0);
  NbrOfDataToTransfer = 0;
  
  while(QSPI_GetFlagStatus(QSPI_FLAG_TC) != SET);
  QSPI_ClearFlag(QSPI_FLAG_TC);
  
  SpiFlash_WaitReady();
}

/**
 * @brief  Erases the specified FLASH sector.
 * @param  address: address of the sector to erase.
 * @return None.
 */
void SpiFlash_SectorErase(uint32_t address)
{
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};
  
  QSPI_WriteEnable();
  
  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_24bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_NoData;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_SectorErase;  // Sector Erase (20h)
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  QSPI_SetDataLength(0);
  NbrOfDataToTransfer = 0;
  QSPI_SetAddress(address);
  
  while(QSPI_GetFlagStatus(QSPI_FLAG_TC) != SET);
  QSPI_ClearFlag(QSPI_FLAG_TC);
  
  SpiFlash_WaitReady();
}

/**
 * @brief  Reads a block of data from the FLASH through Quad I/O.
 * @param  address: FLASH's internal address to read from.
 * @param  buffer: pointer to the buffer that receives the data read from the FLASH.
 * @param  length: number of bytes to read from the FLASH.
 * @return None.
 */
void SpiFlash_QuadFastRead(uint32_t address, uint8_t* buffer, uint32_t length)
{
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};
    
  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_32bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_4Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_4Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_FastReadQuad;  // Quad I/O Fast Read (EBH)
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 4;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  pRxData = buffer;
  QSPI_SetDataLength(length - 1);
  NbrOfDataToTransfer = length;
  
  QSPI_SetAddress((address << 8) | 0xFA);
  
  RxCplt = 0;
  ubRxIndex = 0;
  
  pRxData = buffer;
  RxCplt = 0;
  QSPI_Receive_IT();
  
  while (RxCplt == 0);
}

/**
 * @brief  Writes data to one page (256B) through Quad I/O.
 * @note   The number of byte can't exceed the FLASH page remain size.
 * @param  address: FLASH's internal address to write to.
 * @param  buffer: pointer to the buffer containing the data to be written
 *         to the FLASH.
 * @param  length: number of bytes to write to the FLASH.
 * @return None.
 */
void SpiFlash_QuadPageProgram(uint32_t address, const uint8_t* buffer, uint16_t length)
{
  QSPI_ComConfig_InitTypeDef  QSPI_ComConfig_InitStructure = {0};
  
  QSPI_WriteEnable();
  
  /* Command Config for Write Enable */
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DHHC        = QSPI_ComConfig_DHHC_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DDRMode     = QSPI_ComConfig_DDRMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_FMode       = QSPI_ComConfig_FMode_Indirect_Write;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_SIOOMode    = QSPI_ComConfig_SIOOMode_Disable;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABSize      = QSPI_ComConfig_ABSize_8bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADSize      = QSPI_ComConfig_ADSize_24bit;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DMode       = QSPI_ComConfig_DMode_4Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ADMode      = QSPI_ComConfig_ADMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_ABMode      = QSPI_ComConfig_ABMode_NoAlternateByte;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_IMode       = QSPI_ComConfig_IMode_1Line;
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_Ins         = INS_QuadInputPageProgram; // Quad Input Page Program (32h)
  QSPI_ComConfig_InitStructure.QSPI_ComConfig_DummyCycles = 0;
  while(QSPI_GetFlagStatus(QSPI_FLAG_BUSY) != RESET);
  QSPI_ComConfig_Init(&QSPI_ComConfig_InitStructure);
  
  QSPI_SetDataLength(length - 1);
  NbrOfDataToTransfer = length;
  QSPI_SetAddress(address);
  
  pTxData = (uint8_t*)buffer;
  
  TxCplt = 0;
  ubTxIndex = 0;

  QSPI_Transmit_IT();
  
  while(TxCplt == 0);
  
  SpiFlash_WaitReady();
}

/**
 * @brief  Writes block of data to the FLASH through Quad I/O.
 * @param  address: FLASH's internal address to write to.
 * @param  buffer: pointer to the buffer containing the data to be written
 *         to the FLASH.
 * @param  length: number of bytes to write to the FLASH.
 * @return None.
 */
void SpiFlash_QuadWriteBuffer(uint32_t address, const uint8_t* buffer, uint32_t length)
{
  uint32_t write_len;
  write_len = 256 - (address % 256);    /* calculate the start page remain size */
  if(length <= write_len) write_len = length;
  
  while(1)
  {
    SpiFlash_QuadPageProgram(address, buffer, write_len);
    if(length == write_len) 
      break;    /* Write complete */
    else 
    {
      buffer += write_len;
      address += write_len;
      length -= write_len;
      
      if(length > 256)
        write_len = 256;
      else 
        write_len = length;
    }
  }
}

/**
 * @brief  This function handles QSPI interrupt request.
 * @param  None.
 * @return None.
 */
void QUADSPI_IRQHandler(void)
{  
  /* QSPI Fifo Threshold interrupt occurred ----------------------------------*/
  if (QSPI_GetITStatus(QSPI_IT_FT) != RESET)
  {
    if(QSPI_GetFMode() == QSPI_ComConfig_FMode_Indirect_Write)
    {
      /* Transmission process */
      while (QSPI_GetFlagStatus(QSPI_FLAG_FT))
      {
        if(ubTxIndex < NbrOfDataToTransfer)
        {
          /* Fill the FIFO until it is full */
          QSPI_SendData8(*pTxData++);
          ubTxIndex++;
        }
        else
        {
          if (QSPI_WaitFlagStateUntilTimeout(QSPI_FLAG_BUSY, RESET, QSPI_TIMEOUT_DEFAULT) == SET)
          {
            /* Disable the QSPI FIFO Threshold Interrupt */
            QSPI_ITConfig(QSPI_IT_FT, DISABLE);
            
            /* No more data available for the transfer */
            QSPI_AbortRequest();
          }
          break;
        }
      }
    }
    else if(QSPI_GetFMode() == QSPI_ComConfig_FMode_Indirect_Read)
    {
      /* Receiving Process */
      while(QSPI_GetITStatus(QSPI_IT_FT))
      {
        if(ubRxIndex < NbrOfDataToTransfer)
        {
          /* Read the FIFO until it is empty */
          *pRxData++ = QSPI_ReceiveData8();
          ubRxIndex++;
        }
        else
        {
          /* Disable the QSPI FIFO Threshold Interrupt */
          QSPI_ITConfig(QSPI_IT_FT, DISABLE);
          
          /* All data have been received for the transfer */
          QSPI_AbortRequest();
          break;
        }
      }
    }   
  } 
  /* QSPI Transfer Complete interrupt occurred -------------------------------*/
  if(QSPI_GetITStatus(QSPI_IT_TC) != RESET)
  {
    /* Clear interrupt */
    QSPI_ClearITPendingBit(QSPI_IT_TC);
    
    /* Disable the QSPI FIFO Threshold, Transfer Error and Transfer complete Interrupts */
    QSPI_ITConfig(QSPI_IT_TC | QSPI_IT_TE | QSPI_IT_FT, DISABLE);
    
    /* Transfer complete callback */
    if(QSPI_GetFMode() == QSPI_ComConfig_FMode_Indirect_Write)
    {
      /* Clear Busy bit */
      QSPI_AbortRequest();
            
      /* TX Complete callback */
      TxCplt++;
      CmdCplt++;
    }
    else if(QSPI_GetFMode() == QSPI_ComConfig_FMode_Indirect_Read)
    {
      while(QSPI_GetFIFOLevel() != 0)
      {
        if(ubRxIndex < NbrOfDataToTransfer)
        {
          /* Read the the last data received in the FIFO until it is empty */
          *pRxData++ = QSPI_ReceiveData8();
          ubRxIndex++;          
        }
        else
        {
          /* All data have been received for the transfer */
          QSPI_AbortRequest();
          break;
        }
      }
      /* Workaround - Extra data written in the FIFO at the end of a read transfer */
      QSPI_AbortRequest();
            
      /* RX Complete callback */
      RxCplt ++;
      CmdCplt++;
    }
    else if(QSPI_GetFIFOLevel() != 0)
    {
      /* Command Complete callback */
      CmdCplt++;
    }
  }  
  /* QSPI Status Match interrupt occurred ------------------------------------*/
  if(QSPI_GetITStatus(QSPI_IT_SM) != RESET)
  {
    /* Clear flag */
    QSPI_ClearITPendingBit(QSPI_IT_SM);
    
    /* Disable the QSPI FIFO Threshold, Transfer Error and Status Match Interrupts */
    QSPI_ITConfig(QSPI_IT_SM | QSPI_IT_FT | QSPI_IT_TE, DISABLE);
    
    /* Status match callback */
    StatusMatch++;
  }  
  /* QSPI Transfer Error interrupt occurred ----------------------------------*/
  if(QSPI_GetITStatus(QSPI_IT_TE) != RESET)
  {
    /* Clear interrupt */
    QSPI_ClearITPendingBit(QSPI_IT_TE);
    
    /* Disable all the QSPI Interrupts */
    QSPI_ITConfig(QSPI_IT_SM | QSPI_IT_TC | QSPI_IT_TE | QSPI_IT_FT, DISABLE);
  }
  
  /* QSPI Timeout interrupt occurred -----------------------------------------*/
  if(QSPI_GetITStatus(QSPI_IT_TO) != RESET)
  {
    /* Clear interrupt */
    QSPI_ClearITPendingBit(QSPI_IT_TO);
  }
}



