/* Includes ------------------------------------------------------------------*/
#include "mg32f157.h"
#include "bsp_usart1.h"
#include <stdio.h>

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
uint32_t ADC1_CH1;
uint32_t ADC2_CH10;
uint32_t ADC2_CH11;

/* Private function prototypes -----------------------------------------------*/
void Adc_Init(void);
void ADC_EXTI_Init(void);
void RCC_Configuration(void);
void GPIO_Configuration(void);
void NVIC_Configuration(void);

/* Private functions ---------------------------------------------------------*/

/**
 * @brief  Main program.
 * @param  None
 * @return None
 */
int main(void) 
{
  RCC_Configuration();
  
  usart1_init(SystemCoreClock ,115200);
  
  GPIO_Configuration();
  NVIC_Configuration();
  ADC_EXTI_Init();
  Adc_Init();  
  
  while (1) 
  {
  }
}

/**
 * @brief  Configures the different system clocks.
 * @param  None
 * @return None
 */
void RCC_Configuration(void)
{
  /* ADCCLK = PCLK2/8 */
  RCC_ADCCLKConfig(RCC_PCLK2_Div8); 

  /* Enable peripheral clocks ------------------------------------------------*/

  /* Enable GPIOs and ADC1 clock */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA | RCC_APB2Periph_GPIOC | 
                         RCC_APB2Periph_ADC1 | RCC_APB2Periph_ADC2 |
                         RCC_APB2Periph_AFIO, ENABLE);
}

/**
 * @brief  Configures the different GPIO ports.
 * @param  None
 * @return None
 */
void GPIO_Configuration(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;

  /* Configure PA.01(ADC Channel_1)as analog input --------------------------------------------------------*/
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_1;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);

  /* Configure PC.00 (ADC Channel_10) and PC.01 (ADC Channel_11) as analog input --------------------------*/
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_0 | GPIO_Pin_1;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOC, &GPIO_InitStructure);

  /* Configure EXTI line11 ---------------------------------------------------*/
  GPIO_InitStructure.GPIO_Pin  = GPIO_Pin_11;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IPU;
  GPIO_Init(GPIOC, &GPIO_InitStructure);

  /* Configure EXTI line15 ---------------------------------------------------*/
  GPIO_InitStructure.GPIO_Pin  = GPIO_Pin_15;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IPU;
  GPIO_Init(GPIOA, &GPIO_InitStructure);
}

/**
 * @brief  Configures the different EXTI lines to trigger ADC samples.
 * @param  None
 * @return None
 */
void ADC_EXTI_Init(void)
{
  EXTI_InitTypeDef EXTI_InitStructure;
  
  /* Select the EXTI Line11 the GPIO pin source */
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOC, GPIO_PinSource11);
  /* EXTI line11 configuration -----------------------------------------------*/
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Event;               
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;
  EXTI_InitStructure.EXTI_Line = EXTI_Line11;
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_Init(&EXTI_InitStructure);
  
  /* Select the EXTI Line15 the GPIO pin source */
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOA, GPIO_PinSource15);
  /* EXTI line15 configuration -----------------------------------------------*/
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Event;               
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;
  EXTI_InitStructure.EXTI_Line = EXTI_Line15;
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_Init(&EXTI_InitStructure);
}

/**
 * @brief  Configures NVIC and Vector Table base location.
 * @param  None
 * @return None
 */
void NVIC_Configuration(void)
{
  NVIC_InitTypeDef NVIC_InitStructure;

  /* Configure and enable ADC interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = ADC1_2_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

/**
 * @brief  Configures the ADC1 and ADC2 with EXTI lines trigger mode.
 * @param  None
 * @return None
 */
void Adc_Init(void) {
  ADC_InitTypeDef ADC_InitStructure; 
  
  ADC_DeInit(ADC1);  
  ADC_DeInit(ADC2);  
  /* ADC1 configuration ------------------------------------------------------*/
  ADC_RegularChannelConfig(ADC1, ADC_Channel_1, 1, ADC_SampleTime_239Cycles5);

  ADC_InitStructure.ADC_Mode = ADC_Mode_Independent;  
  ADC_InitStructure.ADC_ScanConvMode = DISABLE;
  ADC_InitStructure.ADC_ContinuousConvMode = DISABLE;
  ADC_InitStructure.ADC_ExternalTrigConv = ADC_ExternalTrigConv_Ext_IT11_TIM8_TRGO;
  ADC_InitStructure.ADC_DataAlign = ADC_DataAlign_Right;
  ADC_InitStructure.ADC_NbrOfChannel = 1;
  ADC_Init(ADC1, &ADC_InitStructure);
  /* ADC2 configuration ------------------------------------------------------*/
  ADC_InjectedSequencerLengthConfig(ADC2, 2);
  ADC_InjectedChannelConfig(ADC2, ADC_Channel_10, 1, ADC_SampleTime_239Cycles5);
  ADC_InjectedChannelConfig(ADC2, ADC_Channel_11, 2, ADC_SampleTime_239Cycles5);

  ADC_InitStructure.ADC_Mode = ADC_Mode_Independent; 
  ADC_InitStructure.ADC_ScanConvMode = ENABLE;
  ADC_InitStructure.ADC_DataAlign = ADC_DataAlign_Right;
  ADC_Init(ADC2, &ADC_InitStructure);
  /* Enable ADC1 and ADC2 external trigger conversion */
  ADC_ExternalTrigConvCmd(ADC1,ENABLE);
  ADC_ExternalTrigInjectedConvConfig(ADC2, ADC_ExternalTrigInjecConv_Ext_IT15_TIM8_CC4);
  ADC_ExternalTrigInjectedConvCmd(ADC2, ENABLE);
  /* Enable EOC and JEOC interrupt */
  ADC_ITConfig(ADC1, ADC_IT_EOC, ENABLE); 
  ADC_ITConfig(ADC2, ADC_IT_JEOC, ENABLE); 
  /* Enable ADC1 and ADC2 */
  ADC_Cmd(ADC1, ENABLE);
  ADC_Cmd(ADC2, ENABLE);
  /* Enable ADC1 reset calibration register */
  ADC_ResetCalibration(ADC1); 
  /* Check the end of ADC1 reset calibration register */
  while(ADC_GetResetCalibrationStatus(ADC1)); 
  /* Start ADC1 calibration */
  ADC_StartCalibration(ADC1); 
  /* Check the end of ADC1 calibration */
  while(ADC_GetCalibrationStatus(ADC1)); 
  
  /* Enable ADC2 reset calibration register */
  ADC_ResetCalibration(ADC2); 
  /* Check the end of ADC2 reset calibration register */
  while(ADC_GetResetCalibrationStatus(ADC2)); 
  /* Start ADC2 calibration */
  ADC_StartCalibration(ADC2); 
  /* Check the end of ADC2 calibration */
  while(ADC_GetCalibrationStatus(ADC2)); 
}

/**
 * @brief  This function handles ADC1 and ADC2 global interrupts requests.
 * @param  None
 * @return None
 */
void ADC1_2_IRQHandler(void)
{
  if(ADC_GetITStatus(ADC1, ADC_IT_EOC) != RESET)
  {
    /* Clear ADC1 EOC pending interrupt bit */
    ADC_ClearITPendingBit(ADC1, ADC_IT_EOC);
    ADC1_CH1 = ADC1->DR;
    printf("ADC1 Channel_1 :\r\n ");
    printf("PA1: %4d, %.2f v\r\n",  ADC1_CH1, 1.0 * ADC1_CH1 * 3.3 / 4095);
  }
  if(ADC_GetITStatus(ADC2, ADC_IT_JEOC) != RESET)
  {
    /* Clear ADC2 JEOC pending interrupt bit */
    ADC_ClearITPendingBit(ADC2, ADC_IT_JEOC);
    ADC2_CH10 = ADC2->JDR3;
    ADC2_CH11 = ADC2->JDR4;
    printf("ADC2 Channel_10 / Channel_11 :\r\n ");
    printf("PC0: %4d, %.2f v\r\n",  ADC2_CH10, 1.0 * ADC2_CH10 * 3.3 / 4095);
    printf("PC1: %4d, %.2f v\r\n",  ADC2_CH11, 1.0 * ADC2_CH11 * 3.3 / 4095);
  }
}

#ifdef  USE_FULL_ASSERT
/**
 * @brief  Reports the name of the source file and the source line number
 *         where the assert_param error has occurred.
 * @param  file: pointer to the source file name
 * @param  line: assert_param error line source number
 * @return None
 */
void assert_failed(uint8_t* file, uint32_t line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif
