/**
 ******************************************************************************
 * @file    timer.h
 * @author  cuij
 * @version V1.0.0
 * @date    11-02-2023
 * @brief   The  code is that use adapted to Arduino interface
 *
 * @par         Project
 *              MG32x02z
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2023 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************
  Copyright (c) 2023 megawin Technology Co., Ltd. All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __TIMER_H
#define __TIMER_H

/* Includes ------------------------------------------------------------------*/
#include "MG32x02z.h"
#include "system_MG32x02z.h"


#ifdef __cplusplus
 extern "C" {
#endif

/* Exported types ------------------------------------------------------------*/
typedef enum {
  TIM00_E = 0,
  TIM01_E = 1,
  TIM10_E = 2,
  TIM16_E = 3,
  TIM20_E = 4,
  TIM26_E = 5,
  TIM36_E = 6,
  NB_TIMER_MANAGED
} timer_id_e;


typedef enum {
  bits_8 = 0xFF,
  bits_16 = 0xFFFF,
  bits_32 = 0xFFFFFFFF
} timer_prescaler_limit;

typedef enum {
    
  TIMER_PERIOD    = 0x00000000,
  TIMER_PWM       = 0x00000001,
  TIMER_ONEPULSE  = 0x00000002,
  TIMER_OC        = 0x00000004,
  TIMER_RESERVED  = 0x00000080,
  TIMER_OTHER     = 0x80000000
} timer_mode_e;


typedef struct{
  GPIO_Struct * port;
  uint32_t pin;
  int32_t count;
  uint8_t state;
}timer_toggle_pin_config_str;

/// @brief defines the global attributes of the TIMER
typedef struct
{
    TM_HandleTypeDef *timInstance;
    IRQn_Type irqtype;
    void (*irqHandle)(timer_id_e);
    void (*User_irqHandle)(timer_id_e);
    void (*timer_clock_init)(void);
    void (*timer_clock_reset)(void);
    timer_mode_e timer_mode;
    timer_prescaler_limit prescalerLimit;
    timer_toggle_pin_config_str toggle_pin;
    uint8_t configured;
}timer_conf_t;

/* Exported constants --------------------------------------------------------*/
#define MAX_FREQ  65535

/* Exported macro ------------------------------------------------------------*/
/* Exported functions ------------------------------------------------------- */
void TIM_OC_MspInit(TM_HandleTypeDef *htim);
void TIM_OC_MspDeInit(TM_HandleTypeDef *htim);


void timer_cfg_init(void);

void timer_enable_clock(TM_HandleTypeDef *htim);
void timer_disable_clock(TM_HandleTypeDef *htim);

void TimerHandleInitByPeriod(timer_id_e timer_id, uint32_t Microsecond );
void TimerHandleInit(timer_id_e timer_id, uint32_t frequency);
void TimerHandleDeinit(timer_id_e timer_id);
void TimerHandleSetCounter(timer_id_e timer_id,  uint16_t TM_Counter, uint16_t TM_CounterReload);
uint16_t TimerHandleGetCounter(timer_id_e timer_id);

void TimerHandleRest(timer_id_e timer_id);
void TimerHandleStart(timer_id_e timer_id);
void TimerHandleStop(timer_id_e timer_id);

void TimerPinInit(GPIO_Struct *port, uint32_t pin,  uint32_t afs, uint32_t frequency, uint32_t duration);
void TimerPinDeinit(GPIO_Struct *port, uint32_t pin);

void TimerPulseInit(timer_id_e timer_id, uint16_t period, uint16_t pulseWidth, void (*irqHandle)(timer_id_e));
void TimerPulseDeinit(timer_id_e timer_id);

void TIM_Base_MspInit(TM_HandleTypeDef *htim_base);
void TIM_Base_MspDeInit(TM_HandleTypeDef *htim_base);

timer_id_e getInactiveTimer(void);
timer_id_e isPinAssociateToTimer(GPIO_Struct *port, uint32_t pin);

uint32_t getTimerCounter(timer_id_e timer_id);
void setTimerCounter(timer_id_e timer_id, uint32_t value);
int setCCRRegister(timer_id_e timer_id, uint32_t channel, uint32_t value);
uint32_t getCCRRegister(timer_id_e timer_id, uint32_t channel);

void TimerAttachIntHandle( timer_id_e timer_id, void (*irqHandle)(timer_id_e)  );


#ifdef __cplusplus
}
#endif

#endif /* __TIMER_H */

