/**
  ******************************************************************************
 *
 * @file        Sample_PW_BODx.c
 *
 * @brief       PW BODx example code
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2025/06/24
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2019 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 

/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Sample_PW_BODx (void);
void printf_mg( const char *format,...);
void SYS_IRQHandler(void);
void Sample_URT0_Init(void);
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	   SYS IRQHandler
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void SYS_IRQHandler(void)
{
    uint32_t SYS_IRQHandler_ITFlag;
  
    if(EXIC_GetITSourceID(SYS_IRQn) == 0x02)
    {
        SYS_IRQHandler_ITFlag  = PW->INT.W;
        SYS_IRQHandler_ITFlag &= PW->STA.W; 
        
        if((SYS_IRQHandler_ITFlag & PW_INT_BOD0_IE_mask_w) != 0)  // When BOD0 flag happened and BOD0  
        {
            while(PW->STA.MBIT.BOD1_S == 1);                        // Wait VDD >2.4V
            printf_mg("\nBOD0F happened");                          // Show "BOD0F happened"
            PW_ClearFlag(PW_BOD0F);                                 // Clear PW_BOD0F
        }        
        
        if((SYS_IRQHandler_ITFlag & PW_INT_BOD1_IE_mask_w) != 0)  // When event flag and
        {
            while(PW->STA.MBIT.BOD1_S == 1);                        // Wait VDD >2.4V
            printf_mg("\nBOD1F happened");                          // Show "BOD1F happened"
            PW_ClearFlag(PW_BOD1F);                                 // Clear PW_BOD1F
        }
            
#if defined(PW_INT_BOD2_IE_mask_w)
        if((SYS_IRQHandler_ITFlag & PW_INT_BOD2_IE_mask_w) != 0)  // When event flag and
        {
            while(PW->STA.MBIT.BOD1_S == 1);                        // Wait VDD >2.4V
            printf_mg("\nBOD2F happened");                          // Show "BOD2F happened"
            PW_ClearFlag(PW_BOD2F);                                 // Clear PW_BOD2F
        }    
#endif
    }
}


/**
 *******************************************************************************
 * @brief	   Sample URT0 initial
 * @details     1. Set CSC initial.
 *     \n       2. Set GPIO initial.
 *     \n       3. Set RX/TX Baudrate.
 *     \n       4. Set data character.
 *     \n       5. Set URT0 mode.
 *     \n       6. Set Data line.
 *     \n       7. Set Data buffer control.
 *     \n       8. Enable URT0 interrupt.
 *     \n       9. Enable URT0 
 *     \n      10. Receive data into interrupt function.
 *     \n      11. Send data of receive.
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void Sample_URT0_Init(void)
{
    URT_BRG_TypeDef  URT_BRG;
    URT_Data_TypeDef DataDef;
    PIN_InitTypeDef PINX_InitStruct;
 
    
    
    //==Set CSC init
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 11.0592M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->Port B/URT0 = Enable
    //Configure Peripheral On Mode Clock->URT0->Select URT0_PR Source = CK_APB(11.0592)
    
    //==Set GPIO init
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin8/9
    //GPIO port initial is 0xFFFF
    //Pin8 mode is PPO/Pin9 mode is ODO
    //Pin8/9 pull-up resister Enable
    //Pin8/9 function URT0_TX/RX
    
    PINX_InitStruct.PINX_Mode				= PINX_Mode_PushPull_O;         // Setting PB8 URT0_TX
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Disable;
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;
    PINX_InitStruct.PINX_Alternate_Function = 3;
    GPIO_PinMode_Config(PINB(8),&PINX_InitStruct);
    
    
    //=====Set Clock=====//
    //---Set BaudRate---//
    URT_BRG.URT_InternalClockSource = URT_BDClock_PROC;
    URT_BRG.URT_BaudRateMode = URT_BDMode_Separated;
    URT_BRG.URT_PrescalerCounterReload = 0;	                //Set PSR
    URT_BRG.URT_BaudRateCounterReload = 12;	                //Set RLR
    URT_BaudRateGenerator_Config(URT0, &URT_BRG);   		//BR115200 = f(CK_URTx)/(PSR+1)/(RLR+1)/(OS_NUM+1)
    URT_BaudRateGenerator_Cmd(URT0, ENABLE);	            //Enable BaudRateGenerator
    //---TX/RX Clock---//
    URT_TXClockSource_Select(URT0, URT_TXClock_Internal);	//URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(URT0, URT_RXClock_Internal);	//URT_RX use BaudRateGenerator
    URT_TXOverSamplingSampleNumber_Select(URT0, 7);	        //Set TX OS_NUM
    URT_RXOverSamplingSampleNumber_Select(URT0, 7);	        //Set RX OS_NUM
    URT_RXOverSamplingMode_Select(URT0, URT_RXSMP_3TIME);
    URT_TX_Cmd(URT0, ENABLE);	                            //Enable TX
    URT_RX_Cmd(URT0, ENABLE);	                            //Enable RX
    
    //=====Set Mode=====//
    //---Set Data character config---//
    DataDef.URT_TX_DataLength  = URT_DataLength_8;
    DataDef.URT_RX_DataLength  = URT_DataLength_8;
    DataDef.URT_TX_DataOrder   = URT_DataTyped_LSB;
    DataDef.URT_RX_DataOrder   = URT_DataTyped_LSB;
    DataDef.URT_TX_Parity      = URT_Parity_No;
    DataDef.URT_RX_Parity      = URT_Parity_No;
    DataDef.URT_TX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_RX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_TX_DataInverse = DISABLE;
    DataDef.URT_RX_DataInverse = DISABLE;
    URT_DataCharacter_Config(URT0, &DataDef);
    //---Set Mode Select---//
    URT_Mode_Select(URT0, URT_URT_mode);
    //---Set DataLine Select---//
    URT_DataLine_Select(URT0, URT_DataLine_2);
    
    //=====Set Error Control=====//
    // to do...
    
    //=====Set Bus Status Detect Control=====//
    // to do...
    
    //=====Set Data Control=====//
    URT_RXShadowBufferThreshold_Select(URT0, URT_RXTH_1BYTE);
    URT_IdlehandleMode_Select(URT0, URT_IDLEMode_No);
    URT_TXGuardTime_Select(URT0, 0);
    
    //=====Enable URT Interrupt=====//
    URT_IT_Config(URT0, URT_IT_RX, ENABLE);
    URT_ITEA_Cmd(URT0, ENABLE);
    NVIC_EnableIRQ(URT0_IRQn);

    //=====Enable URT=====//
    URT_Cmd(URT0, ENABLE);
    URT_ClearTXData(URT0);
		
	//==See MG32_URT0_IRQ.c when interrupt in
}


/**
 *******************************************************************************
 * @brief       PW_BODx_Example
 * @details     PW module BODx example code
 *  @arg\b      1. Initial
 *  @arg\b      2. Goto SLEEP mode and BOD2 event trigger wackup
 * 	@arg\b      3. Goto SLEEP mode and BOD2 event trigger wackup
 * 	@arg\b      4. Goto SLEEP mode and BOD2 event trigger wackup
 * 	@arg\b      5. End
 * @return
 * @note        This sample code used file "Sample_Retarget.c" function.
 *              The file "Sample_Retarget.c" must be brought in.
 * @par         Example
 * @code
    PW_BODx_Example();
 * @endcode
 *******************************************************************************
 */
void Sample_PW_BODx (void)
{
    /*=== Initial ===*/
    Sample_URT0_Init();
    
    printf_mg("\n\nPW_BODx_Example, the action is as follow");  // Show example message
    
    PW_ClearFlag(PW_ALLF);                          //
    
    UnProtectModuleReg(PWprotect);                  // Unprotect PW module register
#if defined(MG32_3RD) || defined(MG32_4TH) || defined(MG32_5TH)
    PW_BOD2Trigger_Select(PW_BODx_FallingEdge);     // Setting BOD2 trigger edge 
    PW_BOD2_Cmd(ENABLE);                            // Enable BOD2 module
#endif
    
    PW_BOD1Threshold_Select(PW_BOD1_2V4);           // Setting BOD1 detect level is 2.4V
    PW_BOD1Trigger_Select(PW_BODx_FallingEdge);     // Setting BOD1 trigger edge 
    PW_BOD1_Cmd(ENABLE);                            // Enable BOD1 module
    
    PW_BOD0_Cmd(ENABLE);                            // Enalbe BOD0
    PW_ClearFlag(PW_ALLF);                          // Clear BOD0/1/2 and WKF
    
    #if !defined(MG32_SYS_PW_IRQ_)
        NVIC_EnableIRQ(SYS_IRQn);                   // Enable SYS_IRQ
        NVIC_SetPriority(SYS_IRQn,0);               // Setting SYS_IRQ priority
    #endif
    SYS_ITEA_Cmd(ENABLE);                           // Enable SYS ITEA
    PW_ITEA_Cmd(ENABLE);                            // Enable PW ITEA
    
    /*=== Goto SLEEP mode and BOD2 event trigger wackup ===*/
    printf_mg("\n1. VDD to 2.2V ");                 // Action 1st
    printf_mg("\n2. VDD to 3.3V ");                 // Action 2md
    PW_IT_Config(PW_INT_BOD1, ENABLE);              // Enable BOD1 interrupt
    SLEEP_WFI();                                    // MCU goto SLEEP mode

    PW_IT_Config(PW_INT_BOD1, DISABLE);             // Disable BOD1 interrupt
                        
    /*=== Goto SLEEP mode and BOD2 event trigger wackup ===*/
    printf_mg("\n3. VDD to 1.65V ");                //
    printf_mg("\n4. VDD to 3.3V "); 
    PW_IT_Config(PW_INT_BOD2, ENABLE);              // Enable BOD2 interrupt
    SLEEP_WFI();                                    // MCU goto SLEEP mode
    
    PW_IT_Config(PW_INT_BOD2, DISABLE);             // Disable BOD2 interrupt
    
    /*=== Goto SLEEP mode and BOD2 event trigger wackup ===*/
    printf_mg("\n5. VDD to 1.4V ");                //
    printf_mg("\n6. VDD to 3.3V ");                 //
    PW_IT_Config(PW_INT_BOD0, ENABLE);              // Enable BOD0 interrupt
    SLEEP_WFI();                                    // MCU goto SLEEP mode

    PW_IT_Config(PW_INT_BOD0, DISABLE);             // Disable BOD0 interrupt
    
    ProtectModuleReg(PWprotect);                    // Protect PW module
    /*=== End ===*/
    printf_mg("\nEnd");  
} 




