/**
  ******************************************************************************
 *
 * @file        Sample_IWDT_WakeUp_Init.c
 *
 * @brief       IWDT Wake Up sample code
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2024/06/07
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */



/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Sample_IWDT_WakeUp_Init (void);
void IWDT_IRQ (void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief  	    Sample IWDT Wake Up init
 * @details     1. Enable CSC to IWDT clock
 *      \n      2. IWDT module clock divider select
 *      \n      3. Enable IWDT module
 *      \n      4. IWDT Wake Up Enable
 *      \n      5. IWDT Interrupt Enable
 * @return	    None	
 * @note        
 * @par         Example
 * @code
    Sample_IWDT_WakeUp_Init();
 * @endcode
 *******************************************************************************
 */
void Sample_IWDT_WakeUp_Init(void)
{
    
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12Mz
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->IWDT = Enable

//    /*=== 1. Enable CSC to IWDT clock ===*/
//    UnProtectModuleReg(CSCprotect);                                 // Unprotect CSC module
//    CSC_PeriphOnModeClock_Config(CSC_ON_IWDT, ENABLE);              // Enable IWDT module clock
//    CSC_PeriphSTOPModeClock_Config(CSC_SLP_IWDT, ENABLE);           // IWDT SLEEP mode module clock enable.
//    CSC_PeriphSTOPModeClock_Config(CSC_STP_IWDT, ENABLE);           // IWDT STOP mode module clock enable.
//    ProtectModuleReg(CSCprotect);                                   // protect CSC module
    
    UnProtectModuleReg(IWDTprotect);                                // Unprotect IWDT module
    
    /*=== 2. IWDT module clock divider select ===*/
    IWDT_Divider_Select(IWDT_DIV_256);                              // DIV output = CK_IWDT /256
    
    /*=== 3. Enable IWDT module ===*/
    IWDT_Cmd(ENABLE);                                               // Enable IWDT module
    
    /*=== 4. IWDT Wake Up Enable ===*/
    IWDT_StopModeWakeUpEvent_Config(IWDT_EW1_WPEN, ENABLE);
    
    /*=== 5. IWDT Interrupt Enable ===*/
    IWDT_IT_Config(IWDT_INT_EW1, ENABLE);
    #if !defined(MG32_SYS_RTC_IRQ_)
        NVIC_EnableIRQ(SYS_IRQn);
        NVIC_SetPriority(SYS_IRQn,0);
    #endif
    SYS_ITEA_Cmd(ENABLE);
    
    /*=== 6. IWDT Wake Up Set ===*/
    UnProtectModuleReg(PWprotect);                                // Unprotect PW module
    PW_PeriphSTOPModeWakeUp_Config(PW_WKSTP_IWDT, ENABLE);        // IWDT Wake Up Set
    ProtectModuleReg(PWprotect);                                  // protect PW module
    
    IWDT_RefreshCounter();                                          // Clear IWDT timer
    
    ProtectModuleReg(IWDTprotect);                                  // Protect IWDT module
    
    //SLEEP_WFI();                                                  // Chip Working at SLEEP mode.
    STOP_WFI();                                                     // Chip Working at STOP mode.
}

/**
 *******************************************************************************
 * @brief  	    IWDT Interrupt Requast
 * @details     
 *      \n      
 * @return	    None
 * @note        Clear all IWDT event flag.
 * @par         Example
 * @code
    IWDD_IRQ();
 * @endcode    
 *******************************************************************************
 */
void IWDT_IRQ (void)
{
    if(IWDT_GetSingleFlagStatus(IWDT_EW1F) == DRV_Happened)
    {
        // When early wake up 1 event happened.
        // To do...
        IWDT_ClearFlag(IWDT_EW1F);
        IWDT_RefreshCounter();
    }
    
    if(IWDT_GetSingleFlagStatus(IWDT_EW0F) == DRV_Happened)
    {
        // When early wake up 0 event happened.
        // To do...
        IWDT_ClearFlag(IWDT_EW0F);
        IWDT_RefreshCounter();
    }
    
    if(IWDT_GetSingleFlagStatus(IWDT_TF) == DRV_Happened)
    {
        // When timer timeout event happened.
        // To do...
        IWDT_ClearFlag(IWDT_TF);
        IWDT_RefreshCounter();
    }
}


