/**
  ******************************************************************************
 *
 * @file        Sample_DMA.c
 *
 * @brief       This is the C code format sample file for DMA.
                include Memory to Memory and 
                        Memory to Peripheral and
                    Peripheral to Memory and 
                    Peripheral to Peripheral
 *
 * @par         Project
 *              MG32
 * @version     V1.11
 * @date        2021/05/25
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2016 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

//*** <<< Use Configuration Wizard in Context Menu >>> ***

//*** <<< end of configuration section >>>    ***



/* Includes ------------------------------------------------------------------*/
#include "MG32_DMA_DRV.h"
#include "MG32_CSC_DRV.h"
#include "Sample__Common.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
SMP_Return Sample_DMA_Init(void);
SMP_Return Sample_DMA_M2M_IT(DMAChannel_Struct* DMACHx,                         \
                             uint32_t SourceAddress,                            \
                             uint32_t DestinationAddress,                       \
                             uint32_t Length);                                  
SMP_Return Sample_DMA_M2P_IT(DMAChannel_Struct* DMACHx,                         \
                             uint32_t SourceAddress,                            \
                             DMA_DestinationRequestDef DestinationPeripheral,   \
                             uint32_t Length);
SMP_Return Sample_DMA_P2M_IT(DMAChannel_Struct* DMACHx,                         \
                             DMA_SourcenRequestDef SourcePeripheral,            \
                             uint32_t DestinationAddress,                       \
                             uint32_t Length);
SMP_Return Sample_DMA_P2P_IT(DMAChannel_Struct* DMACHx,                         \
                             DMA_SourcenRequestDef SourcePeripheral,            \
                             DMA_DestinationRequestDef DestinationPeripheral,   \
                             uint32_t Length);
SMP_Return Sample_DMAChannel_IT_Handle(DMAChannel_Struct* DMACHx);
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/




/**
 *******************************************************************************
 * @brief
 * @details     1.Disable DMA
 *    \n        2.Setting Round-robin / Level
 *    \n        3.Enable DMA
 *
 *******************************************************************************
 */
SMP_Return Sample_DMA_Init(void)
{
    // Enable DMA Clock
    CSC->KEY.H[0] = 0xA217;
    CSC_PeriphOnModeClock_Config(CSC_ON_DMA, ENABLE);
    CSC->KEY.H[0] = 0;

    // Disable DMA
    DMA->CR0.W &= ~DMA_CR0_EN_mask_w;
    DMA->INT.W = 0;
    DMA->CR2.W = 0;

  #if !defined(MG32_2ND)
    // Round-Robin / Level
    // DMA->CR0.W |= DMA_CR0_PRI_MDS_round_w;
    DMA->CR0.W |= DMA_CR0_PRI_MDS_level_w;
  #endif

    // Enable DMA Module Interrupt
    DMA_ITEA_Cmd(DMA, ENABLE);
    NVIC_SetPriority(DMA_IRQn, 1);
    NVIC_EnableIRQ(DMA_IRQn);

    // Enable
    DMA->CR0.W |= DMA_CR0_EN_enable_w;

    return SMP_Success;
}



/**
 *******************************************************************************
 * @brief       Copy Source Memory to Destination Memory by DMA macro
 * @details     1.Enable DMA
 *    \n        2.Enable Channel0
 *    \n        3.Setting M2M simple parameter (only support DMA Channel 0)
 *    \n        4.clear flag
 *    \n        5.start request
 *    \n        6.until complete & clear flag
 *******************************************************************************
 */
SMP_Return Sample_DMA_M2M_IT(DMAChannel_Struct* DMACHx, uint32_t SourceAddress, uint32_t DestinationAddress, uint32_t Length)
{
    // DMA Enable Check
    if((DMA->CR0.W & DMA_CR0_EN_mask_w) == 0)
        return SMP_Failure;

    // DMA Channel Disable and Requst Disable    
    DMACHx->A.B[0] &= (~(DMA_CH0A_CH0_EN_mask_w | \
                         DMA_CH0A_CH0_REQ_mask_w | \
                         DMA_CH0A_CH0_CIE_enable_w | \
                         DMA_CH0A_CH0_HIE_disable_w | \
                         DMA_CH0A_CH0_EIE_enable_w));

    // Clear DMA Channel Flag
    DMACHx->A.B[3] = (DMA_CH0A_CH0_TC2F_mask_b3 | \
                      DMA_CH0A_CH0_TH2F_mask_b3 | \
                      DMA_CH0A_CH0_ERR2F_mask_b3);

    // DMA 
    // DMA transfer data loop
    // DMA Burst size config
    // Set Interrupt Config
  #if !defined(MG32_2ND)
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_PLS_lv2_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #else
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #endif

    // DMA source from SRAM / destination to SRAM Config
    // channel x source/destination auto increase address
  #if !defined(MG32_2ND)
    DMACHx->B.W = DMA_CH0B_CH0_SINC_enable_w | \
                  DMA_CH0B_CH0_DINC_enable_w | \
                  DMA_CH0B_CH0_SSYNC_disable_w | \
                  DMA_CH0B_CH0_DSYNC_disable_w | \
                  DMA_CH0B_CH0_XPIN_trg0_w;
  #else
    DMACHx->B.W = DMA_CH0B_CH0_SINC_enable_w | \
                  DMA_CH0B_CH0_DINC_enable_w | \
                  DMA_CH0B_CH0_SSYNC_disable_w | \
                  DMA_CH0B_CH0_DSYNC_disable_w;
  #endif

    // Data Source / Destination Address config.
    DMACHx->SSA.W = SourceAddress;
    DMACHx->DSA.W = DestinationAddress;

    // DMA transfer data count initial number
    DMACHx->NUM.H[0] = (uint16_t)Length;

    // DMA Channel Enable
    DMACHx->A.B[0] |= DMA_CH0A_CH0_EN_enable_b0;

    return SMP_Success;
}



/**
 *******************************************************************************
 * @brief       Copy Source Memory to Destination Peripheral by DMA macro
 * @details     1.Enable DMA
 *    \n        2.Enable Channel0
 *    \n        3.Setting M2P simple parameter (only support DMA Channel 0)
 *    \n        4.clear flag
 *    \n        5.start request
 *    \n        6.until complete & clear flag
 *******************************************************************************
 */
SMP_Return Sample_DMA_M2P_IT(DMAChannel_Struct* DMACHx, \
                             uint32_t SourceAddress, \
                             DMA_DestinationRequestDef DestinationPeripheral, \
                             uint32_t Length)
{
    // DMA Enable Check
    if((DMA->CR0.W & DMA_CR0_EN_mask_w) == 0)
        return SMP_Failure;

    // DMA Channel Disable and Requst Disable    
    DMACHx->A.B[0] &= (~(DMA_CH0A_CH0_EN_mask_w | \
                         DMA_CH0A_CH0_REQ_mask_w | \
                         DMA_CH0A_CH0_CIE_enable_w | \
                         DMA_CH0A_CH0_HIE_disable_w | \
                         DMA_CH0A_CH0_EIE_enable_w));

    // Clear DMA Channel Flag
    DMACHx->A.B[3] = (DMA_CH0A_CH0_TC2F_mask_b3 | \
                      DMA_CH0A_CH0_TH2F_mask_b3 | \
                      DMA_CH0A_CH0_ERR2F_mask_b3);

    // DMA 
    // DMA transfer data loop
    // DMA Burst size config
    // Set Interrupt Config
  #if !defined(MG32_2ND)
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_PLS_lv1_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #else
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #endif

    // DMA source from SRAM / destination to SRAM Config
    // channel x source/destination auto increase address
  #if !defined(MG32_2ND)
    DMACHx->B.W = (uint32_t) (DestinationPeripheral << 8)       | \
                  (uint32_t) (DMA_CH0B_CH0_SINC_enable_w)       | \
                  (uint32_t) (DMA_CH0B_CH0_DINC_disable_w)      | \
                  (uint32_t) (DMA_CH0B_CH0_SSYNC_disable_w)     | \
                  (uint32_t) (DMA_CH0B_CH0_DSYNC_disable_w)     | \
                  (uint32_t) (DMA_CH0B_CH0_XPIN_trg0_w);
  #else
    DMACHx->B.W = (uint32_t) (DestinationPeripheral << 8)       | \
                  (uint32_t) (DMA_CH0B_CH0_SINC_enable_w)       | \
                  (uint32_t) (DMA_CH0B_CH0_DINC_disable_w)      | \
                  (uint32_t) (DMA_CH0B_CH0_SSYNC_disable_w)     | \
                  (uint32_t) (DMA_CH0B_CH0_DSYNC_disable_w);
  #endif

    // Data Source Address config.
    DMACHx->SSA.W = SourceAddress;

    // DMA transfer data count initial number
    DMACHx->NUM.H[0] = (uint16_t)Length;

    // DMA Channel Enable
    DMACHx->A.B[0] |= DMA_CH0A_CH0_EN_enable_b0;

    return SMP_Success;
}



/**
 *******************************************************************************
 * @brief       Copy Source Peripheral to Destination Memory by DMA macro
 * @details     1.DMA Channel Disable
 *    \n        2.DMA Channel Requset Disable
 *    \n        3.DMA Setting P2M simple parameter (only support DMA Channel 0)
 *    \n        4.clear flag
 *    \n        5.start request
 *    \n        6.until complete & clear flag
 *******************************************************************************
 */
SMP_Return Sample_DMA_P2M_IT(DMAChannel_Struct* DMACHx, DMA_SourcenRequestDef SourcePeripheral, uint32_t DestinationAddress, uint32_t Length)
{
    // DMA Enable Check
    if((DMA->CR0.W & DMA_CR0_EN_mask_w) == 0)
        return SMP_Failure;

    // DMA Channel Disable and Requst Disable    
    DMACHx->A.B[0] &= (~(DMA_CH0A_CH0_EN_mask_w | \
                         DMA_CH0A_CH0_REQ_mask_w | \
                         DMA_CH0A_CH0_CIE_enable_w | \
                         DMA_CH0A_CH0_HIE_disable_w | \
                         DMA_CH0A_CH0_EIE_enable_w));

    // Clear DMA Channel Flag
    DMACHx->A.B[3] = (DMA_CH0A_CH0_TC2F_mask_b3 | \
                      DMA_CH0A_CH0_TH2F_mask_b3 | \
                      DMA_CH0A_CH0_ERR2F_mask_b3);

    // DMA 
    // DMA transfer data loop
    // DMA Burst size config
    // Set Interrupt Config

  #if !defined(MG32_2ND)
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_PLS_lv0_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #else
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #endif

    // DMA source from SRAM / destination to SRAM Config
    // channel x source/destination auto increase address
  #if !defined(MG32_2ND)
    DMACHx->B.W = SourcePeripheral | \
                  DMA_CH0B_CH0_SINC_disable_w | \
                  DMA_CH0B_CH0_DINC_enable_w | \
                  DMA_CH0B_CH0_SSYNC_disable_w | \
                  DMA_CH0B_CH0_DSYNC_disable_w | \
                  DMA_CH0B_CH0_XPIN_trg0_w;
  #else
    DMACHx->B.W = SourcePeripheral | \
                  DMA_CH0B_CH0_SINC_disable_w | \
                  DMA_CH0B_CH0_DINC_enable_w | \
                  DMA_CH0B_CH0_SSYNC_disable_w | \
                  DMA_CH0B_CH0_DSYNC_disable_w;
  #endif

    // Data Destination Address config.
    DMACHx->DSA.W = DestinationAddress;

    // DMA transfer data count initial number
    DMACHx->NUM.H[0] = (uint16_t)Length;

    // DMA Channel Enable
    DMACHx->A.B[0] |= DMA_CH0A_CH0_EN_enable_b0;

    return SMP_Success;
}



/**
 *******************************************************************************
 * @brief       Copy Source Peripheral to Destination Peripheral by DMA macro
 * @details     1.DMA Channel Disable
 *    \n        2.DMA Channel Requset Disable
 *    \n        3.DMA Setting P2P simple parameter (only support DMA Channel 0)
 *    \n        4.clear flag
 *    \n        5.start request
 *    \n        6.until complete & clear flag
 *******************************************************************************
 */
SMP_Return Sample_DMA_P2P_IT(DMAChannel_Struct* DMACHx, DMA_SourcenRequestDef SourcePeripheral, DMA_DestinationRequestDef DestinationPeripheral, uint32_t Length)
{
    // DMA Enable Check
    if((DMA->CR0.W & DMA_CR0_EN_mask_w) == 0)
        return SMP_Failure;

    // DMA Channel Disable and Requst Disable    
    DMACHx->A.B[0] &= (~(DMA_CH0A_CH0_EN_mask_w | \
                         DMA_CH0A_CH0_REQ_mask_w | \
                         DMA_CH0A_CH0_CIE_enable_w | \
                         DMA_CH0A_CH0_HIE_disable_w | \
                         DMA_CH0A_CH0_EIE_enable_w));

    // Clear DMA Channel Flag
    DMACHx->A.B[3] = (DMA_CH0A_CH0_TC2F_mask_b3 | \
                      DMA_CH0A_CH0_TH2F_mask_b3 | \
                      DMA_CH0A_CH0_ERR2F_mask_b3);

    // DMA 
    // DMA transfer data loop
    // DMA Burst size config
    // Set Interrupt Config
  #if !defined(MG32_2ND)
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_PLS_lv0_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #else
    DMACHx->A.W = DMA_CH0A_CH0_HOLD_disable_w | \
                  DMA_CH0A_CH0_LOOP_disable_w | \
                  DMA_CH0A_CH0_ADSEL_normal_w | \
                  DMA_CH0A_CH0_XMDS_disable_w | \
                  DMA_CH0A_CH0_BSIZE_one_w | \
                  DMA_CH0A_CH0_CIE_enable_w | \
                  DMA_CH0A_CH0_HIE_disable_w | \
                  DMA_CH0A_CH0_EIE_enable_w;
  #endif

    // DMA source from SRAM / destination to SRAM Config
    // channel x source/destination auto increase address
  #if !defined(MG32_2ND)
    DMACHx->B.W = (uint32_t) (SourcePeripheral)             | \
                  (uint32_t) (DestinationPeripheral)        | \
                  (uint32_t) (DMA_CH0B_CH0_SINC_disable_w)  | \
                  (uint32_t) (DMA_CH0B_CH0_DINC_enable_w)   | \
                  (uint32_t) (DMA_CH0B_CH0_SSYNC_disable_w) | \
                  (uint32_t) (DMA_CH0B_CH0_DSYNC_disable_w) | \
                  (uint32_t) (DMA_CH0B_CH0_XPIN_trg0_w);
  #else
    DMACHx->B.W = (uint32_t) (SourcePeripheral)             | \
                  (uint32_t) (DestinationPeripheral)        | \
                  (uint32_t) (DMA_CH0B_CH0_SINC_disable_w)  | \
                  (uint32_t) (DMA_CH0B_CH0_DINC_enable_w)   | \
                  (uint32_t) (DMA_CH0B_CH0_SSYNC_disable_w) | \
                  (uint32_t) (DMA_CH0B_CH0_DSYNC_disable_w);
  #endif

    // DMA transfer data count initial number
    DMACHx->NUM.H[0] = (uint16_t)Length;

    // DMA Channel Enable
    DMACHx->A.B[0] |= DMA_CH0A_CH0_EN_enable_b0;

    return SMP_Success;
}



/**
 *******************************************************************************
 * @brief       DMA Channel Handle
 * @details     1.DMA Channel Disable
 *    \n        2.DMA Channel Requset Disable
 *    \n        3.DMA Setting P2P simple parameter (only support DMA Channel 0)
 *    \n        4.clear flag
 *    \n        5.start request
 *    \n        6.until complete & clear flag
 * @param[in]   DMACHx : DMA Channel
 * @return      SMP_Return
 * @exception   None
 * @note        
 * @par         Example
 * @code        
                Sample_DMAChannel_IT_Handle(DMAChannel0);
                Sample_DMAChannel_IT_Handle(DMAChannel1);
                Sample_DMAChannel_IT_Handle(DMAChannel2);
 * @endcode     
 * @par         Modify
 *              SMP_Return Sample_DMAChannel_IT_Handle(DMAChannel_Struct* DMACHx)
 *******************************************************************************
 */
__weak SMP_Return Sample_DMAChannel_IT_Handle(DMAChannel_Struct* DMACHx)
{
    if((DMACHx->A.W & DMA_CH0A_CH0_ERR2F_mask_w) != 0)
        __NOP();

    if((DMACHx->A.W & DMA_CH0A_CH0_TH2F_mask_w) != 0)
        __NOP();

    if((DMACHx->A.W & DMA_CH0A_CH0_TC2F_mask_w) != 0)
        __NOP();

    return SMP_Success;
}


