/**
 ******************************************************************************
 *
 * @file        BSP_12_BLE.c
 * @brief       MG32 BLE board support c code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2022/06/29
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/*==============================================================================
                                 User NOTES
How To use this function:
-----------------------
   1. BSP_BLE_Init();                                    // BLE initial
   
   2. BSP_BLE_SetInitRunningMode();                      // Set BLE init running mode
   3. BSP_BLE_RadioInit(TXPWR_0DBM, &ble_mac_addr);      // Radio init BLE
   4. BSP_BLE_RunIniterruptStart(160*2);                 // 320*0.625=200 ms
   
   4. IrqMcuGotoSleepAndWakeup();                        // IRQ MCU goto sleep and wakeup
   5. BSP_BLE_GetConnectStatus();                        // Get BLE connect status.
   
Driver architecture:
--------------------  
   + void BSP_BLE_WriteTxBuffer (u8* SrcIndex, u8 len); // Update BLE_TX_BUF
   + void BSP_BLE_ClearTxBuffer (void);                 // Clear BLE_TX_BUFF index and flag
   + u8 BSP_BLE_ReadRxBuffer (u8* DestIndex);           // Read BLE_RX_BUFF
   + void BSP_BLE_ClearRxBuffer (void);                 // Clear BLE_RX_BUFF

Known Limitations:
------------------
   1- Run in 24MHz. (CSC control).

Require parameter:
------------------   
    Require module : CSC / SYS / RST / GPIO / SPI / EXIC / IWDT / TM01
                       
    CSC Module :
        Clock source                : CK_IHRCO (12M Hz)
        CK_MAIN frequency           : CK_PLLO
        CK_APB frequency            : 24MHz
        CK_AHB frequency            : 24MHz
        Clock enable in 'On Mode'   : GPIOB / GPIOC / GPIOE
        
    GPIO pin configuration : 
        Pin / IO mode  / AFS        / BLE Pin
        ---- ---------- ------------ --------------------
        PE0 / DIN      / GPIO       / IRQ
        PE1 / DIN      / SPI0_MISO  / MI
        PE2 / PPO      / SPI0_CLK   / SCK
        PE3 / PPO      / SPI0_MOSI  / MO
        PE8 / PPO      / GPIO       / CEN
        
    SPI Module :
        Mode                        : Standard SPI
        Clock Mode                  : (CPOL = 0 , CPHA = 0)
        Baud Rate                   : 6MHz ~ 10MHz

    EXIC Module : 
        Mode                        : Falling edge

    IWDT Module :
        Mode                        : Enable IWDT event trigger warm reset
        Clock frequency             : 32KHz
        Overflow time               : 4ms
    
    TM01 Module :
        Mode                        : Cascade mode
        Overflow time               : 1ms ( in TM00 clock source is 48Mhz)           
           
Example codes:
------------------   

// Includes ------------------------------------------------------------------// 
#include "BSP_BLE.h"
#include "BSP_BLE_app.h"

// Wizard menu ---------------------------------------------------------------//
// Private typedef -----------------------------------------------------------//
// Private define ------------------------------------------------------------//
// Private macro -------------------------------------------------------------//
// Private variables ---------------------------------------------------------//
// Private function prototypes -----------------------------------------------//
// Exported variables --------------------------------------------------------//
unsigned char *ble_mac_addr;     // Add volatile //

// Exported functions --------------------------------------------------------//
void ChipInit(void);
void BSP_BLE_WriteTxBuffer (u8* Index, u8 len);
void BSP_BLE_ClearBleTxBuffer (void);
u8 BSP_BLE_ReadRxBuffer (u8* DestIndex);
void BSP_BLE_ClearBleRxBuffer (void);

// External variables --------------------------------------------------------//


void BLE_Sample (void)
{
    u8 Text[30] = {0};                                      // String array
    u8 TX_CNT;                                              // String length
   
    
    BSP_BLE_Init();                                         // Initial SPI0, URT0, EXIC, IWDT, LED, GPIO and Tick
    BSP_BLE_SetInitRunningMode();                           // Set BLE init running mode
    BSP_BLE_RadioInit(TXPWR_0DBM, &ble_mac_addr);           // Set Radio init BLE
    BSP_BLE_RunIniterruptStart(160*2);                      // Set interrupt time 320*0.625 = 200 ms

    while(1)
    {
        BSP_BLE_IrqMcuGotoSleepAndWakeup();                 // IRQ MCU go to sleep and wakeup

        if(BLE_RXF != 0)                                    // Get BLE recevie flag
        {
            Text[0] = 'G';                                  // Set text buffer "Get "
            Text[1] = 'e';
            Text[2] = 't';
            Text[3] = ' ';
            TX_CNT = 4;                                     // The text buffer count 4

            TX_CNT += BSP_BLE_ReadRxBuffer(&Text[4]);       // Get BLE receive date to "Text" and update TX_CNT.  
            BSP_BLE_ClearBleRxBuffer();                     // Clear RXF and 
            BSP_BLE_WriteTxBuffer (Text, TX_CNT);           // Update BLE TX buffer data
        }
    }
}      
==============================================================================*/


/* Includes ------------------------------------------------------------------*/
#include "__BSP_Wizard.h"
#include "MG32.h"

#if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_MG126

#include "BSP_12_BLE_MG126.h"
#include "BSP_12_BLE_MG126_app.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO 
#define BLE_CSN_PINX           PINE(8)
#define BLE_MOSI_PINX          PINE(3)
#define BLE_MISO_PINX          PINE(1)
#define BLE_SCK_PINX           PINE(2)
#define BLE_IRQ_PINX           PINE(0)
                               
#define BLE_CSN_PIN_AFS        0
#define BLE_MOSI_PIN_AFS       5
#define BLE_MISO_PIN_AFS       5
#define BLE_SCK_PIN_AFS        5
#define BLE_IRQ_PIN_AFS        0

#define BLE_IRQ                PE0
#define SPI_CSE                PE8   

//EXIC 
#define BLE_CSN_EXIC_PORT      EXIC_PE
#define BLE_CSN_EXIC_TRGSPIN   EXIC_TRGS_PIN0
#define BLE_CSN_EXIC_OR_MASK   EXIC_PX_PIN0
#define BLE_CSN_EXIC_FLAG      EXIC_PX_PIN0
#define BLE_CSN_EXIC_IT        EXIC_PE_IT
#define BLE_CSN_EXIC_IRQn      EXINT3_IRQn       

//SPI 
#define BLE_SPI_CLOCKFREQ      48000000                  // SPI clock source frequency (Hz)
                                                         
#define BLE_MODULE             SPI0                      // BLE_MODULE used SPI0
                                                         
//TM                                                     
#define BLE_TM                 TM01                      // BLE Timebase Timer.
                                                         
//BLE control                                            
#define MAX_SIZE               200                       // rxBuf/txBuf data size

//Buffer control
#define BLE_TxBufIndex  BLE_TX_INDEX                     // BLE transmit buffer index


/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static u8 BLE_RX_BUF[MAX_SIZE];                          // BLE RX buffer. // u8 txBuf[MAX_SIZE];    // BLE RX buffer
static u8 BLE_TX_BUF[MAX_SIZE];                          // BLE TX buffer. // u8 rxBuf[MAX_SIZE];    // BLE TX buffer
static u8 BLE_RX_STR = 0;                                // BLE RX start
static u8 BLE_RX_LEN = 0;                                // BLE RX length
byte BLE_RXF  = 0;                                       // BLE RX update compete flag.
byte BLE_TCF  = 1;                                       // BLE TX transmited complet flag.

static unsigned char Temp8;
static unsigned short Temp16;
static unsigned long Temp24;
static unsigned long Temp32;

static volatile boolean CPU_CLK_FAST_ENABLE = FALSE;     // Add volatile
static u16 BLE_TX_INDEX = 0;                             // BLE TX index.
static volatile unsigned int BLETimebase_Count = 0;

static volatile u16 BLE_TxdBufIndex = 0;                 // BLE transmitted buffer index

/* Private function prototypes -----------------------------------------------*/
void EXINT3_IRQHandler(void);
void TM0x_IRQHandler(void);
static void BSP_BLE_TimeBaseIRQHandler(void);

static void BLE_EXIC_Init(void);
static void IWDT_Init(void);
static void BLE_SPI_Init(void);
static void BSP_TM_Init(void);
static DRV_Return SPIx_Init (uint32_t ProcessClockRate);
static void BSP_BLE_SendData (void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief     EXINT3 handler BLE IRQ pin interrupt
 * @details
 * @return      
 * @note      
 *******************************************************************************
 */
void EXINT3_IRQHandler(void)
{
    BLE_IRQHandler();
}
/**
 *******************************************************************************
 * @brief     BLE IRQ pin interrupt handle.
 * @details
 * @return      
 * @note      ble has used, modify code must be careful
 *******************************************************************************
 */
__WEAK void BLE_IRQHandler(void)
{    
    EXIC_ClearPxTriggerEventFlag(BLE_CSN_EXIC_PORT, BLE_CSN_EXIC_FLAG );        // Clear 
    IWDT_RefreshCounter();
    ble_run(0);
}
/**
 *******************************************************************************
 * @brief     BLE timebase interrupt handle.
 * @details
 * @return      
 * @note        
 *******************************************************************************
 */
void TM0x_IRQHandler(void)
{
    BSP_BLE_TimeBaseIRQHandler();
}
/**
 *******************************************************************************
 * @brief     BSP initial
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_Init(void)
{
    /*Parameter Initial*/
    BLETimebase_Count = 0;
    
    /*Module Initial*/
    BLE_SPI_Init();
    BLE_EXIC_Init();
    IWDT_Init();
    BSP_TM_Init();
}
/**
 *******************************************************************************
 * @brief     Get BLE connect status
 * @details
 * @return    BLE connect status
 *  \n        0: Disconnect
 *  \n        1: Connect
 * @note
 *******************************************************************************
 */
uint8_t BSP_BLE_GetConnectStatus(void)
{
    return(GetConnectedStatus());
}
/**
 *******************************************************************************
 * @brief     Write BLE_TX_BUF buffer
 * @details
 * @param[in] SrcIndex: Source address.
 * @param[in] len: Transfer length bytes number. 1 ~ 200 bytes.
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_WriteTxBuffer (u8* SrcIndex, u8 len)
{
    u8  i;                                  // 

    for(i=len; i>0; i--)                        // for loop 
    {
        BLE_TX_BUF[BLE_TX_INDEX] = *SrcIndex;   // Move data to BLE_TX_BUF.
        SrcIndex++;                             // Update source index address.
        BLE_TX_INDEX++;                         // Update BLE TX BUF index.
        if(BLE_TX_INDEX >= MAX_SIZE)            // When BLE TX BUF index >= MAX_SIZE (200).
        {
            BLE_TX_INDEX = 0;                   // BLE TX BUF index = 0.
        }
    }
}
/**
 *******************************************************************************
 * @brief     Move BLE_RX_BUF data to DestIndex 
 * @details
 * @param[in] DestIndex: Destination address.
 * @return    Received data byte number. 1 ~ 200 bytes.
 * @note
 *******************************************************************************
 */
u8 BSP_BLE_ReadRxBuffer (u8* DestIndex)
{
    u8  i;                                          // Loop parameter
    u8  Len;                                        // Get received length

    Len = BLE_RX_LEN - BLE_RX_STR;                  // Get length
    for(i=Len; i>0; i--)                            // Move loop
    {
        *DestIndex++ = BLE_RX_BUF[BLE_RX_STR++];    // Move data to DestIndex
    }
    return Len;                                     // Return receive length
}

/**
 *******************************************************************************
 * @brief     Clear BLE_TX_BUF index and set BLE_TCF = 1
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_ClearBleTxBuffer (void)
{
    BLE_TX_INDEX = 0;
    BLE_TCF = 1;
}
/**
 *******************************************************************************
 * @brief     Clear CLE_RX_BUF index and BLE_RXF = 0
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_ClearBleRxBuffer (void)
{
    BLE_RX_STR = 0;
    BLE_RX_LEN = 0;
    BLE_RXF = 0;
}
/**
 *******************************************************************************
 * @brief     IRQ MCU go to sleep and wakeup
 * @details   this function CAN be invoked at main task, one can invoke 
 *  \n        this function to detect the BLE status then goto MCU sleep.
 * @return    None-zero means MCU can enter into stop/sleep mode.  
 * @note
 *******************************************************************************
 */
void BSP_BLE_IrqMcuGotoSleepAndWakeup(void)
{
    if(ble_run_interrupt_McuCanSleep())
    {
        //mcu lowpower
    }
}

/**
 *******************************************************************************
 * @brief       EXIC module initial for BLE
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
unsigned char aes_encrypt_HW(unsigned char *_data, unsigned char *_key)
{
    (void)_data;    // Solve the parameter warning      
    (void)_key;     // Solve the parameter warning 
    
    return 0; //Hardware AES not supported
}

/**
 * @name For BSP_12_BLE_app
 *  
 */ 
///@{
/**
 *******************************************************************************
 * @brief       Write BLE_RX_BUF buffer
 * @details
 * @param[in]   DestIndex: Destination address.
 * @param[in]   len: Transfer length bytes number. 1 ~ 200 bytes.
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_WriteRxBuffer (u8* DestIndex, u8 len)          //api, BLE received data to txBuf   
{
    unsigned char i;

    if ((BLE_RX_LEN+len)<MAX_SIZE)                          // When txBuf waiting transmit index + updata length < MAX_SIZE (200)
    {
        for (i=0;i<len;i++)                                 // Move BLE get data to BLE_RX_BUF[]
        {
            BLE_RX_BUF[BLE_RX_LEN+i] = *(DestIndex+i);
        }
        BLE_RX_LEN += len;                                  // txBuf waiting transmit index + update length
    }
}
///@}

/**
 * @name static function 
 *  
 */ 
///@{
/**
 *******************************************************************************
 * @brief     BLE timebase handle.
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
static void BSP_BLE_TimeBaseIRQHandler(void)
{
    TM_ClearFlag( BLE_TM, (TMx_TOF2|TMx_TOF));
    BLETimebase_Count ++;
    ble_nMsRoutine();
}
/**
 *******************************************************************************
 * @brief       EXIC module initial for BLE
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
static void BLE_EXIC_Init(void)
{
    PIN_InitTypeDef  PINX_InitStruct;
    EXIC_TRGSTypeDef EXIC_TRGS;
    
    UnProtectModuleReg(CSCprotect);                                         // Unprotect CSC module register
    CSC_PeriphOnModeClock_Config(CSC_ON_PortB,ENABLE);                      // Emable port B moudle clock
    ProtectModuleReg(CSCprotect);                                           // Protect CSC module register
    
    //====GPIO Initial==== 
    PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;          // Select GPIO mode 
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;      // Select wether enable internal pull-up R or not.
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;               // Select wehter enable high speed mode
                                                                            //     1.(PINX_Speed_High mode only for PC0~3 , PD0~3 )    
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;         // Select output drive strength 
                                                                            //     1.(Level 0 & level 3 only for PE0~PE3)
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;    // Select input filter divider.
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;         // Select input signal whether inverse or not.
                                                                            //     1.PINX_Inverse_Disable = L level or falling edge.
                                                                            //     2.PINX_Inverse_Enable  = H level or Rising edge.
    PINX_InitStruct.PINX_Alternate_Function = BLE_IRQ_PIN_AFS;              // Select GPIO mode
    GPIO_PinMode_Config(BLE_IRQ_PINX,&PINX_InitStruct);
    
    //====EXIC Initial====                                   
    EXIC_TRGS.EXIC_Pin = BLE_CSN_EXIC_TRGSPIN;                              //1. You can select any pin of the port.
    EXIC_TRGS.EXIC_TRGS_Mode = Edge;                                        // External interrupt pin edge/level trigger event select.
    EXIC_PxTriggerMode_Select(BLE_CSN_EXIC_PORT,&EXIC_TRGS);                // EXIC trigger mode configuration.
                                                                              
    EXIC_PxTriggerOrMask_Select(BLE_CSN_EXIC_PORT , BLE_CSN_EXIC_OR_MASK);  // OF trigger event select.
                                                                              
    EXIC_ClearPxTriggerEventFlag(BLE_CSN_EXIC_PORT, BLE_CSN_EXIC_FLAG );    // Clear All pin of the port event flag.    
    
    EXIC_PxTriggerITEA_Cmd(BLE_CSN_EXIC_IT , ENABLE);                       // Enable EXIC interrupt
    NVIC_EnableIRQ(BLE_CSN_EXIC_IRQn);                                      // Enable EXIC interrupt of NVIC
}

/**
 *******************************************************************************
 * @brief       IWDT module initial
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
static void IWDT_Init(void)
{
    /*=== 1. Configure IWDT clock ===*/
    UnProtectModuleReg(IWDTprotect);                                // Unprotect IWDT module
    IWDT_Divider_Select(IWDT_DIV_256);                              // DIV output = CK_IWDT /256

    /*=== 2. Configure RST module ===*/                         
    UnProtectModuleReg(RSTprotect);                                 // Unprotect RST moduel register
    RST_WRstSource_Config(RST_IWDT_WE, ENABLE);                     // Enable IWDT event trigger warm reset
    ProtectModuleReg(RSTprotect);                                   // Protect RST moduel register

    IWDT_IT_Config(IWDT_INT_TF, ENABLE);                            // Enable IWDT timeout interupt 
    NVIC_EnableIRQ(SYS_IRQn);
    SYS_ITEA_Cmd(ENABLE);                                           // Enable SYS interupt in NVIC
    IWDT_ClearFlag(IWDT_TF);                                        // Clear IWDT timeout flag

    /*=== 3. Enable IWDT module ===*/
    IWDT_Cmd(ENABLE);                                               // Enable IWDT module
    ProtectModuleReg(IWDTprotect);                                  // Protect IWDT module
    
    /*=== 4. Check flag action ===*/
    while(IWDT_GetSingleFlagStatus(IWDT_EW1F) == DRV_UnHappened);   // Wait IWDT early wakeup-1 happened
    IWDT_ClearFlag(IWDT_EW1F);                                      // Clear EW1F flag 
    
    while(IWDT_GetSingleFlagStatus(IWDT_EW0F) == DRV_UnHappened);   // Wait IWDT early wakeup-0 happened
    IWDT_ClearFlag(IWDT_EW0F);                                      // Clear EW0F flag

    IWDT_RefreshCounter();                                          // Clear IWDT timer
}
/**
 *******************************************************************************
 * @brief       SPI0 module initial
 * @details
 * @param[in]   ProcessClockRate
 *  @arg\b          12000000 ~ 48000000
 * @param[in]   BLEClockRate
 *  @arg\b          0, is auto setting BLE clock rate 6000000 ~ 10000000.
 *  @arg\b          6000000 ~ 10000000, setting BLE clock rate.
 * @return      
 * @note
 *******************************************************************************
 */
static DRV_Return SPIx_Init (uint32_t ProcessClockRate)
{
    volatile uint8_t  CNT;                      /* volatile */
    volatile float    SPI_Clock;                /* volatile */
    volatile uint32_t DIV;                      /* volatile */


    // When CK_AHB > 12MHz
    if(ProcessClockRate > 12000000)
    {
        CPU_CLK_FAST_ENABLE = TRUE;
    }
    else
    {
        CPU_CLK_FAST_ENABLE = FALSE;
    }
    // When connect is SPI0
    //    /*=== 1. Default Initial SPI ===*/
    //    SPI_DeInit(SPI0);

    //    /*=== 2. Configure clock divider ===*/                      // SPI clock = 1MHz
    //    SPI_Clock_Select(SPI0, SPI_CK_SPIx_PR);                     // CK_SPIx = CK_SPIx_PR
    //    SPI_PreDivider_Select(SPI0, SPI_PDIV_1);                    // PDIV outpu = CK_SPIx /2
    //    SPI_Prescaler_Select(SPI0, SPI_PSC_1);                      // Prescaler outpu = PDIV outpu /3
    //    SPI_Divider_Select(SPI0, SPI_DIV_2);                        // DIV outpu = PDIV outpu /2

    //    /*=== 3. Configure SPI data line, mode and data size... ===*/
    //    SPI_DataLine_Select(SPI0, SPI_Standard);                    // SPI Standard mode 
    //    SPI_ModeAndNss_Select(SPI0, SPI_Master);                    // Master
    //    SPI_ClockPhase_Select(SPI0, SPI_LeadingEdge);               // CPHA = 0
    //    SPI_ClockPolarity_Select(SPI0, SPI_Low);                    // CPOL = 0
    //    SPI_FirstBit_Select(SPI0, SPI_MSB);                         // MSB first
    //    SPI_DataSize_Select(SPI0, SPI_8bits);                       // Data size 8bits

    //    /*=== 4. Enable SPI ===*/
    //    SPI_Cmd(SPI0, ENABLE);                                      // Enable SPI0 module

    /* Get Prescaler setting */
    DIV = 0xFF;
    CNT = 0;
    do
    {
        CNT++;
        SPI_Clock = (float) ProcessClockRate;
        SPI_Clock = (SPI_Clock /CNT) /2;
        if((SPI_Clock >= 6000000) && (SPI_Clock < 10000000))        // When SPI clock range in 6M ~ 10M
        {
            DIV = CNT;                                              // Store DIV 
        }
    }while(CNT<8);                                                  // When 
    
    DIV--;
    if(DIV > 8)                                                     // When not get DIV
        return DRV_Failure;
    
    /*=== 1. Default Initial SPI ===*/
    BLE_MODULE->CR0.W = 0x00000000;
    BLE_MODULE->STA.W = 0x00000FD8;
    BLE_MODULE->INT.W = 0x00000000;
    BLE_MODULE->CLK.W = 0x00000000;
    BLE_MODULE->CR1.W = 0x00000000;
    BLE_MODULE->CR2.W = 0x03000100;

    /*=== 2. Configure clock divider ===*/                          // SPI clock = 1MHz
    BLE_MODULE->CLK.W = SPI_CLK_CK_SEL_proc_w |                     // CK_SPIx = CK_SPIx_PR 
                        SPI_CLK_CK_DIV_div2_w |                     // DIV outpu = PDIV outpu /1
                        (DIV << 8) |                                // Prescaler outpu = PDIV outpu /1
                        SPI_CLK_CK_PDIV_div1_w;                     // PDIV outpu = CK_SPIx /2
    
    /*=== 3. Configure SPI data line, mode and data size... ===*/
    BLE_MODULE->CR2.W |= SPI_CR2_DAT_LINE_spi_w |                   // SPI Standard mode 
                        (SPI_CR2_DSIZE_mask_w & 0x00080000);        // Data size 8bits

    BLE_MODULE->CR0.W |= SPI_CR0_MDS_master_w |
                         SPI_CR0_CPHA_leading_edge_w |
                         SPI_CR0_CPOL_low_w |
                         SPI_CR0_LSB_EN_disable_w;

    /*=== 4. Enable SPI ===*/
    BLE_MODULE->CR0.W |= SPI_CR0_EN_enable_w;                       // Enable SPI0 module
    

    return DRV_Success;
}
/**
 *******************************************************************************
 * @brief       SPI0 module initial for BLE
 * @details
 * @return      
 * @note        
 *******************************************************************************
 */
static void BLE_SPI_Init(void)
{
    PIN_InitTypeDef PINX_InitStruct;

    /*=== 1. Config SPI0 IO ===*/
    //SCK
    PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;          // Pin select pusu pull mode
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;       // Enable pull up resistor
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;                // Pin output low speed mode
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;          // Pin output driver full strength.
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;     // Pin input deglitch filter clock divider bypass
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;          // Pin input data not inverse
    PINX_InitStruct.PINX_Alternate_Function = BLE_SCK_PIN_AFS;               
    GPIO_PinMode_Config(BLE_SCK_PINX,&PINX_InitStruct);                      
         
    //MOSI    
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level2;          // Pin output drive strength 1/4
    GPIO_PinMode_Config(BLE_MOSI_PINX,&PINX_InitStruct);                     
                                                      
    //CSN    
    PINX_InitStruct.PINX_Alternate_Function = BLE_CSN_PIN_AFS;               
    GPIO_PinMode_Config(BLE_CSN_PINX,&PINX_InitStruct);                      
         
    //MISO         
    PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;           // Setting digital  mode
    PINX_InitStruct.PINX_Alternate_Function = BLE_MISO_PIN_AFS;              
    GPIO_PinMode_Config(BLE_MISO_PINX,&PINX_InitStruct);  
    
    /*=== 2. Configure SPI ===*/
    SPIx_Init(BLE_SPI_CLOCKFREQ);
}
/**
 *******************************************************************************
 * @brief       SPI0 module initial for BLE
 * @details
 * @return      
 * @note        
 *******************************************************************************
 */
static void BSP_TM_Init(void)
{
    TM_TimeBaseInitTypeDef  BLE_TimeBase;
    
    /*Timer initial*/
    //Timer Default initial
    NVIC_DisableIRQ( TM1x_IRQn);
    TM_DeInit(BLE_TM);
    
    //TimeBase structure initial
    TM_TimeBaseStruct_Init(&BLE_TimeBase);
    
    //Set Step motor rotating speed.
    //Speed = TM clock source freq / ((TM_Prescaler + 1) * (TM_Period + 1));
    BLE_TimeBase.TM_MainClockDirection = TM_UpCount;
    BLE_TimeBase.TM_Prescaler          = 240 - 1;                             
    BLE_TimeBase.TM_Period             = 200 - 1;                                 
    BLE_TimeBase.TM_CounterMode        = Cascade;                           // Timer overflow is 1ms in 48MHz.
    TM_TimeBase_Init(BLE_TM, &BLE_TimeBase);
    
    //Interrupt configure
    TM_ClearFlag( BLE_TM, TMx_TOF);
    TM_IT_Config(BLE_TM, TMx_TIE_IE,ENABLE);
    TM_ITEA_Cmd( BLE_TM, ENABLE);
    NVIC_EnableIRQ( TM0x_IRQn);
    
    //Enable Timer
    TM_Timer_Cmd(BLE_TM,ENABLE);
}
/**
 *******************************************************************************
 * @brief       Send BLE_TX_BUF data
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
static void BSP_BLE_SendData (void) 
{
    u16 send;

    if(BLE_TxBufIndex != BLE_TxdBufIndex)           // When BLE transmit buffer index != BLE transmitted buffer index
    {
        
        if(!GetConnectedStatus())                   // When not Connected
        {
            BLE_TxdBufIndex = BLE_TxBufIndex;       // Update BLE transmitted buffer index, BLE transmit buffer index.
        }
        else                                        // When Connected
        {
            if(BLE_TxBufIndex > BLE_TxdBufIndex)    // When BLE transmit buffer index > BLE transmitted buffer index
            {
                send = sconn_notifydata(BLE_TX_BUF+BLE_TxdBufIndex, (unsigned char)(BLE_TxBufIndex - BLE_TxdBufIndex));
                                                    // BLE transmit data and return transmit length.
                BLE_TxdBufIndex += send;            // Updata BLE transmitted index
            }
            else                                    // When BLE transmit buffer index =< BLE transmitted buffer index
            {
                send = sconn_notifydata(BLE_TX_BUF+BLE_TxdBufIndex, (unsigned char)(MAX_SIZE - BLE_TxdBufIndex));
                                                    // BLE transmit data and return transmit length.
                BLE_TxdBufIndex += send;            // Updata BLE transmitted buffer index
                BLE_TxdBufIndex %= MAX_SIZE;        // Calculate BLE transmitted index.
                
                /*=== Transmit test ===*/
            }
        }
        
        if(BLE_TxBufIndex == BLE_TxdBufIndex)       // When BLE_TX_BUF transmit complete.
        {
            BLE_TCF = 1;                            // BLE TX transmited complet flag.
        }
    }
}



///@}


/**
 * @name For mg_BLE_libv4_m0.lib use.
 *  
 */ 
///@{

/**
 *******************************************************************************
 * @brief       IS IRQ Enable
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
char IsIrqEnabled(void) //porting api
{
    return !(BLE_IRQ);
}
/**
 *******************************************************************************
 * @brief       Get sys tick count
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
unsigned int GetSysTickCount(void) //porting api
{
    return BLETimebase_Count;
}
/**
 *******************************************************************************
 * @brief       User Proc Callback
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void UsrProcCallback(void)                  // Porting api
{    
    IWDT_RefreshCounter();                  // Refresh IWDT counter

    BSP_BLE_SendData();                     // Move rxBuf data to BLE
    
    if ((BLE_RX_LEN!=0) && (0 == BLE_RX_STR))  // When BLE receive start = 0 and BLE recevied length
    {
        BLE_RXF = 1;                        /* Trigger BLE RX Flag */

        BSP_BLE_ReadCallback();
    }
}
/**
 *******************************************************************************
 * @brief       SPI write buffer
 * @details  
 * @param[in]   reg: Command
 * @param[in]   *pBuf: Write buffer index
 * @param[in]   len: Data length
 *  @arg\b          0~255
 * @return      DRV_Return
 *  @arg\b          0
 * @note
 *******************************************************************************
 */
unsigned char SPI_WriteBuf(unsigned char reg, unsigned char const *pBuf, unsigned char len)
{
    volatile signed char i=0,j=0;               /* volatile */
    
    
    SPI_CSE = 0;                                                    // NSS = 0
    
    BLE_MODULE->TDAT.B[0] = reg;                                    // Write 1 byte
    
    for (i=(signed char)len;i>3;i-=4)
    {                                                               // Write 4 bytes
        Temp32 = *(pBuf+j) | 
                 (unsigned long) (*(pBuf+j+1)<<8) | 
                 (unsigned long) (*(pBuf+j+2)<<16) | 
                 (unsigned long) (*(pBuf+j+3)<<24);
        j+=4;
        BLE_MODULE->TDAT.W = Temp32;
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }
    
    for (;i>2;i-=3)
    {                                                               // Write 3 bytes
        Temp24 = *(pBuf+j) | 
                 (unsigned long) (*(pBuf+j+1)<<8) | 
                 (unsigned long) (*(pBuf+j+2)<<16);
        j+=3;
        BLE_MODULE->TDAT3.W = Temp24;
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }
    
    for (;i>1;i-=2)
    {                                                               // Write 2 bytes
        Temp16 = *(pBuf+j) | 
                 (unsigned short)(*(pBuf+j+1)<<8);
        j+=2;                                                       // Index parameter + 2
        BLE_MODULE->TDAT.H[0] = Temp16;                             // Write 2 bytes to TX
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }
    
    for (;i>0;i--)
    {
        Temp8 = *(pBuf+j);                                          // Write 1 byte
        j++;
        BLE_MODULE->TDAT.B[0] = Temp8;
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }    
    
    
    while((BLE_MODULE->STA.B[0] & SPI_STA_TCF_mask_b0) == 0);   // Wait TX data transmit complete
    BLE_MODULE->CR1.B[0] = SPI_CR1_RDAT_CLR_mask_b0;            // Clear RX buffe data
    BLE_MODULE->STA.H[0] = SPI_ALLF;                            // Clear SPI all flag
    
    SPI_CSE = 1;                                                // NSS = 1
    return 0;
}
/**
 *******************************************************************************
 * @brief       SPI Read buffer
 * @details  
 * @param[in]   reg: Command
 * @param[in]   *pBuf: Read buffer index
 * @param[in]   len: Data length
 *  @arg\b          0~255
 * @return      DRV_Return
 *  @arg\b          0
 * @note
 *******************************************************************************
 */
unsigned char SPI_ReadBuf(unsigned char reg, unsigned char *pBuf, unsigned char len)
{
    volatile unsigned char i=0;                 /* volatile */



    SPI_CSE = 0;                                                    // NSS = 0
    
    BLE_MODULE->TDAT.B[0] = reg;                                    // Write 1 byte
    if(CPU_CLK_FAST_ENABLE == FALSE)                                // When CK_AHB <= 12MHz
    {
        BLE_MODULE->TDAT.B[0] = 0xff;                                   // Write 1 byte 
    }
    while((BLE_MODULE->STA.B[0] & SPI_STA_RXF_mask_b0) == 0);       // Wait RDAT not empty
    BLE_MODULE->RDAT.B[0];                                          // Clear RDAT
    
    for (i=0;i<len;i++)                                             // Waiting  
    {
        BLE_MODULE->TDAT.B[0] = 0xff;                               // Write 1 byte
        while((BLE_MODULE->STA.B[0] & SPI_STA_RXF_mask_b0) == 0);   // Wait RDAT not empty
        *(pBuf+i) = BLE_MODULE->RDAT.B[0];                          // Read data
    }
       /*wait tCF transm complete*/
    SPI_CSE = 1;                                                    // NSS = 1
    
    BLE_MODULE->CR1.B[0] = SPI_CR1_RDAT_CLR_mask_b0;            // Clear RX buffe data
    

    return 0;
}
void SysClk8to48(void)
{
}
void SysClk48to8(void)
{
}
void DisableEnvINT(void)
{
}
void EnableEnvINT(void)
{
}
void McuGotoSleepAndWakeup(void)
{
}
void UpdateLEDValueAll(void)
{
}

__WEAK void BSP_BLE_ReadCallback(void)
{
    //To to ....
}
///@}

#endif

/**
 *******************************************************************************
 * @brief   Prevent the file is empty compilcation warning. 
 * @details
 * @return      
 * @note
 *******************************************************************************
 */
void BSP_BLE_MG126PreventComilcationWarning(void);

__WEAK void BSP_BLE_MG126PreventComilcationWarning(void)
{
    __NOP();
}



