/**
 ******************************************************************************
 *
 * @file        BSP_ARGB_WS2812.c
 * @brief       
 *
 * @par         Project
 *              MG32F02N128
 * @version     V1.00
 * @date        2024/08/09
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_GPIO_DRV.h"
#include "MG32_APX_DRV.h"
#include "BSP_ARGB_WS2812.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

#define BSP_ARGB_PINX                 PIND(13)

#define BSP_ARGB_PINAFS               10

#define BSP_ARGB_ASBChannel           0

#define BSP_ARGB_TXFLAG               APX_ASB0_TXF
#define BSP_ARGB_TCFLAG               APX_ASB0_TCF

#define BSP_ARGB_TXIE                 APX_ASB0_TIE
#define BSP_ARGB_TCIE                 APX_ASB0_TCIE

#define __BSP_ARGB_EnableChannel()    *(uint32_t*)(APX_ASBChannelCR0_Base + (APX_ASBChannel_SHIFT * BSP_ARGB_ASBChannel)) |= APX_ASB00_ASB0_EN_mask_w

#define BPS_ARGB_STATUS_IDLE          0x00000000

#define BSP_ARGB_STATUS_BUSY_MASK     0x80000000
#define BSP_ARGB_STATUS_COUNT_MASK    0x0000FFFF


/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static uint8_t  *BSP_ARGB_TXBuf;
static uint32_t  BSP_ARGB_Status;

/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief       ARGB IRQHandler function.   
 * @details     
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void BSP_ARGB_IRQHandler(void)
{
    uint32_t ARGB_IRQHandler_Status;

    ARGB_IRQHandler_Status  = (APX_GetAllFlagStatus(APX) & APX->INT.W);

    if( ARGB_IRQHandler_Status & BSP_ARGB_TXFLAG)
    {
        switch(BSP_ARGB_Status & BSP_ARGB_STATUS_COUNT_MASK)
        {
            case 1:
                    APX_ASBChannel_SendData(BSP_ARGB_ASBChannel, *BSP_ARGB_TXBuf ++);
                    BSP_ARGB_Status = BSP_ARGB_Status - 1;
                    break;
            case 0:
                    __BSP_ARGB_IT_DISABLE(BSP_ARGB_TXIE);
                    __BSP_ARGB_IT_ENABLE(BSP_ARGB_TCIE);
                    break;
            default:
                    APX_ASBChannel_SendData(BSP_ARGB_ASBChannel, *BSP_ARGB_TXBuf ++);
                    APX_ASBChannel_SendData(BSP_ARGB_ASBChannel, *BSP_ARGB_TXBuf ++);
                    BSP_ARGB_Status = BSP_ARGB_Status - 2;
                    break;
        }
    }

    if( ARGB_IRQHandler_Status & BSP_ARGB_TCFLAG)
    {
        __BSP_ARGB_IT_DISABLE(BSP_ARGB_TCIE);
        APX_ASBChannel_TriggerResetSignal(BSP_ARGB_ASBChannel);
        BSP_ARGB_Status = BPS_ARGB_STATUS_IDLE;
        BSP_ARGB_SendDataCpltCallback();
    }
}

/**
 *******************************************************************************
 * @brief       ARGB parameter initial  
 * @details     
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void BSP_ARGB_ParameterInit(void)
{
    BSP_ARGB_Status = BPS_ARGB_STATUS_IDLE;
}

/**
 *******************************************************************************
 * @brief       ARGB initial   
 * @details     
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void BSP_ARGB_Init(void)
{
    PIN_InitTypeDef      ARGB_Pin;
    ASB_TypeDef          ASB_CTR;
    ASBChannel_TypeDef   ASBChannel;

    /*======GPIO initial======*/
    /*SDK, D0, D1 pins configure*/
    ARGB_Pin.PINX_Mode               = PINX_Mode_PushPull_O;
    ARGB_Pin.PINX_PUResistant        = PINX_PUResistant_Disable;
    ARGB_Pin.PINX_Speed              = PINX_Speed_High;
    ARGB_Pin.PINX_Inverse            = PINX_Inverse_Disable;
    ARGB_Pin.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    ARGB_Pin.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    ARGB_Pin.PINX_Alternate_Function = BSP_ARGB_PINAFS;
    GPIO_PinMode_Config(BSP_ARGB_PINX, &ARGB_Pin);

    /* ASB Initial (for WS2812B) */
    ASB_CTR.SyncCode                 = ASB_SYNCCODE_CODE0;
    ASB_CTR.IdleStatus               = ASB_IDLESTATUS_LOW;
    ASB_CTR.ResetStatus              = ASB_RSTSTATUS_LOW;
    ASB_CTR.ResetSync                = ASB_RST_SYNC_DISABLE;
    ASB_CTR.ClockInverse             = ASB_CLOKC_INVERSE_DISABLE;
    ASB_CTR.BitTimePrescaler         = 2  - 1; 
    ASB_CTR.BitTimeCounter           = 30 - 1;                           // Bit time = 1.25us  in 48MHz
    ASB_CTR.ResetTime                = 40 - 1;                           // Reset time = 50us in 48MHz
    ASB_CTR.Code0_HighWidthTime      = 8 - 1;                            // Code0's T0H is about 0.3us in 48MHz
    ASB_CTR.Code1_HighWidthTime      = 22 - 1;                           // Code1's T1H is about 0.9us in 48Mhz
    APX_ASB_Config(&ASB_CTR);

    ASBChannel.OutputMode            = ASB_MODE_ARGB;
    ASBChannel.DataOrder             = ASB_DATA_MSB;
    ASBChannel.DataInverse           = ASB_DATA_INVERSE_DISABLE;
    ASBChannel.SignalInverse         = ASB_SIGNAL_INVERSE_DISABLE;
    ASBChannel.PixelLength           = ASB_FRAME_PIXEL_3BYTE;
    ASBChannel.TX_FIFOThreshold      = ASB_TX_FIFOTHRESHOLD_2BYTE;
    APX_ASBChannel_Config( BSP_ARGB_ASBChannel, &ASBChannel);

    __BSP_ARGB_EnableChannel();
    __BSP_ARGB_IT_DISABLE((BSP_ARGB_TXIE | BSP_ARGB_TCIE));

    APX_ITEA_Cmd(APX, ENABLE);
    NVIC_EnableIRQ(APX_IRQn);

    /*Parametr initial*/
    BSP_ARGB_ParameterInit();
}

/**
 *******************************************************************************
 * @brief       Transmit ARGB data.
 * @details     
 * @param[in]   ARGB_TxBuf : data source pointer.
 * @param[in]   ARGB_UpdateByteSize : Transmit byte number.
 * @return      This trigger whether success or not
 * @exception   No
 * @note        No
 *******************************************************************************
 */
uint32_t BSP_ARGB_SendData(uint8_t *ARGB_TxBuf, uint16_t ARGB_UpdateByteSize)
{
    uint8_t ARGB_SendData_Tmp;

    if((BSP_ARGB_Status & BSP_ARGB_STATUS_BUSY_MASK) || \
       (ARGB_UpdateByteSize == 0)                    || \
       (ARGB_TxBuf == NULL))
    {
        return(BSP_ARGB_FAILURE);
    }

    BSP_ARGB_TXBuf    = ARGB_TxBuf;
    BSP_ARGB_Status   = ARGB_UpdateByteSize;
    ARGB_SendData_Tmp = 0;

    do{
        APX_ASBChannel_SendData(BSP_ARGB_ASBChannel, *BSP_ARGB_TXBuf ++);
        ARGB_SendData_Tmp = ARGB_SendData_Tmp + 1;
        BSP_ARGB_Status   = BSP_ARGB_Status - 1;
    }while((ARGB_SendData_Tmp<4) && (BSP_ARGB_Status != 0));

    BSP_ARGB_Status |= BSP_ARGB_STATUS_BUSY_MASK;

    __BSP_ARGB_IT_ENABLE(BSP_ARGB_TXIE);

    return(BSP_ARGB_SUCCESS);
}

/**
 *******************************************************************************
 * @brief        ARGB transmit complete call back function.
 * @details     
 * @return      
 * @note        
 *******************************************************************************
 */
__WEAK void BSP_ARGB_SendDataCpltCallback(void)
{
    //=========================================================
    //Note : This function should not be modified, when the callback
    //       is needed, the BSP_ARGB_SendDataCpltCallback can
    //       be implemented in the user file.
}

