

/**
 ******************************************************************************
 *
 * @file        BSP_13_SPIFlash.h
 * @brief       This is SPI Flash H file.
 
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2022/05/23
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


#ifndef _BSP_15_SPIFLASH_H

#define _BSP_15_SPIFLASH_H

#include "MG32_GPIO_DRV.h"
#include "MG32_SPI_DRV.h"

/**
 * @name    define
 *  
 */ 
///@{
//MX25R512F Command
#define FLASH_READ                  0x03    /*!< Read data bytes                                        */
#define FLASH_FAST_READ             0x0B    /*!< Read data bytes at higher speed                        */
#define FLASH_2READ                 0xBB    /*!< 2 x I/O read mode                                      */
#define FLASH_DREAD                 0x3B    /*!< Dual read mode                                         */
#define FLASH_4READ                 0xEB    /*!< 4 x I/O read mode                                      */
#define FLASH_QRead                 0x6B    /*!< Quad read mode                                         */

#define FLASH_PP                    0x02    /*!< Page program                                           */
#define FLASH_4PP                   0x38    /*!< 4 x I/O page program                                   */

#define FLASH_SECTOR_ERASE          0x20    /*!< 1 sector = 4K bytes                                    */
#define FLASH_BLOCK32K_ERASE        0x52    /*!< Block erase 32K bytes                                  */
#define FLASH_BLOCK64K_ERASE        0xD8    /*!< Block erase 62K bytes                                  */
#define FLASH_CHIP_ERASE            0x60    /*!< Chip erase                                             */

#define FLASH_SFDP_READ             0x5A    /*!< Read SFDP mode                                         */
#define FLASH_WRITE_ENABLE          0x06    /*!< Write enable                                           */
#define FLASH_WRITE_DISABLE         0x04    /*!< Write disable                                          */
#define FLASH_READ_STATUS           0x05    /*!< Read Status Register                                   */
#define FLASH_READ_CONFIGURATION    0x15    /*!< Read configuration register                            */
#define FLASH_WRITE_STATUS          0x01    /*!< Write status register                                  */
#define FLASH_SUSPEND_PROGRAM       0x75    /*!< Suspends erase                                         */
#define FLASH_SUSPEND_ERASE         0xB0    /*!< Suapends erase                                         */
#define FLASH_RESUMES_PROGRAM       0x7A    /*!< Resume program                                         */
#define FLASH_RESUMES_REASE         0x30    /*!< Resume erase                                           */
#define FLASH_DEEP_POWER_DOWN       0xB9    /*!< Deep power-down                                        */
#define FLASH_SET_BURST_LENGTH      0xC0    /*!< Burst read                                             */
#define FLASH_READ_IDENTIFICATION   0x9F    /*!< Read manufacturer ID + memory type + memory density.   */
#define FLASH_READ_ELECTRONIC_ID    0xAB    /*!< Read electronic ID.                                    */
#define FLASH_READ_REMS             0x90    /*!< Read manufacturer & device ID.                         */
#define FLASH_ENTER_SECURED_OTP     0xB1    /*!< Enter secured OTP                                      */
#define FLASH_EXIT_SECURED_OTP      0xC1    /*!< Exit secured OTP                                       */
#define FLASH_READ_SECURITY_REG     0x2B    /*!< Read security register                                 */
#define FLASH_WRITE_SECURITY_REG    0x2F    /*!< Write security register                                */
#define FLASH_NOP                   0x00    /*!< Nop                                                    */
#define FLASH_RESET_ENABLE          0x66    /*!< Reset enable                                           */
#define FLAHS_RESET_MEMORY          0x99    /*!< Reset memory                                           */
///@}

/**
 * @name    typedef
 *  
 */ 
///@{


/*! @enum   Flash_EraseypeDef
    @brief  The Flash Erase type.
*/
typedef enum
{
    Flash_Erase_Sector   = FLASH_SECTOR_ERASE,          /*!< Erase type = Sector(4K byte) erase.                  */
    Flash_Erase_Block32K = FLASH_BLOCK32K_ERASE,        /*!< Erase type = 32K byte block erase.                   */
    Flash_Erase_Block64K = FLASH_BLOCK64K_ERASE,        /*!< Erase type = 64K byte block erase.                   */
    Flash_Erase_Chip     = FLASH_CHIP_ERASE,            /*!< Erase type = Chip erase.                             */
}Flash_EraseypeDef;

/*! @enum   Flash_ReadTypeDef
    @brief  The Flash read mode.
*/
typedef enum
{
    Flash_Read     = FLASH_FAST_READ,
    Flash_4IO_Read = FLASH_4READ,
}Flash_ReadTypeDef;

/*! @enum   Flash_ProgramTypeDef
    @brief  The Flash program mode.
*/
typedef enum
{
    Flash_PageProgram     = FLASH_PP,
    Flash_4IO_PageProgram = FLASH_4PP,
}Flash_ProgramTypeDef;


/*! @struct FlashCTR_TypeDef
    @brief  Flash data control.
*/ 
typedef struct __attribute__((packed))
{
    uint32_t  Address;
    uint32_t  Total_Length;
    uint8_t   *Buffer;

}FlashCTR_TypeDef;

///@}


/**
 * @name variables
 *  
 */ 
///@{
extern FlashCTR_TypeDef SPIFlash_CTR;

///@}

/**
 * @name function
 *  
 */ 
///@{
void BSP_SPIFlash_Init(void);

uint32_t BSP_SPIFlash_ReadFlashRDID (void);
uint16_t BSP_SPIFlash_ReadFlashRDCR (void);

void BSP_SPIFlash_Erase(Flash_EraseypeDef EraseType , uint32_t Page_Num);
void BSP_SPIFlash_Read(FlashCTR_TypeDef rFlash, Flash_ReadTypeDef rFlash_Mode);
void BPS_SPIFlash_Program(FlashCTR_TypeDef pFlash, Flash_ProgramTypeDef pFlash_Mode);


///@}


#endif








